import { PrismaClient, ContentType, ContentStatus } from '@prisma/client'

const prisma = new PrismaClient()

async function seedContent() {
  try {
    console.log('🌱 Seeding content...')

    // Find admin user
    const admin = await prisma.user.findFirst({
      where: { role: 'ADMIN' }
    })

    if (!admin) {
      console.error('❌ No admin user found. Please create an admin user first.')
      return
    }

    console.log(`✅ Found admin user: ${admin.email}`)

    // Sample content for different pages
    const contentData = [
      {
        title: 'Home - Hero Section',
        slug: 'home-hero',
        type: 'PAGE' as ContentType,
        category: 'Home Page',
        status: 'PUBLISHED' as ContentStatus,
        tags: JSON.stringify(['home', 'hero', 'landing']),
        content: `<h1>Welcome to Nuwair Systems</h1>
<p class="text-xl">Your trusted partner for web hosting, domain registration, and digital solutions.</p>
<p>We provide enterprise-grade infrastructure with 24/7 support to power your online presence.</p>
<div class="flex space-x-4 mt-6">
  <a href="/services" class="btn-primary">Explore Services</a>
  <a href="/contact" class="btn-secondary">Contact Sales</a>
</div>`,
        authorId: admin.id
      },
      {
        title: 'Home - Services Overview',
        slug: 'home-services',
        type: 'PAGE' as ContentType,
        category: 'Home Page',
        status: 'PUBLISHED' as ContentStatus,
        tags: JSON.stringify(['home', 'services', 'overview']),
        content: `<h2>Our Services</h2>
<p>Comprehensive solutions for your digital needs</p>
<div class="grid grid-cols-3 gap-6">
  <div class="service-card">
    <h3>Web Hosting</h3>
    <p>Reliable and fast hosting solutions with 99.9% uptime guarantee.</p>
  </div>
  <div class="service-card">
    <h3>Domain Registration</h3>
    <p>Secure your perfect domain name from our extensive selection.</p>
  </div>
  <div class="service-card">
    <h3>Website Development</h3>
    <p>Professional websites built with modern technologies.</p>
  </div>
</div>`,
        authorId: admin.id
      },
      {
        title: 'Home - Statistics',
        slug: 'home-stats',
        type: 'PAGE' as ContentType,
        category: 'Home Page',
        status: 'PUBLISHED' as ContentStatus,
        tags: JSON.stringify(['home', 'statistics', 'metrics']),
        content: `<h2>Trusted by Thousands</h2>
<div class="stats-grid">
  <div class="stat">
    <h3>2,000+</h3>
    <p>Active Clients</p>
  </div>
  <div class="stat">
    <h3>99.9%</h3>
    <p>Uptime Guarantee</p>
  </div>
  <div class="stat">
    <h3>24/7</h3>
    <p>Expert Support</p>
  </div>
  <div class="stat">
    <h3>15+</h3>
    <p>Years Experience</p>
  </div>
</div>`,
        authorId: admin.id
      },
      {
        title: 'Services - Hero Section',
        slug: 'services-services-hero',
        type: 'PAGE' as ContentType,
        category: 'Services',
        status: 'PUBLISHED' as ContentStatus,
        tags: JSON.stringify(['services', 'hero']),
        content: `<h1>Our Services</h1>
<p class="text-xl">Comprehensive hosting and digital solutions tailored to your needs</p>
<p>From small businesses to large enterprises, we have the perfect solution for you.</p>`,
        authorId: admin.id
      },
      {
        title: 'Services - Service List',
        slug: 'services-services-list',
        type: 'SERVICE' as ContentType,
        category: 'Services',
        status: 'PUBLISHED' as ContentStatus,
        tags: JSON.stringify(['services', 'pricing', 'hosting']),
        content: `<h2>Choose Your Perfect Plan</h2>

<div class="service-plans">
  <div class="plan">
    <h3>Starter Hosting</h3>
    <p class="price">$9.99/month</p>
    <ul>
      <li>10 GB Storage</li>
      <li>100 GB Bandwidth</li>
      <li>1 Domain</li>
      <li>Email Support</li>
    </ul>
    <button>Get Started</button>
  </div>

  <div class="plan featured">
    <h3>Business Hosting</h3>
    <p class="price">$24.99/month</p>
    <ul>
      <li>50 GB Storage</li>
      <li>Unlimited Bandwidth</li>
      <li>5 Domains</li>
      <li>24/7 Support</li>
      <li>Free SSL Certificate</li>
    </ul>
    <button>Most Popular</button>
  </div>

  <div class="plan">
    <h3>Enterprise Hosting</h3>
    <p class="price">$49.99/month</p>
    <ul>
      <li>200 GB Storage</li>
      <li>Unlimited Bandwidth</li>
      <li>Unlimited Domains</li>
      <li>Priority Support</li>
      <li>Free SSL & Backups</li>
      <li>Dedicated Resources</li>
    </ul>
    <button>Contact Sales</button>
  </div>
</div>`,
        authorId: admin.id
      },
      {
        title: 'About Us - Company Overview',
        slug: 'about-about-hero',
        type: 'PAGE' as ContentType,
        category: 'About Us',
        status: 'PUBLISHED' as ContentStatus,
        tags: JSON.stringify(['about', 'company', 'mission']),
        content: `<h1>About Nuwair Systems</h1>
<p class="text-xl">Empowering businesses with reliable digital infrastructure since 2010</p>

<h2>Our Mission</h2>
<p>To provide world-class hosting and digital solutions that help businesses thrive in the digital age. We combine cutting-edge technology with exceptional customer service to deliver unmatched value.</p>

<h2>Our Vision</h2>
<p>To be the most trusted and innovative technology partner for businesses worldwide, enabling them to focus on growth while we handle their digital infrastructure.</p>`,
        authorId: admin.id
      },
      {
        title: 'Contact - Contact Information',
        slug: 'contact-contact-hero',
        type: 'PAGE' as ContentType,
        category: 'Contact',
        status: 'PUBLISHED' as ContentStatus,
        tags: JSON.stringify(['contact', 'support']),
        content: `<h1>Get in Touch</h1>
<p class="text-xl">We're here to help you succeed</p>

<div class="contact-info">
  <div class="info-item">
    <h3>Email</h3>
    <p>sales@nuwair.com</p>
    <p>support@nuwair.com</p>
  </div>

  <div class="info-item">
    <h3>Phone</h3>
    <p>+1 (555) 123-4567</p>
    <p>Available 24/7</p>
  </div>

  <div class="info-item">
    <h3>Address</h3>
    <p>123 Tech Street</p>
    <p>Silicon Valley, CA 94025</p>
  </div>
</div>

<h2>Business Hours</h2>
<p>Our support team is available 24/7 to assist you with any questions or concerns.</p>`,
        authorId: admin.id
      }
    ]

    // Create content
    for (const data of contentData) {
      const existing = await prisma.content.findUnique({
        where: { slug: data.slug }
      })

      if (existing) {
        console.log(`⏭️  Skipping ${data.title} (already exists)`)
        continue
      }

      await prisma.content.create({
        data: {
          ...data,
          publishedAt: data.status === 'PUBLISHED' ? new Date() : null
        }
      })
      console.log(`✅ Created: ${data.title}`)
    }

    console.log('\n🎉 Content seeding completed!')
    console.log('\n📝 You can now:')
    console.log('1. Go to http://localhost:3000/admin?tab=cms')
    console.log('2. Click on "Website Pages" tab')
    console.log('3. Click "Edit" on any section with content')
    console.log('4. Make changes in the WYSIWYG editor')
    console.log('5. Click "Save & Publish"\n')

  } catch (error) {
    console.error('❌ Error seeding content:', error)
  } finally {
    await prisma.$disconnect()
  }
}

seedContent()
