'use client'

import { useState, useEffect, useRef } from 'react'
import { useSession } from 'next-auth/react'
import { useRouter } from 'next/navigation'
import { motion, AnimatePresence } from 'framer-motion'
import Link from 'next/link'
import {
  ChatBubbleLeftRightIcon,
  UserGroupIcon,
  PaperAirplaneIcon,
  MagnifyingGlassIcon,
  CheckCircleIcon,
  ClockIcon,
  XCircleIcon,
  EllipsisVerticalIcon,
  PaperClipIcon,
  FaceSmileIcon,
  BoltIcon,
  SparklesIcon,
  UserCircleIcon,
  PhoneIcon,
  VideoCameraIcon,
  InformationCircleIcon
} from '@heroicons/react/24/outline'

interface StaffMember {
  id: string
  name: string
  email: string
  avatar?: string
  status: 'online' | 'away' | 'busy' | 'offline'
  activeChats: number
  department: string
  lastSeen?: string
}

interface ChatMessage {
  id: string
  sender: 'user' | 'staff' | 'ai'
  senderName: string
  message: string
  timestamp: string
  isTyping?: boolean
}

interface ChatSession {
  id: string
  customerName: string
  customerEmail: string
  status: 'active' | 'waiting' | 'closed'
  assignedTo?: string
  startTime: string
  lastMessage: string
  unreadCount: number
  priority: 'low' | 'medium' | 'high'
}

export default function AdminChat() {
  const { data: session, status } = useSession()
  const router = useRouter()
  const [loading, setLoading] = useState(true)
  const [selectedChat, setSelectedChat] = useState<string | null>(null)
  const [message, setMessage] = useState('')
  const [searchTerm, setSearchTerm] = useState('')
  const [aiAssistEnabled, setAiAssistEnabled] = useState(true)
  const messagesEndRef = useRef<HTMLDivElement>(null)

  const [staffMembers, setStaffMembers] = useState<StaffMember[]>([])
  const [chatSessions, setChatSessions] = useState<ChatSession[]>([])
  const [messages, setMessages] = useState<ChatMessage[]>([])

  useEffect(() => {
    if (status === 'unauthenticated') {
      router.push('/auth/signin')
    }

    // Load staff and chat data
    setTimeout(() => {
      setStaffMembers([
        {
          id: '1',
          name: 'Sarah Johnson',
          email: 'sarah@nuwair.com',
          status: 'online',
          activeChats: 3,
          department: 'Technical Support'
        },
        {
          id: '2',
          name: 'Mike Chen',
          email: 'mike@nuwair.com',
          status: 'online',
          activeChats: 2,
          department: 'Sales'
        },
        {
          id: '3',
          name: 'Emily Davis',
          email: 'emily@nuwair.com',
          status: 'away',
          activeChats: 1,
          department: 'Billing'
        },
        {
          id: '4',
          name: 'John Smith',
          email: 'john@nuwair.com',
          status: 'busy',
          activeChats: 5,
          department: 'Technical Support'
        },
        {
          id: '5',
          name: 'Lisa Wang',
          email: 'lisa@nuwair.com',
          status: 'offline',
          activeChats: 0,
          department: 'Customer Success',
          lastSeen: '2 hours ago'
        }
      ])

      setChatSessions([
        {
          id: 'chat-1',
          customerName: 'Tech Solutions Ltd',
          customerEmail: 'support@techsolutions.com',
          status: 'active',
          assignedTo: 'Sarah Johnson',
          startTime: '10 mins ago',
          lastMessage: 'Can you help me upgrade my VPS plan?',
          unreadCount: 2,
          priority: 'high'
        },
        {
          id: 'chat-2',
          customerName: 'Digital Marketing Co',
          customerEmail: 'admin@digitalmarket.com',
          status: 'waiting',
          startTime: '15 mins ago',
          lastMessage: 'I need assistance with domain transfer',
          unreadCount: 5,
          priority: 'medium'
        },
        {
          id: 'chat-3',
          customerName: 'Startup Ventures',
          customerEmail: 'contact@startup.io',
          status: 'active',
          assignedTo: 'Mike Chen',
          startTime: '25 mins ago',
          lastMessage: 'Thank you for your help!',
          unreadCount: 0,
          priority: 'low'
        },
        {
          id: 'chat-4',
          customerName: 'E-commerce Plus',
          customerEmail: 'info@ecommerceplus.com',
          status: 'waiting',
          startTime: '1 hour ago',
          lastMessage: 'Server is down, urgent help needed',
          unreadCount: 8,
          priority: 'high'
        }
      ])

      setLoading(false)
    }, 800)
  }, [status, router])

  useEffect(() => {
    if (selectedChat) {
      // Load messages for selected chat
      setMessages([
        {
          id: '1',
          sender: 'user',
          senderName: 'Tech Solutions Ltd',
          message: 'Hello, I need help with my VPS hosting plan',
          timestamp: '10:30 AM'
        },
        {
          id: '2',
          sender: 'ai',
          senderName: 'AI Assistant',
          message: 'Hello! I can help you with that. Let me connect you with our technical support team.',
          timestamp: '10:30 AM'
        },
        {
          id: '3',
          sender: 'staff',
          senderName: 'Sarah Johnson',
          message: 'Hi! I\'m Sarah from Technical Support. How can I assist you with your VPS plan today?',
          timestamp: '10:31 AM'
        },
        {
          id: '4',
          sender: 'user',
          senderName: 'Tech Solutions Ltd',
          message: 'Can you help me upgrade my VPS plan? I need more resources.',
          timestamp: '10:32 AM'
        },
        {
          id: '5',
          sender: 'staff',
          senderName: 'Sarah Johnson',
          message: 'Absolutely! Let me check your current plan and available upgrade options. One moment please.',
          timestamp: '10:32 AM'
        }
      ])
      messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' })
    }
  }, [selectedChat])

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'online':
        return 'bg-green-500'
      case 'away':
        return 'bg-yellow-500'
      case 'busy':
        return 'bg-red-500'
      case 'offline':
        return 'bg-gray-400'
      default:
        return 'bg-gray-400'
    }
  }

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'high':
        return 'border-l-4 border-red-500 bg-red-50'
      case 'medium':
        return 'border-l-4 border-yellow-500 bg-yellow-50'
      case 'low':
        return 'border-l-4 border-blue-500 bg-blue-50'
      default:
        return 'border-l-4 border-gray-500'
    }
  }

  const handleSendMessage = () => {
    if (!message.trim()) return

    const newMessage: ChatMessage = {
      id: Date.now().toString(),
      sender: 'staff',
      senderName: session?.user?.name || 'Staff',
      message: message.trim(),
      timestamp: new Date().toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })
    }

    setMessages([...messages, newMessage])
    setMessage('')
    
    setTimeout(() => {
      messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' })
    }, 100)
  }

  const filteredSessions = chatSessions.filter(session =>
    session.customerName.toLowerCase().includes(searchTerm.toLowerCase()) ||
    session.customerEmail.toLowerCase().includes(searchTerm.toLowerCase())
  )

  if (status === 'loading' || loading) {
    return (
      <div className="min-h-screen bg-slate-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-slate-600">Loading chat...</p>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-slate-50">
      {/* Header */}
      <header className="bg-slate-900 text-white shadow-lg">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
          <div className="flex items-center justify-between">
            <div>
              <Link href="/admin/dashboard" className="text-slate-400 hover:text-white text-sm mb-2 inline-block">
                ← Back to Dashboard
              </Link>
              <h1 className="text-3xl font-bold flex items-center">
                <ChatBubbleLeftRightIcon className="w-8 h-8 mr-3" />
                Live Chat Management
              </h1>
              <p className="text-slate-300 mt-1">
                Manage customer conversations with AI assistance
              </p>
            </div>
            <div className="flex items-center space-x-4">
              <button
                onClick={() => setAiAssistEnabled(!aiAssistEnabled)}
                className={`px-4 py-2 rounded-lg flex items-center transition-colors ${
                  aiAssistEnabled
                    ? 'bg-blue-600 hover:bg-blue-700'
                    : 'bg-slate-700 hover:bg-slate-600'
                }`}
              >
                <SparklesIcon className="w-5 h-5 mr-2" />
                AI Assist {aiAssistEnabled ? 'ON' : 'OFF'}
              </button>
            </div>
          </div>
        </div>
      </header>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div className="grid grid-cols-1 lg:grid-cols-4 gap-6">
          {/* Staff Status Sidebar */}
          <motion.div
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            className="bg-white rounded-xl shadow-sm border border-slate-200 p-6"
          >
            <h2 className="text-lg font-bold text-slate-900 mb-4 flex items-center">
              <UserGroupIcon className="w-5 h-5 mr-2 text-blue-600" />
              Staff Online ({staffMembers.filter(s => s.status === 'online').length})
            </h2>
            <div className="space-y-3">
              {staffMembers.map((staff) => (
                <div
                  key={staff.id}
                  className="flex items-center justify-between p-3 bg-slate-50 rounded-lg hover:bg-slate-100 transition-colors"
                >
                  <div className="flex items-center space-x-3">
                    <div className="relative">
                      <div className="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center">
                        <UserCircleIcon className="w-6 h-6 text-blue-600" />
                      </div>
                      <div className={`absolute bottom-0 right-0 w-3 h-3 ${getStatusColor(staff.status)} border-2 border-white rounded-full`}></div>
                    </div>
                    <div>
                      <p className="text-sm font-semibold text-slate-900">{staff.name}</p>
                      <p className="text-xs text-slate-600">{staff.department}</p>
                      {staff.status === 'offline' && staff.lastSeen && (
                        <p className="text-xs text-slate-500">{staff.lastSeen}</p>
                      )}
                    </div>
                  </div>
                  <div className="text-right">
                    <span className={`text-xs font-semibold px-2 py-1 rounded-full ${
                      staff.activeChats > 0 ? 'bg-blue-100 text-blue-800' : 'bg-gray-100 text-gray-600'
                    }`}>
                      {staff.activeChats} chats
                    </span>
                  </div>
                </div>
              ))}
            </div>
          </motion.div>

          {/* Chat List */}
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="bg-white rounded-xl shadow-sm border border-slate-200 overflow-hidden"
          >
            <div className="p-4 border-b border-slate-200">
              <div className="relative">
                <MagnifyingGlassIcon className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400" />
                <input
                  type="text"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  placeholder="Search conversations..."
                  className="w-full pl-10 pr-4 py-2 border border-slate-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
                />
              </div>
              <div className="mt-3 flex items-center justify-between text-sm">
                <span className="text-slate-600">
                  {chatSessions.filter(s => s.status === 'active').length} Active
                </span>
                <span className="text-yellow-600 font-semibold">
                  {chatSessions.filter(s => s.status === 'waiting').length} Waiting
                </span>
              </div>
            </div>
            <div className="overflow-y-auto max-h-[600px]">
              {filteredSessions.map((chat) => (
                <div
                  key={chat.id}
                  onClick={() => setSelectedChat(chat.id)}
                  className={`p-4 border-b border-slate-100 cursor-pointer transition-colors ${
                    selectedChat === chat.id ? 'bg-blue-50' : 'hover:bg-slate-50'
                  } ${getPriorityColor(chat.priority)}`}
                >
                  <div className="flex items-start justify-between mb-2">
                    <div>
                      <h3 className="font-semibold text-slate-900">{chat.customerName}</h3>
                      <p className="text-xs text-slate-600">{chat.customerEmail}</p>
                    </div>
                    {chat.unreadCount > 0 && (
                      <span className="bg-blue-600 text-white text-xs font-bold px-2 py-1 rounded-full">
                        {chat.unreadCount}
                      </span>
                    )}
                  </div>
                  <p className="text-sm text-slate-700 truncate mb-2">{chat.lastMessage}</p>
                  <div className="flex items-center justify-between text-xs">
                    <span className="text-slate-500">{chat.startTime}</span>
                    {chat.assignedTo ? (
                      <span className="text-blue-600 font-medium">{chat.assignedTo}</span>
                    ) : (
                      <span className="text-yellow-600 font-medium">Unassigned</span>
                    )}
                  </div>
                </div>
              ))}
            </div>
          </motion.div>

          {/* Chat Window */}
          <div className="lg:col-span-2">
            {selectedChat ? (
              <motion.div
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                className="bg-white rounded-xl shadow-sm border border-slate-200 overflow-hidden flex flex-col h-[700px]"
              >
                {/* Chat Header */}
                <div className="p-4 border-b border-slate-200 bg-slate-50">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-3">
                      <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
                        <UserCircleIcon className="w-8 h-8 text-blue-600" />
                      </div>
                      <div>
                        <h3 className="font-bold text-slate-900">Tech Solutions Ltd</h3>
                        <div className="flex items-center space-x-2 text-sm text-slate-600">
                          <CheckCircleIcon className="w-4 h-4 text-green-600" />
                          <span>Active • Assigned to Sarah Johnson</span>
                        </div>
                      </div>
                    </div>
                    <div className="flex items-center space-x-2">
                      <button className="p-2 hover:bg-slate-200 rounded-lg transition-colors">
                        <PhoneIcon className="w-5 h-5 text-slate-600" />
                      </button>
                      <button className="p-2 hover:bg-slate-200 rounded-lg transition-colors">
                        <VideoCameraIcon className="w-5 h-5 text-slate-600" />
                      </button>
                      <button className="p-2 hover:bg-slate-200 rounded-lg transition-colors">
                        <InformationCircleIcon className="w-5 h-5 text-slate-600" />
                      </button>
                      <button className="p-2 hover:bg-slate-200 rounded-lg transition-colors">
                        <EllipsisVerticalIcon className="w-5 h-5 text-slate-600" />
                      </button>
                    </div>
                  </div>
                </div>

                {/* Messages */}
                <div className="flex-1 overflow-y-auto p-4 space-y-4 bg-slate-50">
                  <AnimatePresence>
                    {messages.map((msg, index) => (
                      <motion.div
                        key={msg.id}
                        initial={{ opacity: 0, y: 10 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ delay: index * 0.05 }}
                        className={`flex ${msg.sender === 'staff' ? 'justify-end' : 'justify-start'}`}
                      >
                        <div
                          className={`max-w-[70%] rounded-lg px-4 py-3 ${
                            msg.sender === 'user'
                              ? 'bg-white border border-slate-200'
                              : msg.sender === 'ai'
                              ? 'bg-gradient-to-r from-purple-100 to-blue-100 border border-purple-200'
                              : 'bg-blue-600 text-white'
                          }`}
                        >
                          <div className="flex items-center space-x-2 mb-1">
                            <span className={`text-xs font-semibold ${
                              msg.sender === 'staff' ? 'text-blue-100' : 'text-slate-600'
                            }`}>
                              {msg.sender === 'ai' && <SparklesIcon className="w-3 h-3 inline mr-1" />}
                              {msg.senderName}
                            </span>
                            <span className={`text-xs ${
                              msg.sender === 'staff' ? 'text-blue-100' : 'text-slate-500'
                            }`}>
                              {msg.timestamp}
                            </span>
                          </div>
                          <p className={`text-sm ${
                            msg.sender === 'staff' ? 'text-white' : 'text-slate-900'
                          }`}>
                            {msg.message}
                          </p>
                        </div>
                      </motion.div>
                    ))}
                  </AnimatePresence>
                  <div ref={messagesEndRef} />
                </div>

                {/* AI Suggestions (when enabled) */}
                {aiAssistEnabled && (
                  <div className="px-4 py-2 bg-gradient-to-r from-purple-50 to-blue-50 border-t border-purple-200">
                    <div className="flex items-center space-x-2 text-xs">
                      <BoltIcon className="w-4 h-4 text-purple-600" />
                      <span className="text-purple-900 font-semibold">AI Suggestions:</span>
                      <button className="px-3 py-1 bg-white hover:bg-purple-50 border border-purple-200 rounded-full text-purple-700 transition-colors">
                        Send VPS upgrade options
                      </button>
                      <button className="px-3 py-1 bg-white hover:bg-purple-50 border border-purple-200 rounded-full text-purple-700 transition-colors">
                        Check billing history
                      </button>
                    </div>
                  </div>
                )}

                {/* Message Input */}
                <div className="p-4 border-t border-slate-200 bg-white">
                  <div className="flex items-end space-x-2">
                    <button className="p-2 hover:bg-slate-100 rounded-lg transition-colors">
                      <PaperClipIcon className="w-6 h-6 text-slate-600" />
                    </button>
                    <button className="p-2 hover:bg-slate-100 rounded-lg transition-colors">
                      <FaceSmileIcon className="w-6 h-6 text-slate-600" />
                    </button>
                    <textarea
                      value={message}
                      onChange={(e) => setMessage(e.target.value)}
                      onKeyPress={(e) => {
                        if (e.key === 'Enter' && !e.shiftKey) {
                          e.preventDefault()
                          handleSendMessage()
                        }
                      }}
                      placeholder="Type your message..."
                      rows={2}
                      className="flex-1 px-4 py-2 border border-slate-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent resize-none"
                    />
                    <button
                      onClick={handleSendMessage}
                      className="p-3 bg-blue-600 hover:bg-blue-700 rounded-lg transition-colors"
                    >
                      <PaperAirplaneIcon className="w-6 h-6 text-white" />
                    </button>
                  </div>
                </div>
              </motion.div>
            ) : (
              <motion.div
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                className="bg-white rounded-xl shadow-sm border border-slate-200 h-[700px] flex items-center justify-center"
              >
                <div className="text-center">
                  <ChatBubbleLeftRightIcon className="w-16 h-16 text-slate-300 mx-auto mb-4" />
                  <h3 className="text-xl font-semibold text-slate-700 mb-2">
                    Select a Conversation
                  </h3>
                  <p className="text-slate-500">
                    Choose a chat from the list to start messaging
                  </p>
                </div>
              </motion.div>
            )}
          </div>
        </div>
      </div>
    </div>
  )
}
