'use client'

import { useState, useEffect } from 'react'
import { useSession } from 'next-auth/react'
import { useRouter } from 'next/navigation'
import { motion } from 'framer-motion'
import Link from 'next/link'
import {
  UserGroupIcon,
  ShoppingBagIcon,
  ChartBarIcon,
  CurrencyDollarIcon,
  ServerIcon,
  DocumentTextIcon,
  BellIcon,
  Cog6ToothIcon,
  ArrowTrendingUpIcon,
  ArrowTrendingDownIcon,
  CheckCircleIcon,
  ClockIcon,
  ExclamationTriangleIcon,
  XCircleIcon,
  EyeIcon,
  ChatBubbleLeftRightIcon,
  ShieldCheckIcon,
  GlobeAltIcon
} from '@heroicons/react/24/outline'

interface DashboardStats {
  totalUsers: number
  userGrowth: number
  totalOrders: number
  orderGrowth: number
  totalRevenue: number
  revenueGrowth: number
  activeServices: number
  serviceGrowth: number
}

interface RecentOrder {
  id: string
  customer: string
  service: string
  amount: number
  status: 'pending' | 'processing' | 'completed' | 'cancelled'
  date: string
}

interface SystemMetric {
  name: string
  value: string
  status: 'healthy' | 'warning' | 'critical'
  uptime: string
}

export default function AdminDashboard() {
  const { data: session, status } = useSession()
  const router = useRouter()
  const [loading, setLoading] = useState(true)
  const [stats, setStats] = useState<DashboardStats>({
    totalUsers: 0,
    userGrowth: 0,
    totalOrders: 0,
    orderGrowth: 0,
    totalRevenue: 0,
    revenueGrowth: 0,
    activeServices: 0,
    serviceGrowth: 0
  })
  const [recentOrders, setRecentOrders] = useState<RecentOrder[]>([])
  const [systemMetrics, setSystemMetrics] = useState<SystemMetric[]>([])

  useEffect(() => {
    if (status === 'unauthenticated') {
      router.push('/auth/signin')
    }
    
    // Simulate loading data
    setTimeout(() => {
      setStats({
        totalUsers: 1247,
        userGrowth: 12.5,
        totalOrders: 3456,
        orderGrowth: 8.3,
        totalRevenue: 456789,
        revenueGrowth: 15.7,
        activeServices: 2891,
        serviceGrowth: 5.2
      })

      setRecentOrders([
        {
          id: 'ORD-001',
          customer: 'Tech Solutions Ltd',
          service: 'VPS Hosting - Professional',
          amount: 48.88,
          status: 'completed',
          date: '2025-11-13'
        },
        {
          id: 'ORD-002',
          customer: 'Digital Marketing Co',
          service: 'Dedicated Server - Xeon E-2236',
          amount: 99.88,
          status: 'processing',
          date: '2025-11-13'
        },
        {
          id: 'ORD-003',
          customer: 'Startup Ventures',
          service: 'Shared Hosting - Business',
          amount: 7.99,
          status: 'completed',
          date: '2025-11-12'
        },
        {
          id: 'ORD-004',
          customer: 'E-commerce Plus',
          service: 'Reseller Hosting - Standard',
          amount: 19.97,
          status: 'pending',
          date: '2025-11-12'
        },
        {
          id: 'ORD-005',
          customer: 'Web Agency Pro',
          service: 'Domain Registration - .com',
          amount: 12.99,
          status: 'completed',
          date: '2025-11-11'
        }
      ])

      setSystemMetrics([
        {
          name: 'Web Servers',
          value: '98.7%',
          status: 'healthy',
          uptime: '99.98%'
        },
        {
          name: 'Database Cluster',
          value: '99.2%',
          status: 'healthy',
          uptime: '99.99%'
        },
        {
          name: 'Email Services',
          value: '95.3%',
          status: 'warning',
          uptime: '99.95%'
        },
        {
          name: 'Control Panel',
          value: '99.8%',
          status: 'healthy',
          uptime: '99.97%'
        }
      ])

      setLoading(false)
    }, 1000)
  }, [status, router])

  if (status === 'loading' || loading) {
    return (
      <div className="min-h-screen bg-slate-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-slate-600">Loading dashboard...</p>
        </div>
      </div>
    )
  }

  const statCards = [
    {
      title: 'Total Users',
      value: stats.totalUsers.toLocaleString(),
      growth: stats.userGrowth,
      icon: UserGroupIcon,
      color: 'blue',
      link: '/admin/users'
    },
    {
      title: 'Total Orders',
      value: stats.totalOrders.toLocaleString(),
      growth: stats.orderGrowth,
      icon: ShoppingBagIcon,
      color: 'green',
      link: '/admin/orders'
    },
    {
      title: 'Revenue (30d)',
      value: `$${(stats.totalRevenue / 1000).toFixed(1)}K`,
      growth: stats.revenueGrowth,
      icon: CurrencyDollarIcon,
      color: 'purple',
      link: '/admin/revenue'
    },
    {
      title: 'Active Services',
      value: stats.activeServices.toLocaleString(),
      growth: stats.serviceGrowth,
      icon: ServerIcon,
      color: 'orange',
      link: '/admin/services'
    }
  ]

  const quickActions = [
    {
      title: 'User Management',
      description: 'Manage users, roles & permissions',
      icon: UserGroupIcon,
      color: 'blue',
      link: '/admin/users'
    },
    {
      title: 'Order Management',
      description: 'View and process orders',
      icon: ShoppingBagIcon,
      color: 'green',
      link: '/admin/orders'
    },
    {
      title: 'Content Management',
      description: 'Edit pages and content',
      icon: DocumentTextIcon,
      color: 'purple',
      link: '/admin/cms'
    },
    {
      title: 'Analytics & Reports',
      description: 'View detailed analytics',
      icon: ChartBarIcon,
      color: 'orange',
      link: '/admin/analytics'
    },
    {
      title: 'System Settings',
      description: 'Configure system settings',
      icon: Cog6ToothIcon,
      color: 'slate',
      link: '/admin/settings'
    },
    {
      title: 'Live Chat',
      description: 'Manage customer support',
      icon: ChatBubbleLeftRightIcon,
      color: 'indigo',
      link: '/admin/chat'
    }
  ]

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed':
        return 'bg-green-100 text-green-800 border-green-200'
      case 'processing':
        return 'bg-blue-100 text-blue-800 border-blue-200'
      case 'pending':
        return 'bg-yellow-100 text-yellow-800 border-yellow-200'
      case 'cancelled':
        return 'bg-red-100 text-red-800 border-red-200'
      default:
        return 'bg-gray-100 text-gray-800 border-gray-200'
    }
  }

  const getMetricStatusIcon = (status: string) => {
    switch (status) {
      case 'healthy':
        return <CheckCircleIcon className="w-5 h-5 text-green-600" />
      case 'warning':
        return <ExclamationTriangleIcon className="w-5 h-5 text-yellow-600" />
      case 'critical':
        return <XCircleIcon className="w-5 h-5 text-red-600" />
      default:
        return <ClockIcon className="w-5 h-5 text-gray-600" />
    }
  }

  return (
    <div className="min-h-screen bg-slate-50">
      {/* Header */}
      <header className="bg-slate-900 text-white shadow-lg">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-3xl font-bold">Admin Dashboard</h1>
              <p className="text-slate-300 mt-1">
                Welcome back, {session?.user?.name || 'Administrator'}
              </p>
            </div>
            <div className="flex items-center space-x-4">
              <Link
                href="/"
                className="px-4 py-2 bg-slate-800 hover:bg-slate-700 rounded-lg transition-colors flex items-center"
              >
                <GlobeAltIcon className="w-5 h-5 mr-2" />
                View Site
              </Link>
              <button className="relative p-2 bg-slate-800 hover:bg-slate-700 rounded-lg transition-colors">
                <BellIcon className="w-6 h-6" />
                <span className="absolute top-0 right-0 w-2 h-2 bg-red-500 rounded-full"></span>
              </button>
            </div>
          </div>
        </div>
      </header>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Stats Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          {statCards.map((stat, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: index * 0.1 }}
            >
              <Link href={stat.link}>
                <div className="bg-white rounded-xl shadow-sm border border-slate-200 p-6 hover:shadow-md transition-all hover:border-blue-300 cursor-pointer group">
                  <div className="flex items-center justify-between mb-4">
                    <div className={`p-3 bg-${stat.color}-100 rounded-lg group-hover:scale-110 transition-transform`}>
                      <stat.icon className={`w-6 h-6 text-${stat.color}-600`} />
                    </div>
                    <div className="flex items-center text-sm">
                      {stat.growth >= 0 ? (
                        <ArrowTrendingUpIcon className="w-4 h-4 text-green-600 mr-1" />
                      ) : (
                        <ArrowTrendingDownIcon className="w-4 h-4 text-red-600 mr-1" />
                      )}
                      <span className={stat.growth >= 0 ? 'text-green-600' : 'text-red-600'}>
                        {Math.abs(stat.growth)}%
                      </span>
                    </div>
                  </div>
                  <h3 className="text-sm font-medium text-slate-600 mb-1">{stat.title}</h3>
                  <p className="text-2xl font-bold text-slate-900">{stat.value}</p>
                </div>
              </Link>
            </motion.div>
          ))}
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Main Content - 2 columns */}
          <div className="lg:col-span-2 space-y-8">
            {/* Quick Actions */}
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.4 }}
              className="bg-white rounded-xl shadow-sm border border-slate-200 p-6"
            >
              <h2 className="text-xl font-bold text-slate-900 mb-6 flex items-center">
                <ShieldCheckIcon className="w-6 h-6 mr-2 text-blue-600" />
                Quick Actions
              </h2>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                {quickActions.map((action, index) => (
                  <Link key={index} href={action.link}>
                    <div className="p-4 border border-slate-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition-all cursor-pointer group">
                      <div className="flex items-start">
                        <div className={`p-2 bg-${action.color}-100 rounded-lg mr-3 group-hover:scale-110 transition-transform`}>
                          <action.icon className={`w-5 h-5 text-${action.color}-600`} />
                        </div>
                        <div>
                          <h3 className="font-semibold text-slate-900 group-hover:text-blue-600 transition-colors">
                            {action.title}
                          </h3>
                          <p className="text-sm text-slate-600 mt-1">{action.description}</p>
                        </div>
                      </div>
                    </div>
                  </Link>
                ))}
              </div>
            </motion.div>

            {/* Recent Orders */}
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.5 }}
              className="bg-white rounded-xl shadow-sm border border-slate-200 p-6"
            >
              <div className="flex items-center justify-between mb-6">
                <h2 className="text-xl font-bold text-slate-900 flex items-center">
                  <ShoppingBagIcon className="w-6 h-6 mr-2 text-green-600" />
                  Recent Orders
                </h2>
                <Link
                  href="/admin/orders"
                  className="text-sm text-blue-600 hover:text-blue-700 font-medium flex items-center"
                >
                  View All
                  <EyeIcon className="w-4 h-4 ml-1" />
                </Link>
              </div>
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead>
                    <tr className="border-b border-slate-200">
                      <th className="text-left py-3 px-2 text-sm font-semibold text-slate-700">Order ID</th>
                      <th className="text-left py-3 px-2 text-sm font-semibold text-slate-700">Customer</th>
                      <th className="text-left py-3 px-2 text-sm font-semibold text-slate-700">Service</th>
                      <th className="text-right py-3 px-2 text-sm font-semibold text-slate-700">Amount</th>
                      <th className="text-center py-3 px-2 text-sm font-semibold text-slate-700">Status</th>
                    </tr>
                  </thead>
                  <tbody>
                    {recentOrders.map((order, index) => (
                      <tr key={index} className="border-b border-slate-100 hover:bg-slate-50">
                        <td className="py-3 px-2 text-sm font-medium text-slate-900">{order.id}</td>
                        <td className="py-3 px-2 text-sm text-slate-700">{order.customer}</td>
                        <td className="py-3 px-2 text-sm text-slate-600">{order.service}</td>
                        <td className="py-3 px-2 text-sm text-right font-semibold text-slate-900">
                          ${order.amount.toFixed(2)}
                        </td>
                        <td className="py-3 px-2">
                          <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full border ${getStatusColor(order.status)}`}>
                            {order.status}
                          </span>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </motion.div>
          </div>

          {/* Sidebar - 1 column */}
          <div className="space-y-8">
            {/* System Health */}
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.6 }}
              className="bg-white rounded-xl shadow-sm border border-slate-200 p-6"
            >
              <h2 className="text-xl font-bold text-slate-900 mb-6 flex items-center">
                <ServerIcon className="w-6 h-6 mr-2 text-purple-600" />
                System Health
              </h2>
              <div className="space-y-4">
                {systemMetrics.map((metric, index) => (
                  <div key={index} className="flex items-center justify-between p-3 bg-slate-50 rounded-lg">
                    <div className="flex items-center space-x-3">
                      {getMetricStatusIcon(metric.status)}
                      <div>
                        <p className="text-sm font-semibold text-slate-900">{metric.name}</p>
                        <p className="text-xs text-slate-600">Uptime: {metric.uptime}</p>
                      </div>
                    </div>
                    <span className="text-sm font-bold text-slate-900">{metric.value}</span>
                  </div>
                ))}
              </div>
              <Link
                href="/status"
                className="mt-4 block text-center py-2 bg-slate-100 hover:bg-slate-200 rounded-lg text-sm font-medium text-slate-700 transition-colors"
              >
                View Full Status
              </Link>
            </motion.div>

            {/* Quick Stats */}
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.7 }}
              className="bg-gradient-to-br from-blue-600 to-indigo-600 rounded-xl shadow-sm p-6 text-white"
            >
              <h2 className="text-xl font-bold mb-6 flex items-center">
                <ChartBarIcon className="w-6 h-6 mr-2" />
                Today's Overview
              </h2>
              <div className="space-y-4">
                <div className="flex justify-between items-center">
                  <span className="text-blue-100">New Users</span>
                  <span className="text-2xl font-bold">24</span>
                </div>
                <div className="flex justify-between items-center">
                  <span className="text-blue-100">New Orders</span>
                  <span className="text-2xl font-bold">47</span>
                </div>
                <div className="flex justify-between items-center">
                  <span className="text-blue-100">Revenue</span>
                  <span className="text-2xl font-bold">$3,247</span>
                </div>
                <div className="flex justify-between items-center">
                  <span className="text-blue-100">Support Tickets</span>
                  <span className="text-2xl font-bold">12</span>
                </div>
              </div>
            </motion.div>
          </div>
        </div>
      </div>
    </div>
  )
}
