'use client'

import { useState, useEffect } from 'react'
import { useSession } from 'next-auth/react'
import { useRouter } from 'next/navigation'
import {
  PencilIcon,
  TrashIcon,
  PlusIcon,
  CheckIcon,
  XMarkIcon,
  EyeIcon,
  EyeSlashIcon,
  ArrowPathIcon,
  SparklesIcon,
  ChartBarIcon,
  UserGroupIcon,
  EnvelopeIcon,
  CogIcon
} from '@heroicons/react/24/outline'

interface ContentItem {
  id: string
  section: string
  key: string
  value: string
  type: string
  order: number
  enabled: boolean
}

interface GroupedContent {
  [section: string]: ContentItem[]
}

const SECTION_INFO = {
  hero: {
    name: 'Hero Section',
    icon: SparklesIcon,
    color: 'blue',
    description: 'Main landing section with headline and call-to-actions'
  },
  stats: {
    name: 'Statistics Section',
    icon: ChartBarIcon,
    color: 'purple',
    description: 'Domain checker and business statistics'
  },
  services: {
    name: 'Services Section',
    icon: CogIcon,
    color: 'green',
    description: 'Services listing and descriptions'
  },
  whychooseus: {
    name: 'Why Choose Us',
    icon: UserGroupIcon,
    color: 'orange',
    description: 'Features and benefits section'
  },
  contact: {
    name: 'Contact Section',
    icon: EnvelopeIcon,
    color: 'pink',
    description: 'Contact information and form'
  }
}

export default function HomePageCMS() {
  const { data: session, status } = useSession()
  const router = useRouter()
  const [content, setContent] = useState<GroupedContent>({})
  const [loading, setLoading] = useState(true)
  const [saving, setSaving] = useState(false)
  const [editingId, setEditingId] = useState<string | null>(null)
  const [editValue, setEditValue] = useState('')
  const [addingNew, setAddingNew] = useState(false)
  const [newItem, setNewItem] = useState({
    section: 'hero',
    key: '',
    value: '',
    type: 'text',
    order: 0
  })
  const [activeSection, setActiveSection] = useState<string>('hero')
  const [message, setMessage] = useState<{ type: 'success' | 'error', text: string } | null>(null)

  useEffect(() => {
    if (status === 'unauthenticated') {
      router.push('/auth/signin')
    } else if (session?.user?.role !== 'ADMIN') {
      router.push('/')
    } else {
      fetchContent()
    }
  }, [session, status, router])

  const fetchContent = async () => {
    try {
      setLoading(true)
      const response = await fetch('/api/homepage')
      const result = await response.json()

      if (result.success) {
        // Convert flat data to grouped format
        const grouped: GroupedContent = {}
        Object.keys(result.data).forEach(section => {
          grouped[section] = Object.keys(result.data[section]).map(key => ({
            id: result.data[section][key].id,
            section,
            key,
            value: result.data[section][key].value,
            type: result.data[section][key].type,
            order: 0,
            enabled: true
          }))
        })
        setContent(grouped)
      }
    } catch (error) {
      console.error('Error fetching content:', error)
      showMessage('error', 'Failed to load content')
    } finally {
      setLoading(false)
    }
  }

  const showMessage = (type: 'success' | 'error', text: string) => {
    setMessage({ type, text })
    setTimeout(() => setMessage(null), 5000)
  }

  const handleEdit = (item: ContentItem) => {
    setEditingId(item.id)
    setEditValue(item.value)
  }

  const handleSave = async (item: ContentItem) => {
    try {
      setSaving(true)
      const response = await fetch('/api/homepage', {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          id: item.id,
          value: editValue
        })
      })

      const result = await response.json()

      if (result.success) {
        showMessage('success', 'Content updated successfully')
        fetchContent()
        setEditingId(null)
      } else {
        showMessage('error', result.error || 'Failed to update content')
      }
    } catch (error) {
      console.error('Error updating content:', error)
      showMessage('error', 'Failed to update content')
    } finally {
      setSaving(false)
    }
  }

  const handleToggleEnabled = async (item: ContentItem) => {
    try {
      const response = await fetch('/api/homepage', {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          id: item.id,
          enabled: !item.enabled
        })
      })

      const result = await response.json()

      if (result.success) {
        showMessage('success', `Content ${!item.enabled ? 'enabled' : 'disabled'}`)
        fetchContent()
      } else {
        showMessage('error', result.error || 'Failed to toggle content')
      }
    } catch (error) {
      console.error('Error toggling content:', error)
      showMessage('error', 'Failed to toggle content')
    }
  }

  const handleDelete = async (id: string) => {
    if (!confirm('Are you sure you want to delete this content?')) return

    try {
      const response = await fetch(`/api/homepage?id=${id}`, {
        method: 'DELETE'
      })

      const result = await response.json()

      if (result.success) {
        showMessage('success', 'Content deleted successfully')
        fetchContent()
      } else {
        showMessage('error', result.error || 'Failed to delete content')
      }
    } catch (error) {
      console.error('Error deleting content:', error)
      showMessage('error', 'Failed to delete content')
    }
  }

  const handleAddNew = async () => {
    try {
      setSaving(true)
      const response = await fetch('/api/homepage', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(newItem)
      })

      const result = await response.json()

      if (result.success) {
        showMessage('success', 'Content added successfully')
        fetchContent()
        setAddingNew(false)
        setNewItem({
          section: 'hero',
          key: '',
          value: '',
          type: 'text',
          order: 0
        })
      } else {
        showMessage('error', result.error || 'Failed to add content')
      }
    } catch (error) {
      console.error('Error adding content:', error)
      showMessage('error', 'Failed to add content')
    } finally {
      setSaving(false)
    }
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <ArrowPathIcon className="w-8 h-8 animate-spin text-blue-600 mx-auto mb-4" />
          <p className="text-gray-600">Loading content...</p>
        </div>
      </div>
    )
  }

  const sectionItems = content[activeSection] || []

  return (
    <div className="min-h-screen bg-gradient-to-br from-gray-50 to-gray-100">
      {/* Header */}
      <div className="bg-white border-b border-gray-200 sticky top-0 z-10 shadow-sm">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-3xl font-bold text-gray-900">Homepage Content Manager</h1>
              <p className="text-gray-600 mt-1">Manage every element of your homepage</p>
            </div>
            <button
              onClick={() => setAddingNew(true)}
              className="inline-flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
            >
              <PlusIcon className="w-5 h-5" />
              Add Content
            </button>
          </div>
        </div>
      </div>

      {/* Message Banner */}
      {message && (
        <div className={`${message.type === 'success' ? 'bg-green-50 border-green-200' : 'bg-red-50 border-red-200'} border-l-4 p-4`}>
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <p className={`${message.type === 'success' ? 'text-green-800' : 'text-red-800'}`}>
              {message.text}
            </p>
          </div>
        </div>
      )}

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div className="grid lg:grid-cols-4 gap-6">
          {/* Sidebar - Section Navigation */}
          <div className="lg:col-span-1">
            <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-4 sticky top-24">
              <h2 className="text-sm font-semibold text-gray-900 mb-3 uppercase tracking-wide">Sections</h2>
              <nav className="space-y-1">
                {Object.entries(SECTION_INFO).map(([key, info]) => {
                  const Icon = info.icon
                  const isActive = activeSection === key
                  return (
                    <button
                      key={key}
                      onClick={() => setActiveSection(key)}
                      className={`w-full flex items-center gap-3 px-3 py-2.5 rounded-lg transition-all ${
                        isActive
                          ? `bg-${info.color}-50 text-${info.color}-700 font-semibold`
                          : 'text-gray-700 hover:bg-gray-50'
                      }`}
                    >
                      <Icon className={`w-5 h-5 ${isActive ? `text-${info.color}-600` : 'text-gray-400'}`} />
                      <span className="text-sm">{info.name}</span>
                      <span className={`ml-auto text-xs px-2 py-0.5 rounded-full ${
                        isActive ? `bg-${info.color}-100 text-${info.color}-700` : 'bg-gray-100 text-gray-600'
                      }`}>
                        {content[key]?.length || 0}
                      </span>
                    </button>
                  )
                })}
              </nav>
            </div>
          </div>

          {/* Main Content Area */}
          <div className="lg:col-span-3">
            <div className="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
              {/* Section Header */}
              <div className={`bg-gradient-to-r from-${SECTION_INFO[activeSection as keyof typeof SECTION_INFO]?.color}-500 to-${SECTION_INFO[activeSection as keyof typeof SECTION_INFO]?.color}-600 px-6 py-4`}>
                <h2 className="text-xl font-bold text-white">
                  {SECTION_INFO[activeSection as keyof typeof SECTION_INFO]?.name}
                </h2>
                <p className="text-white/90 text-sm mt-1">
                  {SECTION_INFO[activeSection as keyof typeof SECTION_INFO]?.description}
                </p>
              </div>

              {/* Content Items */}
              <div className="divide-y divide-gray-200">
                {sectionItems.length === 0 ? (
                  <div className="px-6 py-12 text-center text-gray-500">
                    <p>No content items in this section yet.</p>
                    <button
                      onClick={() => {
                        setNewItem({ ...newItem, section: activeSection })
                        setAddingNew(true)
                      }}
                      className="mt-4 text-blue-600 hover:text-blue-700 font-semibold"
                    >
                      Add your first item
                    </button>
                  </div>
                ) : (
                  sectionItems.map((item) => (
                    <div key={item.id} className={`px-6 py-4 ${!item.enabled ? 'bg-gray-50 opacity-60' : ''}`}>
                      <div className="flex items-start justify-between gap-4">
                        <div className="flex-1 min-w-0">
                          {/* Key */}
                          <div className="flex items-center gap-2 mb-2">
                            <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
                              {item.key}
                            </span>
                            <span className="text-xs text-gray-500">({item.type})</span>
                          </div>

                          {/* Value - Editable */}
                          {editingId === item.id ? (
                            <div className="space-y-2">
                              {item.type === 'textarea' ? (
                                <textarea
                                  value={editValue}
                                  onChange={(e) => setEditValue(e.target.value)}
                                  rows={4}
                                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                                />
                              ) : (
                                <input
                                  type="text"
                                  value={editValue}
                                  onChange={(e) => setEditValue(e.target.value)}
                                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                                />
                              )}
                              <div className="flex gap-2">
                                <button
                                  onClick={() => handleSave(item)}
                                  disabled={saving}
                                  className="inline-flex items-center gap-1 px-3 py-1.5 bg-green-600 text-white text-sm rounded-lg hover:bg-green-700 disabled:opacity-50"
                                >
                                  <CheckIcon className="w-4 h-4" />
                                  Save
                                </button>
                                <button
                                  onClick={() => {
                                    setEditingId(null)
                                    setEditValue('')
                                  }}
                                  className="inline-flex items-center gap-1 px-3 py-1.5 bg-gray-200 text-gray-700 text-sm rounded-lg hover:bg-gray-300"
                                >
                                  <XMarkIcon className="w-4 h-4" />
                                  Cancel
                                </button>
                              </div>
                            </div>
                          ) : (
                            <p className={`text-sm ${item.type === 'textarea' ? 'whitespace-pre-wrap' : ''} ${!item.enabled ? 'line-through' : ''}`}>
                              {item.value}
                            </p>
                          )}
                        </div>

                        {/* Actions */}
                        {editingId !== item.id && (
                          <div className="flex items-center gap-2">
                            <button
                              onClick={() => handleEdit(item)}
                              className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                              title="Edit"
                            >
                              <PencilIcon className="w-5 h-5" />
                            </button>
                            <button
                              onClick={() => handleToggleEnabled(item)}
                              className={`p-2 ${item.enabled ? 'text-gray-600 hover:bg-gray-50' : 'text-green-600 hover:bg-green-50'} rounded-lg transition-colors`}
                              title={item.enabled ? 'Disable' : 'Enable'}
                            >
                              {item.enabled ? (
                                <EyeSlashIcon className="w-5 h-5" />
                              ) : (
                                <EyeIcon className="w-5 h-5" />
                              )}
                            </button>
                            <button
                              onClick={() => handleDelete(item.id)}
                              className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors"
                              title="Delete"
                            >
                              <TrashIcon className="w-5 h-5" />
                            </button>
                          </div>
                        )}
                      </div>
                    </div>
                  ))
                )}
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Add New Content Modal */}
      {addingNew && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl shadow-2xl max-w-2xl w-full max-h-[90vh] overflow-y-auto">
            <div className="px-6 py-4 border-b border-gray-200">
              <h3 className="text-xl font-bold text-gray-900">Add New Content</h3>
            </div>

            <div className="px-6 py-4 space-y-4">
              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Section</label>
                <select
                  value={newItem.section}
                  onChange={(e) => setNewItem({ ...newItem, section: e.target.value })}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                >
                  {Object.entries(SECTION_INFO).map(([key, info]) => (
                    <option key={key} value={key}>{info.name}</option>
                  ))}
                </select>
              </div>

              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Key</label>
                <input
                  type="text"
                  value={newItem.key}
                  onChange={(e) => setNewItem({ ...newItem, key: e.target.value })}
                  placeholder="e.g., title, subtitle, cta_text"
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                />
              </div>

              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Type</label>
                <select
                  value={newItem.type}
                  onChange={(e) => setNewItem({ ...newItem, type: e.target.value })}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                >
                  <option value="text">Text</option>
                  <option value="textarea">Textarea</option>
                  <option value="url">URL</option>
                  <option value="image">Image URL</option>
                  <option value="number">Number</option>
                </select>
              </div>

              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Value</label>
                {newItem.type === 'textarea' ? (
                  <textarea
                    value={newItem.value}
                    onChange={(e) => setNewItem({ ...newItem, value: e.target.value })}
                    rows={4}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                  />
                ) : (
                  <input
                    type={newItem.type === 'number' ? 'number' : 'text'}
                    value={newItem.value}
                    onChange={(e) => setNewItem({ ...newItem, value: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                  />
                )}
              </div>

              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Display Order</label>
                <input
                  type="number"
                  value={newItem.order}
                  onChange={(e) => setNewItem({ ...newItem, order: parseInt(e.target.value) || 0 })}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                />
              </div>
            </div>

            <div className="px-6 py-4 border-t border-gray-200 flex justify-end gap-3">
              <button
                onClick={() => {
                  setAddingNew(false)
                  setNewItem({
                    section: 'hero',
                    key: '',
                    value: '',
                    type: 'text',
                    order: 0
                  })
                }}
                className="px-4 py-2 bg-gray-200 text-gray-700 rounded-lg hover:bg-gray-300 transition-colors"
              >
                Cancel
              </button>
              <button
                onClick={handleAddNew}
                disabled={saving || !newItem.key || !newItem.value}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 transition-colors"
              >
                {saving ? 'Adding...' : 'Add Content'}
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  )
}
