'use client'

import { useState, useEffect, Suspense } from 'react'
import { useSession } from 'next-auth/react'
import { useRouter, useSearchParams } from 'next/navigation'
import { motion } from 'framer-motion'
import Link from 'next/link'
import dynamic from 'next/dynamic'
import {
  UserGroupIcon,
  CogIcon,
  ShoppingBagIcon,
  ChartBarIcon,
  ClipboardDocumentListIcon,
  CurrencyDollarIcon,
  ServerIcon,
  ExclamationTriangleIcon,
  CheckCircleIcon,
  ClockIcon,
  ArrowUpIcon,
  ArrowDownIcon,
  EyeIcon,
  PencilIcon,
  TrashIcon,
  MagnifyingGlassIcon,
  FunnelIcon,
  ChartPieIcon,
  BellIcon,
  CalendarIcon,
  DocumentTextIcon,
  FolderIcon,
  TagIcon,
  DocumentDuplicateIcon,
  PlusIcon
} from '@heroicons/react/24/outline'

// Dynamically import the WYSIWYG editor with no SSR
const WYSIWYGEditor = dynamic(() => import('@/components/WYSIWYGEditor'), {
  ssr: false,
  loading: () => (
    <div className="border border-gray-300 rounded-lg p-8 bg-gray-50 text-center">
      <div className="animate-pulse">
        <div className="h-4 bg-gray-200 rounded w-3/4 mx-auto mb-4"></div>
        <div className="h-4 bg-gray-200 rounded w-1/2 mx-auto"></div>
      </div>
      <p className="text-gray-500 mt-4">Loading editor...</p>
    </div>
  ),
})

interface User {
  id: string
  name: string
  email: string
  role: 'ADMIN' | 'CLIENT' | 'STAFF'
  status: 'active' | 'inactive' | 'suspended'
  joinDate: string
  lastLogin: string
  totalSpent: number
}

interface Order {
  id: string
  customer: string
  customerEmail: string
  service: string
  amount: number
  status: 'pending' | 'processing' | 'completed' | 'cancelled'
  date: string
}

interface ServiceStats {
  id: string
  name: string
  active: number
  suspended: number
  revenue: number
  growth: number
}

interface Content {
  id: string
  title: string
  slug: string
  type: string
  category: string
  status: string
  content?: string
  createdAt: string
  updatedAt: string
  author: {
    name: string
    email: string
  }
  _count: {
    versions: number
    comments: number
  }
}

const stats = [
  { 
    id: 1, 
    name: 'Total Users', 
    value: '2,847', 
    icon: UserGroupIcon, 
    change: '+12%', 
    trend: 'up',
    description: 'Active registered users'
  },
  { 
    id: 2, 
    name: 'Monthly Revenue', 
    value: '$127,540', 
    icon: CurrencyDollarIcon, 
    change: '+18%', 
    trend: 'up',
    description: 'This month vs last month'
  },
  { 
    id: 3, 
    name: 'Active Services', 
    value: '1,423', 
    icon: ServerIcon, 
    change: '+8%', 
    trend: 'up',
    description: 'Currently active hosting services'
  },
  { 
    id: 4, 
    name: 'Support Tickets', 
    value: '47', 
    icon: ExclamationTriangleIcon, 
    change: '-23%', 
    trend: 'down',
    description: 'Open tickets requiring attention'
  },
]

const contentTypes = ['PAGE', 'POST', 'BLOG', 'NEWS', 'SERVICE', 'PORTFOLIO', 'TESTIMONIAL', 'FAQ']
const contentStatuses = ['DRAFT', 'PUBLISHED', 'ARCHIVED', 'SCHEDULED']

function AdminDashboardContent() {
  const { data: session } = useSession()
  const router = useRouter()
  const searchParams = useSearchParams()
  const [activeTab, setActiveTab] = useState('dashboard')
  const [users, setUsers] = useState<User[]>([])
  const [orders, setOrders] = useState<Order[]>([])
  const [services, setServices] = useState<ServiceStats[]>([])
  const [content, setContent] = useState<Content[]>([])
  const [websitePages, setWebsitePages] = useState<any[]>([])
  const [allServices, setAllServices] = useState<any[]>([])
  const [loading, setLoading] = useState(true)
  const [searchTerm, setSearchTerm] = useState('')
  const [filterStatus, setFilterStatus] = useState('all')
  
  // CMS filters
  const [typeFilter, setTypeFilter] = useState('')
  const [statusFilter, setStatusFilter] = useState('')
  const [categoryFilter, setCategoryFilter] = useState('')
  const [cmsView, setCmsView] = useState<'pages' | 'content' | 'services'>('pages')

  // Editor modal state
  const [showEditor, setShowEditor] = useState(false)
  const [editingContent, setEditingContent] = useState<Content | null>(null)
  const [editorFormData, setEditorFormData] = useState({
    title: '',
    slug: '',
    type: '',
    category: '',
    status: '',
    content: ''
  })
  const [isSaving, setIsSaving] = useState(false)
  const [saveError, setSaveError] = useState<string | null>(null)

  useEffect(() => {
    // Read tab from URL parameter
    const tab = searchParams.get('tab')
    if (tab && ['dashboard', 'users', 'orders', 'services', 'cms', 'analytics', 'reports'].includes(tab)) {
      setActiveTab(tab)
    }
  }, [searchParams])

  useEffect(() => {
    fetchAdminData()
  }, [])

  const fetchAdminData = async () => {
    try {
      setLoading(true)
      
      // Fetch real CMS content from database
      try {
        const cmsResponse = await fetch('/api/cms/content?limit=100')
        if (cmsResponse.ok) {
          const cmsData = await cmsResponse.json()
          if (cmsData.content && Array.isArray(cmsData.content)) {
            setContent(cmsData.content.map((item: { id: number; title: string; slug: string; type: string; category: string; status: string; content: string; createdAt: string; updatedAt: string; author: { name: string; email: string }; _count: { versions: number; comments: number } }) => ({
              id: String(item.id),
              title: item.title,
              slug: item.slug,
              type: item.type,
              category: item.category || 'Main',
              status: item.status,
              content: item.content || '',
              createdAt: item.createdAt,
              updatedAt: item.updatedAt,
              author: item.author,
              _count: item._count
            })))
          }
        }
      } catch (error) {
        console.error('Error fetching CMS content:', error)
      }

      // Fetch website pages structure
      try {
        const pagesResponse = await fetch('/api/cms/pages')
        if (pagesResponse.ok) {
          const pagesData = await pagesResponse.json()
          setWebsitePages(pagesData.pages || [])
          setAllServices(pagesData.services || [])
        }
      } catch (error) {
        console.error('Error fetching pages:', error)
      }
      
      // Mock data for other sections - replace with actual API calls later
      setUsers([
        {
          id: '1',
          name: 'John Smith',
          email: 'john@example.com',
          role: 'CLIENT',
          status: 'active',
          joinDate: '2024-01-15',
          lastLogin: '2024-10-29',
          totalSpent: 489.99
        },
        {
          id: '2',
          name: 'Sarah Johnson',
          email: 'sarah@business.com',
          role: 'CLIENT',
          status: 'active',
          joinDate: '2024-03-22',
          lastLogin: '2024-10-28',
          totalSpent: 1249.50
        },
        {
          id: '3',
          name: 'Mike Wilson',
          email: 'mike@startup.io',
          role: 'CLIENT',
          status: 'suspended',
          joinDate: '2024-02-10',
          lastLogin: '2024-10-20',
          totalSpent: 89.99
        }
      ])

      setOrders([
        {
          id: 'ORD-001',
          customer: 'John Smith',
          customerEmail: 'john@example.com',
          service: 'Business Hosting Pro',
          amount: 24.99,
          status: 'completed',
          date: '2024-10-29'
        },
        {
          id: 'ORD-002',
          customer: 'Sarah Johnson',
          customerEmail: 'sarah@business.com',
          service: 'Enterprise Hosting',
          amount: 49.99,
          status: 'processing',
          date: '2024-10-28'
        },
        {
          id: 'ORD-003',
          customer: 'Mike Wilson',
          customerEmail: 'mike@startup.io',
          service: 'Domain Registration',
          amount: 12.99,
          status: 'pending',
          date: '2024-10-27'
        }
      ])

      setServices([
        {
          id: '1',
          name: 'Business Hosting',
          active: 892,
          suspended: 23,
          revenue: 22298.08,
          growth: 12
        },
        {
          id: '2',
          name: 'Enterprise Hosting',
          active: 234,
          suspended: 5,
          revenue: 11699.66,
          growth: 8
        },
        {
          id: '3',
          name: 'Domain Services',
          active: 1547,
          suspended: 12,
          revenue: 20010.53,
          growth: 15
        }
      ])

      // Mock CMS content data
      setContent([
        {
          id: '1',
          title: 'Welcome to Nuwair Systems',
          slug: 'welcome-to-nuwair-systems',
          type: 'PAGE',
          category: 'Main',
          status: 'PUBLISHED',
          content: '<h1>Welcome to Nuwair Systems</h1><p>We are a leading provider of web hosting, domain registration, and digital solutions. Our mission is to empower businesses with reliable and scalable technology infrastructure.</p><h2>Our Services</h2><ul><li>Business Hosting Solutions</li><li>Domain Registration & Management</li><li>Website Development</li><li>Cloud Infrastructure</li></ul><blockquote>Trusted by over 2,000 businesses worldwide</blockquote>',
          createdAt: '2024-10-15T10:00:00Z',
          updatedAt: '2024-10-29T14:30:00Z',
          author: {
            name: 'Admin User',
            email: 'sales@nuwair.com'
          },
          _count: {
            versions: 3,
            comments: 0
          }
        },
        {
          id: '2',
          title: 'Our Premium Hosting Services',
          slug: 'premium-hosting-services',
          type: 'SERVICE',
          category: 'Hosting',
          status: 'PUBLISHED',
          content: '<h1>Premium Hosting Services</h1><p>Experience enterprise-grade hosting with our premium plans.</p><h2>Features Include:</h2><ul><li>99.9% Uptime Guarantee</li><li>24/7 Technical Support</li><li>Free SSL Certificates</li><li>Daily Backups</li><li>DDoS Protection</li></ul><h3>Pricing</h3><p>Starting from <strong>$24.99/month</strong></p><p><em>Contact our sales team for custom enterprise solutions.</em></p>',
          createdAt: '2024-10-20T09:15:00Z',
          updatedAt: '2024-10-28T16:45:00Z',
          author: {
            name: 'Admin User',
            email: 'sales@nuwair.com'
          },
          _count: {
            versions: 2,
            comments: 5
          }
        },
        {
          id: '3',
          title: 'Latest Updates and Features',
          slug: 'latest-updates-features',
          type: 'BLOG',
          category: 'News',
          status: 'DRAFT',
          content: '<h1>Latest Updates - November 2024</h1><p>We are excited to announce several new features and improvements to our platform:</p><h2>New Features</h2><ol><li><strong>Advanced Dashboard:</strong> Redesigned admin interface with better analytics</li><li><strong>Enhanced Security:</strong> Two-factor authentication now available</li><li><strong>Performance:</strong> 50% faster page load times</li></ol><h2>Coming Soon</h2><p>Stay tuned for our upcoming <strong>AI-powered recommendations</strong> and <strong>automated backups</strong>.</p>',
          createdAt: '2024-10-29T08:30:00Z',
          updatedAt: '2024-10-29T12:15:00Z',
          author: {
            name: 'Admin User',
            email: 'sales@nuwair.com'
          },
          _count: {
            versions: 1,
            comments: 0
          }
        }
      ])
    } catch (error) {
      console.error('Error fetching admin data:', error)
    } finally {
      setLoading(false)
    }
  }

  const navigation = [
    { name: 'Dashboard', id: 'dashboard', icon: ChartBarIcon },
    { name: 'Users', id: 'users', icon: UserGroupIcon },
    { name: 'Orders', id: 'orders', icon: ShoppingBagIcon },
    { name: 'Services', id: 'services', icon: ServerIcon },
    { name: 'Content Management', id: 'cms', icon: DocumentTextIcon },
    { name: 'Analytics', id: 'analytics', icon: ChartPieIcon },
    { name: 'Reports', id: 'reports', icon: ClipboardDocumentListIcon },
  ]

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active':
      case 'completed':
        return 'text-green-600 bg-green-100'
      case 'processing':
      case 'pending':
        return 'text-yellow-600 bg-yellow-100'
      case 'suspended':
      case 'cancelled':
        return 'text-red-600 bg-red-100'
      case 'inactive':
        return 'text-gray-600 bg-gray-100'
      default:
        return 'text-gray-600 bg-gray-100'
    }
  }

  const filteredUsers = users.filter(user => {
    const matchesSearch = user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         user.email.toLowerCase().includes(searchTerm.toLowerCase())
    const matchesFilter = filterStatus === 'all' || user.status === filterStatus
    return matchesSearch && matchesFilter
  })

  const filteredOrders = orders.filter(order => {
    const matchesSearch = order.customer.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         order.service.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         order.id.toLowerCase().includes(searchTerm.toLowerCase())
    const matchesFilter = filterStatus === 'all' || order.status === filterStatus
    return matchesSearch && matchesFilter
  })

  // Action Handlers
  const handleViewUser = (userId: string) => {
    console.log('View user:', userId)
    alert(`View user details for ID: ${userId}\n\nThis would open a modal or navigate to user details page.`)
  }

  const handleEditUser = (userId: string) => {
    console.log('Edit user:', userId)
    alert(`Edit user for ID: ${userId}\n\nThis would open an edit form or modal.`)
  }

  const handleDeleteUser = (userId: string) => {
    console.log('Delete user:', userId)
    if (confirm(`Are you sure you want to delete user ${userId}?`)) {
      alert('User would be deleted from the database.')
      // Add actual delete logic here
    }
  }

  const handleViewOrder = (orderId: string) => {
    console.log('View order:', orderId)
    // Navigate to order details page
    router.push(`/dashboard/orders/${orderId}`)
  }

  const handleEditOrder = (orderId: string) => {
    console.log('Edit order:', orderId)
    alert(`Edit order for ID: ${orderId}\n\nThis would open an edit form or modal.`)
  }

  const handleViewContent = (contentId: string) => {
    console.log('View content:', contentId)
    const item = content.find(c => c.id === contentId)
    if (item) {
      alert(`View content: ${item.title}\nSlug: ${item.slug}\nType: ${item.type}`)
    }
  }

  const handleEditContent = (contentId: string, contentItem?: any) => {
    console.log('=== handleEditContent called ===')
    console.log('contentId:', contentId)
    console.log('contentItem:', contentItem)
    
    // If content item is passed directly, use it. Otherwise search in the content array
    const item = contentItem || content.find(c => c.id === contentId)
    
    if (item) {
      console.log('Found content item:', {
        id: item.id,
        title: item.title,
        slug: item.slug,
        hasContent: !!item.content,
        contentLength: item.content?.length || 0
      })
      
      setEditingContent(item)
      setEditorFormData({
        title: item.title,
        slug: item.slug,
        type: item.type,
        category: item.category || 'Main',
        status: item.status,
        content: item.content || '' // Load actual content from the database
      })
      setSaveError(null)
      setShowEditor(true)
      console.log('Editor modal opened with content length:', (item.content || '').length)
    } else {
      console.error('Content not found:', contentId)
      alert('Error: Content not found. Please refresh the page and try again.')
    }
  }

  const handleSaveContent = async (publishStatus?: string) => {
    if (!editingContent) return

    setIsSaving(true)
    setSaveError(null)

    try {
      const dataToSave = {
        ...editorFormData,
        status: publishStatus || editorFormData.status
      }

      const response = await fetch(`/api/cms/content/${editingContent.id}`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(dataToSave)
      })

      const data = await response.json()

      if (!response.ok) {
        throw new Error(data.error || 'Failed to save content')
      }

      // Update local state with the updated content
      setContent(prevContent => 
        prevContent.map(item => 
          item.id === editingContent.id ? {
            ...item,
            ...dataToSave,
            updatedAt: new Date().toISOString()
          } : item
        )
      )

      // Show success message
      alert(`Content "${data.content?.title || editorFormData.title}" saved successfully!${publishStatus === 'PUBLISHED' ? '\n\nStatus: Published' : publishStatus === 'DRAFT' ? '\n\nStatus: Draft' : ''}`)
      
      setShowEditor(false)
      setEditingContent(null)
      
      // Refresh content list
      fetchAdminData()
    } catch (error) {
      console.error('Error saving content:', error)
      setSaveError(error instanceof Error ? error.message : 'Failed to save content')
    } finally {
      setIsSaving(false)
    }
  }

  const handleCreateSection = async (pageId: string, pageName: string, sectionId: string, sectionName: string) => {
    // Open editor with new content template
    const newContent: Content = {
      id: 'new',
      title: `${pageName} - ${sectionName}`,
      slug: `${pageId}-${sectionId}`,
      type: 'PAGE',
      category: pageName,
      status: 'DRAFT',
      content: `<h1>${sectionName}</h1><p>Add your content here...</p>`,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
      author: {
        name: 'Admin User',
        email: session?.user?.email || 'admin@nuwair.com'
      },
      _count: {
        versions: 0,
        comments: 0
      }
    }

    setEditingContent(newContent)
    setEditorFormData({
      title: newContent.title,
      slug: newContent.slug,
      type: newContent.type,
      category: newContent.category,
      status: newContent.status,
      content: newContent.content || ''
    })
    setSaveError(null)
    setShowEditor(true)
  }

  const handleSaveNewContent = async (publishStatus?: string) => {
    if (!editingContent || editingContent.id !== 'new') {
      // Use regular save for existing content
      return handleSaveContent(publishStatus)
    }

    setIsSaving(true)
    setSaveError(null)

    try {
      const dataToSave = {
        ...editorFormData,
        status: publishStatus || editorFormData.status
      }

      const response = await fetch('/api/cms/content', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(dataToSave)
      })

      const data = await response.json()

      if (!response.ok) {
        throw new Error(data.error || 'Failed to create content')
      }

      // Show success message
      alert(`Content "${data.content?.title || editorFormData.title}" created successfully!${publishStatus === 'PUBLISHED' ? '\n\nStatus: Published' : publishStatus === 'DRAFT' ? '\n\nStatus: Draft' : ''}`)
      
      setShowEditor(false)
      setEditingContent(null)
      
      // Refresh content list
      fetchAdminData()
    } catch (error) {
      console.error('Error creating content:', error)
      setSaveError(error instanceof Error ? error.message : 'Failed to create content')
    } finally {
      setIsSaving(false)
    }
  }

  const handleDeleteContent = async (contentId: string) => {
    console.log('Delete content:', contentId)
    const item = content.find(c => c.id === contentId)
    if (item && confirm(`Are you sure you want to delete "${item.title}"?\n\nThis will also delete:\n- All version history\n- All comments\n\nThis action cannot be undone.`)) {
      try {
        const response = await fetch(`/api/cms/content/${contentId}`, {
          method: 'DELETE'
        })

        const data = await response.json()

        if (!response.ok) {
          throw new Error(data.error || 'Failed to delete content')
        }

        // Update local state - remove the deleted content
        setContent(prevContent => prevContent.filter(c => c.id !== contentId))
        
        alert(`Content "${item.title}" has been deleted successfully.`)
      } catch (error) {
        console.error('Error deleting content:', error)
        alert(error instanceof Error ? error.message : 'Failed to delete content')
      }
    }
  }

  const handleDuplicateContent = (contentId: string) => {
    console.log('Duplicate content:', contentId)
    const item = content.find(c => c.id === contentId)
    if (item) {
      alert(`Duplicate content: ${item.title}\n\nThis would create a copy of the content.`)
    }
  }

  const renderDashboard = () => (
    <div className="space-y-8">
      {/* Stats Overview */}
      <div className="grid grid-cols-1 gap-6 sm:grid-cols-2 lg:grid-cols-4">
        {stats.map((stat, index) => (
          <motion.div
            key={stat.id}
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: index * 0.1 }}
            className="bg-white overflow-hidden shadow-sm rounded-lg border border-gray-200"
          >
            <div className="p-6">
              <div className="flex items-center">
                <div className="flex-shrink-0">
                  <div className="p-3 bg-blue-100 rounded-lg">
                    <stat.icon className="h-6 w-6 text-blue-600" />
                  </div>
                </div>
                <div className="ml-4 w-0 flex-1">
                  <dl>
                    <dt className="text-sm font-medium text-gray-500 truncate">
                      {stat.name}
                    </dt>
                    <dd className="flex items-baseline">
                      <div className="text-2xl font-semibold text-gray-900">
                        {stat.value}
                      </div>
                      <div className={`ml-2 flex items-baseline text-sm font-semibold ${
                        stat.trend === 'up' ? 'text-green-600' : 'text-red-600'
                      }`}>
                        {stat.trend === 'up' ? (
                          <ArrowUpIcon className="w-4 h-4 mr-1" />
                        ) : (
                          <ArrowDownIcon className="w-4 h-4 mr-1" />
                        )}
                        {stat.change}
                      </div>
                    </dd>
                    <dd className="text-xs text-gray-500 mt-1">
                      {stat.description}
                    </dd>
                  </dl>
                </div>
              </div>
            </div>
          </motion.div>
        ))}
      </div>

      {/* Quick Actions */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.4 }}
        className="bg-white rounded-lg shadow-sm border border-gray-200 p-6"
      >
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Quick Actions</h3>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <Link
            href="/admin/cms"
            className="flex items-center p-4 bg-blue-50 rounded-lg hover:bg-blue-100 transition-colors"
          >
            <PencilIcon className="w-6 h-6 text-blue-600 mr-3" />
            <div>
              <h4 className="font-medium text-blue-900">Manage Content</h4>
              <p className="text-sm text-blue-700">Edit website content and blog posts</p>
            </div>
          </Link>
          <button 
            onClick={() => setActiveTab('users')}
            className="flex items-center p-4 bg-green-50 rounded-lg hover:bg-green-100 transition-colors"
          >
            <UserGroupIcon className="w-6 h-6 text-green-600 mr-3" />
            <div>
              <h4 className="font-medium text-green-900">User Management</h4>
              <p className="text-sm text-green-700">Manage user accounts and permissions</p>
            </div>
          </button>
          <button 
            onClick={() => setActiveTab('analytics')}
            className="flex items-center p-4 bg-purple-50 rounded-lg hover:bg-purple-100 transition-colors"
          >
            <ChartPieIcon className="w-6 h-6 text-purple-600 mr-3" />
            <div>
              <h4 className="font-medium text-purple-900">View Analytics</h4>
              <p className="text-sm text-purple-700">Business insights and performance</p>
            </div>
          </button>
        </div>
      </motion.div>

      {/* Service Performance */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.5 }}
        className="bg-white rounded-lg shadow-sm border border-gray-200"
      >
        <div className="p-6 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">Service Performance</h3>
        </div>
        <div className="p-6">
          <div className="space-y-6">
            {services.map((service) => (
              <div key={service.id} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                <div className="flex-1">
                  <h4 className="font-medium text-gray-900">{service.name}</h4>
                  <div className="flex items-center space-x-4 mt-1 text-sm text-gray-600">
                    <span>{service.active} active</span>
                    <span className="text-red-600">{service.suspended} suspended</span>
                  </div>
                </div>
                <div className="text-right">
                  <p className="font-semibold text-gray-900">${service.revenue.toLocaleString()}</p>
                  <div className={`flex items-center text-sm ${
                    service.growth >= 0 ? 'text-green-600' : 'text-red-600'
                  }`}>
                    {service.growth >= 0 ? (
                      <ArrowUpIcon className="w-4 h-4 mr-1" />
                    ) : (
                      <ArrowDownIcon className="w-4 h-4 mr-1" />
                    )}
                    {Math.abs(service.growth)}% growth
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </motion.div>

      {/* Recent Activity */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.6 }}
        className="bg-white rounded-lg shadow-sm border border-gray-200"
      >
        <div className="p-6 border-b border-gray-200 flex justify-between items-center">
          <h3 className="text-lg font-semibold text-gray-900">Recent Orders</h3>
          <button
            onClick={() => setActiveTab('orders')}
            className="text-sm text-blue-600 hover:text-blue-800 font-medium"
          >
            View All Orders →
          </button>
        </div>
        <div className="overflow-x-auto">
          <table className="min-w-full divide-y divide-gray-200">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Order ID
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Customer
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Service
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Amount
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Status
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Date
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {orders.slice(0, 5).map((order) => (
                <tr key={order.id} className="hover:bg-gray-50">
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-blue-600">
                    {order.id}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div>
                      <div className="text-sm font-medium text-gray-900">{order.customer}</div>
                      <div className="text-sm text-gray-500">{order.customerEmail}</div>
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    {order.service}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                    ${order.amount}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full capitalize ${getStatusColor(order.status)}`}>
                      {order.status}
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    {order.date}
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </motion.div>
    </div>
  )

  if (!session?.user || session.user.role !== 'ADMIN') {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-50">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-gray-900 mb-4">Access Denied</h1>
          <p className="text-gray-600 mb-8">You need admin privileges to access this page.</p>
          <Link
            href="/"
            className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            Go Home
          </Link>
        </div>
      </div>
    )
  }

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-50">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
          <p className="text-gray-600 mt-4">Loading admin dashboard...</p>
        </div>
      </div>
    )
  }

  const renderUsers = () => (
    <div className="space-y-6">
      {/* Search and Filters */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <div className="flex flex-col md:flex-row justify-between items-start md:items-center space-y-4 md:space-y-0">
          <div className="relative flex-1 max-w-md">
            <MagnifyingGlassIcon className="w-5 h-5 text-gray-400 absolute left-3 top-1/2 transform -translate-y-1/2" />
            <input
              type="text"
              placeholder="Search users..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-10 pr-4 py-2 w-full border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
          
          <div className="flex space-x-2">
            {['all', 'active', 'inactive', 'suspended'].map((status) => (
              <button
                key={status}
                onClick={() => setFilterStatus(status)}
                className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors capitalize ${
                  filterStatus === status
                    ? 'bg-blue-100 text-blue-700'
                    : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                }`}
              >
                {status}
              </button>
            ))}
          </div>
        </div>
      </div>

      {/* Users Table */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="p-6 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">User Management</h3>
        </div>
        <div className="overflow-x-auto">
          <table className="min-w-full divide-y divide-gray-200">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  User
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Role
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Status
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Total Spent
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Join Date
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Last Login
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Actions
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {filteredUsers.map((user) => (
                <tr key={user.id} className="hover:bg-gray-50">
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div>
                      <div className="text-sm font-medium text-gray-900">{user.name}</div>
                      <div className="text-sm text-gray-500">{user.email}</div>
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full capitalize ${
                      user.role === 'ADMIN' ? 'bg-purple-100 text-purple-800' : 'bg-gray-100 text-gray-800'
                    }`}>
                      {user.role}
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full capitalize ${getStatusColor(user.status)}`}>
                      {user.status}
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                    ${user.totalSpent}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    {user.joinDate}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    {user.lastLogin}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm space-x-2">
                    <button 
                      onClick={() => handleViewUser(user.id)}
                      className="text-blue-600 hover:text-blue-900"
                      title="View User"
                    >
                      <EyeIcon className="w-4 h-4" />
                    </button>
                    <button 
                      onClick={() => handleEditUser(user.id)}
                      className="text-green-600 hover:text-green-900"
                      title="Edit User"
                    >
                      <PencilIcon className="w-4 h-4" />
                    </button>
                    <button 
                      onClick={() => handleDeleteUser(user.id)}
                      className="text-red-600 hover:text-red-900"
                      title="Delete User"
                    >
                      <TrashIcon className="w-4 h-4" />
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    </div>
  )

  const renderOrders = () => (
    <div className="space-y-6">
      {/* Search and Filters */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <div className="flex flex-col md:flex-row justify-between items-start md:items-center space-y-4 md:space-y-0">
          <div className="relative flex-1 max-w-md">
            <MagnifyingGlassIcon className="w-5 h-5 text-gray-400 absolute left-3 top-1/2 transform -translate-y-1/2" />
            <input
              type="text"
              placeholder="Search orders..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-10 pr-4 py-2 w-full border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
          
          <div className="flex space-x-2">
            {['all', 'pending', 'processing', 'completed', 'cancelled'].map((status) => (
              <button
                key={status}
                onClick={() => setFilterStatus(status)}
                className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors capitalize ${
                  filterStatus === status
                    ? 'bg-blue-100 text-blue-700'
                    : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                }`}
              >
                {status}
              </button>
            ))}
          </div>
        </div>
      </div>

      {/* Orders Table */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="p-6 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">Order Management</h3>
        </div>
        <div className="overflow-x-auto">
          <table className="min-w-full divide-y divide-gray-200">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Order ID
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Customer
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Service
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Amount
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Status
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Date
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Actions
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {filteredOrders.map((order) => (
                <tr key={order.id} className="hover:bg-gray-50">
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-blue-600">
                    {order.id}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div>
                      <div className="text-sm font-medium text-gray-900">{order.customer}</div>
                      <div className="text-sm text-gray-500">{order.customerEmail}</div>
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    {order.service}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                    ${order.amount}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full capitalize ${getStatusColor(order.status)}`}>
                      {order.status}
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    {order.date}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm space-x-2">
                    <button 
                      onClick={() => handleViewOrder(order.id)}
                      className="text-blue-600 hover:text-blue-900"
                      title="View Order"
                    >
                      <EyeIcon className="w-4 h-4" />
                    </button>
                    <button 
                      onClick={() => handleEditOrder(order.id)}
                      className="text-green-600 hover:text-green-900"
                      title="Edit Order"
                    >
                      <PencilIcon className="w-4 h-4" />
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    </div>
  )

  const renderAnalytics = () => (
    <div className="space-y-6">
      {/* Revenue Chart Placeholder */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Revenue Analytics</h3>
        <div className="h-64 bg-gray-100 rounded-lg flex items-center justify-center">
          <div className="text-center">
            <ChartBarIcon className="w-12 h-12 text-gray-400 mx-auto mb-2" />
            <p className="text-gray-500">Revenue chart will be displayed here</p>
          </div>
        </div>
      </div>

      {/* Service Usage */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Service Usage Distribution</h3>
        <div className="space-y-4">
          {services.map((service) => (
            <div key={service.id} className="flex items-center justify-between">
              <span className="text-sm font-medium text-gray-700">{service.name}</span>
              <div className="flex items-center space-x-3">
                <div className="w-32 bg-gray-200 rounded-full h-2">
                  <div 
                    className="bg-blue-600 h-2 rounded-full" 
                    style={{ width: `${(service.active / 1000) * 100}%` }}
                  ></div>
                </div>
                <span className="text-sm text-gray-600">{service.active}</span>
              </div>
            </div>
          ))}
        </div>
      </div>

      {/* Growth Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h4 className="text-md font-semibold text-gray-900 mb-4">User Growth</h4>
          <div className="h-32 bg-gray-100 rounded-lg flex items-center justify-center">
            <p className="text-gray-500">User growth chart</p>
          </div>
        </div>
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h4 className="text-md font-semibold text-gray-900 mb-4">Revenue Growth</h4>
          <div className="h-32 bg-gray-100 rounded-lg flex items-center justify-center">
            <p className="text-gray-500">Revenue growth chart</p>
          </div>
        </div>
      </div>
    </div>
  )

  const renderCMS = () => {
    const filteredContent = content.filter(item => {
      const matchesType = !typeFilter || item.type === typeFilter
      const matchesStatus = !statusFilter || item.status === statusFilter
      const matchesCategory = !categoryFilter || item.category === categoryFilter
      return matchesType && matchesStatus && matchesCategory
    })

    return (
      <div className="space-y-6">
        {/* CMS View Switcher */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
          <div className="flex space-x-2">
            <button
              onClick={() => setCmsView('pages')}
              className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                cmsView === 'pages'
                  ? 'bg-blue-600 text-white'
                  : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
              }`}
            >
              <DocumentTextIcon className="w-4 h-4 inline-block mr-2" />
              Website Pages
            </button>
            <button
              onClick={() => setCmsView('content')}
              className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                cmsView === 'content'
                  ? 'bg-blue-600 text-white'
                  : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
              }`}
            >
              <FolderIcon className="w-4 h-4 inline-block mr-2" />
              All Content
            </button>
            <button
              onClick={() => setCmsView('services')}
              className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                cmsView === 'services'
                  ? 'bg-blue-600 text-white'
                  : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
              }`}
            >
              <ShoppingBagIcon className="w-4 h-4 inline-block mr-2" />
              Services & Packages
            </button>
          </div>
        </div>

        {/* Pages View */}
        {cmsView === 'pages' && (
          <div className="space-y-4">
            <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Website Pages & Sections</h3>
              <p className="text-sm text-gray-600 mb-6">Manage content for each page and section of your website</p>
              
              <div className="space-y-4">
                {websitePages.map((page) => (
                  <div key={page.id} className="border border-gray-200 rounded-lg p-4 hover:border-blue-300 transition-colors">
                    <div className="flex items-center justify-between mb-3">
                      <div>
                        <h4 className="text-base font-semibold text-gray-900">{page.name}</h4>
                        <p className="text-sm text-gray-500">Route: {page.route}</p>
                      </div>
                      <span className={`px-3 py-1 rounded-full text-xs font-medium ${
                        page.hasContent ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-600'
                      }`}>
                        {page.hasContent ? `${page.content.length} sections` : 'No content'}
                      </span>
                    </div>
                    
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3 mt-3">
                      {page.sections.map((section: { id: string; name: string; description: string }) => {
                        // Try multiple slug patterns to find content:
                        // 1. page-section (e.g., home-hero)
                        // 2. section (e.g., services-hero - when section.id already has page prefix)
                        // 3. Match by category
                        const sectionContent = page.content.find((c: { slug: string; category: string }) => 
                          c.slug === `${page.id}-${section.id}` || 
                          c.slug === section.id ||
                          (c.slug.includes(section.id) && c.category === page.name)
                        )
                        
                        // Debug logging
                        if (sectionContent) {
                          console.log(`Found content for ${page.name} → ${section.name}:`, {
                            id: sectionContent.id,
                            slug: sectionContent.slug,
                            hasContent: !!sectionContent.content,
                            contentLength: sectionContent.content?.length || 0
                          })
                        }
                        
                        return (
                          <div key={section.id} className="border border-gray-200 rounded-lg p-3 bg-gray-50">
                            <div className="flex items-start justify-between">
                              <div className="flex-1">
                                <h5 className="text-sm font-medium text-gray-900">{section.name}</h5>
                                <p className="text-xs text-gray-500 mt-1">{section.description}</p>
                              </div>
                              {sectionContent && (
                                <span className={`ml-2 px-2 py-0.5 rounded text-xs font-medium ${
                                  sectionContent.status === 'PUBLISHED'
                                    ? 'bg-green-100 text-green-700'
                                    : 'bg-yellow-100 text-yellow-700'
                                }`}>
                                  {sectionContent.status}
                                </span>
                              )}
                            </div>
                            <div className="flex space-x-2 mt-3">
                              {sectionContent ? (
                                <>
                                  <button
                                    onClick={() => handleEditContent(String(sectionContent.id), sectionContent)}
                                    className="flex-1 px-3 py-1.5 bg-blue-600 text-white text-xs font-medium rounded hover:bg-blue-700 transition-colors"
                                  >
                                    <PencilIcon className="w-3 h-3 inline mr-1" />
                                    Edit
                                  </button>
                                  <button
                                    onClick={() => handleViewContent(String(sectionContent.id))}
                                    className="px-3 py-1.5 bg-gray-200 text-gray-700 text-xs font-medium rounded hover:bg-gray-300 transition-colors"
                                  >
                                    <EyeIcon className="w-3 h-3 inline" />
                                  </button>
                                </>
                              ) : (
                                <button
                                  onClick={() => handleCreateSection(page.id, page.name, section.id, section.name)}
                                  className="flex-1 px-3 py-1.5 border border-dashed border-gray-300 text-gray-600 text-xs font-medium rounded hover:border-blue-400 hover:text-blue-600 transition-colors"
                                >
                                  <PlusIcon className="w-3 h-3 inline mr-1" />
                                  Create Content
                                </button>
                              )}
                            </div>
                          </div>
                        )
                      })}
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>
        )}

        {/* Content View (existing table) */}
        {cmsView === 'content' && (
          <div className="space-y-6">
        {/* CMS Header with Actions */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between space-y-4 sm:space-y-0">
            <div>
              <h3 className="text-lg font-semibold text-gray-900">Content Management</h3>
              <p className="text-sm text-gray-600 mt-1">Create, edit, and manage your website content</p>
            </div>
            <div className="flex space-x-3">
              <Link
                href="/admin/cms/content/new"
                className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
              >
                <PlusIcon className="w-4 h-4 mr-2" />
                New Content
              </Link>
            </div>
          </div>

          {/* Filters */}
          <div className="mt-6 grid grid-cols-1 sm:grid-cols-3 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Content Type</label>
              <select
                value={typeFilter}
                onChange={(e) => setTypeFilter(e.target.value)}
                className="block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm"
              >
                <option value="">All Types</option>
                {contentTypes.map(type => (
                  <option key={type} value={type}>{type}</option>
                ))}
              </select>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Status</label>
              <select
                value={statusFilter}
                onChange={(e) => setStatusFilter(e.target.value)}
                className="block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm"
              >
                <option value="">All Statuses</option>
                {contentStatuses.map(status => (
                  <option key={status} value={status}>{status}</option>
                ))}
              </select>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Category</label>
              <input
                type="text"
                value={categoryFilter}
                onChange={(e) => setCategoryFilter(e.target.value)}
                placeholder="Filter by category..."
                className="block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm"
              />
            </div>
          </div>
        </div>

        {/* Content Stats */}
        <div className="grid grid-cols-1 sm:grid-cols-4 gap-4">
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
            <div className="flex items-center">
              <DocumentTextIcon className="w-8 h-8 text-blue-600" />
              <div className="ml-3">
                <p className="text-sm font-medium text-gray-700">Total Content</p>
                <p className="text-2xl font-bold text-gray-900">{content.length}</p>
              </div>
            </div>
          </div>
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
            <div className="flex items-center">
              <CheckCircleIcon className="w-8 h-8 text-green-600" />
              <div className="ml-3">
                <p className="text-sm font-medium text-gray-700">Published</p>
                <p className="text-2xl font-bold text-gray-900">
                  {content.filter(item => item.status === 'PUBLISHED').length}
                </p>
              </div>
            </div>
          </div>
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
            <div className="flex items-center">
              <ClockIcon className="w-8 h-8 text-yellow-600" />
              <div className="ml-3">
                <p className="text-sm font-medium text-gray-700">Draft</p>
                <p className="text-2xl font-bold text-gray-900">
                  {content.filter(item => item.status === 'DRAFT').length}
                </p>
              </div>
            </div>
          </div>
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
            <div className="flex items-center">
              <FolderIcon className="w-8 h-8 text-purple-600" />
              <div className="ml-3">
                <p className="text-sm font-medium text-gray-700">Categories</p>
                <p className="text-2xl font-bold text-gray-900">
                  {new Set(content.map(item => item.category)).size}
                </p>
              </div>
            </div>
          </div>
        </div>

        {/* Content Table */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
          <div className="px-6 py-4 border-b border-gray-200">
            <h4 className="text-lg font-medium text-gray-900">Content Items</h4>
          </div>
          <div className="overflow-x-auto">
            <table className="min-w-full divide-y divide-gray-200">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Title & Type
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Status
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Category
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Author
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Updated
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Actions
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {filteredContent.map((item) => (
                  <tr key={item.id} className="hover:bg-gray-50">
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div>
                        <div className="text-sm font-medium text-gray-900">{item.title}</div>
                        <div className="text-sm text-gray-500">
                          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
                            {item.type}
                          </span>
                        </div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                        item.status === 'PUBLISHED' ? 'bg-green-100 text-green-800' :
                        item.status === 'DRAFT' ? 'bg-yellow-100 text-yellow-800' :
                        item.status === 'ARCHIVED' ? 'bg-gray-100 text-gray-800' :
                        'bg-blue-100 text-blue-800'
                      }`}>
                        {item.status}
                      </span>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                      {item.category}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                      {item.author.name}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                      {new Date(item.updatedAt).toLocaleDateString()}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                      <div className="flex space-x-2">
                        <button 
                          onClick={() => handleViewContent(item.id)}
                          className="text-blue-600 hover:text-blue-900"
                          title="View Content"
                        >
                          <EyeIcon className="w-4 h-4" />
                        </button>
                        <button 
                          onClick={() => handleEditContent(item.id)}
                          className="text-gray-600 hover:text-gray-900"
                          title="Edit Content"
                        >
                          <PencilIcon className="w-4 h-4" />
                        </button>
                        <button 
                          onClick={() => handleDeleteContent(item.id)}
                          className="text-red-600 hover:text-red-900"
                          title="Delete Content"
                        >
                          <TrashIcon className="w-4 h-4" />
                        </button>
                        <button 
                          onClick={() => handleDuplicateContent(item.id)}
                          className="text-gray-600 hover:text-gray-900"
                          title="Duplicate Content"
                        >
                          <DocumentDuplicateIcon className="w-4 h-4" />
                        </button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>

        {filteredContent.length === 0 && (
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-12 text-center">
            <DocumentTextIcon className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No content found</h3>
            <p className="text-gray-600 mb-6">
              {typeFilter || statusFilter || categoryFilter ? 
                'No content matches your current filters.' : 
                'Get started by creating your first piece of content.'
              }
            </p>
            <Link
              href="/admin/cms/content/new"
              className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700"
            >
              <PlusIcon className="w-4 h-4 mr-2" />
              Create Content
            </Link>
          </div>
        )}
          </div>
        )}

        {/* Services View */}
        {cmsView === 'services' && (
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Services & Packages</h3>
            <p className="text-sm text-gray-600 mb-6">Manage your service offerings and pricing packages</p>
            
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {allServices.map((service) => (
                <div key={service.id} className="border border-gray-200 rounded-lg p-4 hover:border-blue-300 transition-colors">
                  <div className="flex items-start justify-between mb-3">
                    <div>
                      <h4 className="text-base font-semibold text-gray-900">{service.name}</h4>
                      <p className="text-sm text-gray-500 mt-1">{service.description}</p>
                    </div>
                  </div>
                  <div className="mt-3 space-y-2">
                    <div className="flex justify-between text-sm">
                      <span className="text-gray-600">Price:</span>
                      <span className="font-semibold text-gray-900">${service.price}</span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span className="text-gray-600">Billing:</span>
                      <span className="text-gray-700">{service.billingCycle}</span>
                    </div>
                  </div>
                  <div className="mt-4 flex space-x-2">
                    <button className="flex-1 px-3 py-2 bg-blue-600 text-white text-sm font-medium rounded hover:bg-blue-700 transition-colors">
                      <PencilIcon className="w-4 h-4 inline mr-1" />
                      Edit Service
                    </button>
                  </div>
                </div>
              ))}
            </div>
            
            {allServices.length === 0 && (
              <div className="text-center py-12">
                <ShoppingBagIcon className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                <h4 className="text-lg font-medium text-gray-900 mb-2">No Services Yet</h4>
                <p className="text-gray-600 mb-6">Create your first service offering</p>
                <button className="inline-flex items-center px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-lg hover:bg-blue-700 transition-colors">
                  <PlusIcon className="w-4 h-4 mr-2" />
                  Add Service
                </button>
              </div>
            )}
          </div>
        )}
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-white shadow-sm border-b border-gray-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <h1 className="text-xl font-semibold text-gray-900">Nuwair Admin Dashboard</h1>
            </div>
            <div className="flex items-center space-x-4">
              <button className="p-2 text-gray-400 hover:text-gray-600">
                <BellIcon className="w-5 h-5" />
              </button>
              <div className="text-sm text-gray-600">
                Welcome, {session?.user?.name || 'Admin'}
              </div>
              <Link
                href="/api/auth/signout"
                className="text-sm text-gray-600 hover:text-gray-900"
              >
                Sign Out
              </Link>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div className="flex flex-col lg:flex-row gap-8">
          {/* Sidebar */}
          <div className="lg:w-64 flex-shrink-0">
            <nav className="space-y-1">
              {navigation.map((item) => (
                <button
                  key={item.id}
                  onClick={() => {
                    setActiveTab(item.id)
                    setSearchTerm('')
                    setFilterStatus('all')
                    router.push(`/admin?tab=${item.id}`)
                  }}
                  className={`w-full flex items-center px-3 py-2 text-sm font-medium rounded-md transition-colors ${
                    activeTab === item.id
                      ? 'bg-blue-100 text-blue-700'
                      : 'text-gray-600 hover:bg-gray-100 hover:text-gray-900'
                  }`}
                >
                  <item.icon className="mr-3 h-5 w-5" />
                  {item.name}
                </button>
              ))}
            </nav>
          </div>

          {/* Main Content */}
          <div className="flex-1">
            <div className="mb-8">
              <h2 className="text-2xl font-bold text-gray-900 capitalize">{activeTab}</h2>
              <p className="mt-1 text-sm text-gray-600">
                {activeTab === 'dashboard' && 'Overview of your business performance and key metrics'}
                {activeTab === 'users' && 'Manage user accounts, roles, and permissions'}
                {activeTab === 'orders' && 'View and manage customer orders and transactions'}
                {activeTab === 'services' && 'Configure and monitor your hosting services'}
                {activeTab === 'cms' && 'Manage website content, pages, and media'}
                {activeTab === 'analytics' && 'Detailed analytics and business intelligence'}
                {activeTab === 'reports' && 'Generate comprehensive business reports'}
              </p>
            </div>

            {activeTab === 'dashboard' && renderDashboard()}
            {activeTab === 'users' && renderUsers()}
            {activeTab === 'orders' && renderOrders()}
            {activeTab === 'services' && (
              <div className="bg-white shadow-sm rounded-lg border border-gray-200 p-6">
                <h3 className="text-lg font-medium text-gray-900 mb-4">Service Management</h3>
                <p className="text-gray-600">Service management functionality will be implemented here.</p>
              </div>
            )}
            {activeTab === 'cms' && renderCMS()}
            {activeTab === 'analytics' && renderAnalytics()}
            {activeTab === 'reports' && (
              <div className="bg-white shadow-sm rounded-lg border border-gray-200 p-6">
                <h3 className="text-lg font-medium text-gray-900 mb-4">Reports & Business Intelligence</h3>
                <p className="text-gray-600">Advanced reporting functionality will be implemented here.</p>
              </div>
            )}
          </div>
        </div>
      </div>

      {/* Editor Modal */}
      {showEditor && editingContent && (
        <div className="fixed inset-0 z-50 overflow-y-auto">
          <div className="flex items-center justify-center min-h-screen px-4 pt-4 pb-20 text-center sm:block sm:p-0">
            {/* Background overlay */}
            <div 
              className="fixed inset-0 transition-opacity bg-gray-500 bg-opacity-75"
              onClick={() => setShowEditor(false)}
            ></div>

            {/* Modal panel */}
            <div className="inline-block align-bottom bg-white rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-4xl sm:w-full">
              {/* Header */}
              <div className="bg-gradient-to-r from-blue-600 to-purple-600 px-6 py-4">
                <div className="flex items-center justify-between">
                  <h3 className="text-lg font-semibold text-white flex items-center">
                    <PencilIcon className="w-5 h-5 mr-2" />
                    {editingContent.id === 'new' ? 'Create New Content' : `Edit Content: ${editingContent.title}`}
                  </h3>
                  <button
                    onClick={() => setShowEditor(false)}
                    className="text-white hover:text-gray-200 transition-colors"
                  >
                    <svg className="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                    </svg>
                  </button>
                </div>
              </div>

              {/* Content */}
              <div className="bg-white px-6 py-6 max-h-[70vh] overflow-y-auto">
                {/* Error Message */}
                {saveError && (
                  <div className="mb-4 p-4 bg-red-50 border border-red-200 rounded-lg">
                    <div className="flex items-center">
                      <svg className="w-5 h-5 text-red-500 mr-2" fill="currentColor" viewBox="0 0 20 20">
                        <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clipRule="evenodd" />
                      </svg>
                      <span className="text-sm text-red-800">{saveError}</span>
                    </div>
                  </div>
                )}

                <div className="space-y-4">
                  {/* Title */}
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Title
                    </label>
                    <input
                      type="text"
                      value={editorFormData.title}
                      onChange={(e) => setEditorFormData({...editorFormData, title: e.target.value})}
                      className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      placeholder="Enter content title"
                    />
                  </div>

                  {/* Slug */}
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Slug (URL)
                    </label>
                    <input
                      type="text"
                      value={editorFormData.slug}
                      onChange={(e) => setEditorFormData({...editorFormData, slug: e.target.value})}
                      className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      placeholder="content-url-slug"
                    />
                  </div>

                  {/* Row: Type, Category, Status */}
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Type
                      </label>
                      <select
                        value={editorFormData.type}
                        onChange={(e) => setEditorFormData({...editorFormData, type: e.target.value})}
                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      >
                        <option value="PAGE">Page</option>
                        <option value="BLOG">Blog Post</option>
                        <option value="SERVICE">Service</option>
                        <option value="PORTFOLIO">Portfolio</option>
                      </select>
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Category
                      </label>
                      <input
                        type="text"
                        value={editorFormData.category}
                        onChange={(e) => setEditorFormData({...editorFormData, category: e.target.value})}
                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="Main, News, etc."
                      />
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Status
                      </label>
                      <select
                        value={editorFormData.status}
                        onChange={(e) => setEditorFormData({...editorFormData, status: e.target.value})}
                        className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      >
                        <option value="DRAFT">Draft</option>
                        <option value="PUBLISHED">Published</option>
                        <option value="ARCHIVED">Archived</option>
                        <option value="SCHEDULED">Scheduled</option>
                      </select>
                    </div>
                  </div>

                  {/* Content Editor */}
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Content
                    </label>
                    <WYSIWYGEditor
                      content={editorFormData.content}
                      onChange={(content) => setEditorFormData({...editorFormData, content})}
                      placeholder="Enter your content here..."
                    />
                  </div>

                  {/* Metadata */}
                  <div className="bg-gray-50 rounded-lg p-4">
                    <h4 className="text-sm font-semibold text-gray-700 mb-3">Metadata</h4>
                    <div className="grid grid-cols-2 gap-4 text-sm">
                      <div>
                        <span className="text-gray-600">Author:</span>
                        <span className="ml-2 font-medium text-gray-900">{editingContent.author.name}</span>
                      </div>
                      <div>
                        <span className="text-gray-600">Last Updated:</span>
                        <span className="ml-2 font-medium text-gray-900">
                          {new Date(editingContent.updatedAt).toLocaleDateString()}
                        </span>
                      </div>
                      <div>
                        <span className="text-gray-600">Versions:</span>
                        <span className="ml-2 font-medium text-gray-900">{editingContent._count.versions}</span>
                      </div>
                      <div>
                        <span className="text-gray-600">Comments:</span>
                        <span className="ml-2 font-medium text-gray-900">{editingContent._count.comments}</span>
                      </div>
                    </div>
                  </div>
                </div>
              </div>

              {/* Footer */}
              <div className="bg-gray-50 px-6 py-4 flex items-center justify-between">
                <button
                  onClick={() => setShowEditor(false)}
                  disabled={isSaving}
                  className="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  Cancel
                </button>
                <div className="flex space-x-3">
                  <button
                    onClick={() => editingContent?.id === 'new' ? handleSaveNewContent('DRAFT') : handleSaveContent('DRAFT')}
                    disabled={isSaving}
                    className="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex items-center"
                  >
                    {isSaving ? (
                      <>
                        <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-gray-700" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                          <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                          <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                        </svg>
                        Saving...
                      </>
                    ) : (
                      'Save as Draft'
                    )}
                  </button>
                  <button
                    onClick={() => editingContent?.id === 'new' ? handleSaveNewContent('PUBLISHED') : handleSaveContent('PUBLISHED')}
                    disabled={isSaving}
                    className="px-6 py-2 text-sm font-medium text-white bg-gradient-to-r from-blue-600 to-purple-600 rounded-lg hover:from-blue-700 hover:to-purple-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex items-center"
                  >
                    {isSaving ? (
                      <>
                        <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                          <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                          <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                        </svg>
                        Publishing...
                      </>
                    ) : (
                      'Save & Publish'
                    )}
                  </button>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  )
}

export default function AdminDashboard() {
  return (
    <Suspense fallback={
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Loading admin dashboard...</p>
        </div>
      </div>
    }>
      <AdminDashboardContent />
    </Suspense>
  )
}