"use client";
import { useState } from "react";
import { useRouter } from "next/navigation";

const initialState = {
  name: "",
  slug: "",
  category: "HOSTING",
  description: "",
  shortDescription: "",
  price: 0,
  pricingModel: "MONTHLY",
  features: "",
  popular: false,
  active: true,
  image: "",
  configurationOptions: ""
};

export default function NewServicePage() {
  const [form, setForm] = useState(initialState);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const router = useRouter();

  function handleChange(e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>) {
    const { name, value, type } = e.target;
    let fieldValue: string | boolean = value;
    if (type === "checkbox" && "checked" in e.target) {
      fieldValue = (e.target as HTMLInputElement).checked;
    }
    setForm((prev) => ({
      ...prev,
      [name]: fieldValue
    }));
  }

  async function handleSubmit(e: React.FormEvent<HTMLFormElement>) {
    e.preventDefault();
    setLoading(true);
    setError(null);
    try {
      const res = await fetch("/api/admin/services", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          ...form,
          price: form.price,
          configurationOptions: form.configurationOptions ? JSON.parse(form.configurationOptions) : null
        })
      });
      if (!res.ok) throw new Error("Failed to create service");
      router.push("/admin/services");
    } catch (err) {
      if (err instanceof Error) {
        setError(err.message);
      } else {
        setError("Failed to create service");
      }
    } finally {
      setLoading(false);
    }
  }

  return (
    <div className="max-w-2xl mx-auto p-6 bg-white rounded shadow">
      <h1 className="text-2xl font-bold mb-4">Add New Service</h1>
      <form onSubmit={handleSubmit} className="space-y-4">
        <div>
          <label className="block font-medium">Name</label>
          <input name="name" value={form.name} onChange={handleChange} required className="w-full border px-3 py-2 rounded" />
        </div>
        <div>
          <label className="block font-medium">Slug</label>
          <input name="slug" value={form.slug} onChange={handleChange} required className="w-full border px-3 py-2 rounded" />
        </div>
        <div>
          <label className="block font-medium">Category</label>
          <select name="category" value={form.category} onChange={handleChange} className="w-full border px-3 py-2 rounded">
            <option value="HOSTING">HOSTING</option>
            <option value="DOMAINS">DOMAINS</option>
            <option value="MARKETING">MARKETING</option>
            <option value="DEVELOPMENT">DEVELOPMENT</option>
          </select>
        </div>
        <div>
          <label className="block font-medium">Description</label>
          <textarea name="description" value={form.description} onChange={handleChange} className="w-full border px-3 py-2 rounded" />
        </div>
        <div>
          <label className="block font-medium">Short Description</label>
          <input name="shortDescription" value={form.shortDescription} onChange={handleChange} className="w-full border px-3 py-2 rounded" />
        </div>
        <div>
          <label className="block font-medium">Price</label>
          <input name="price" type="number" step="0.01" value={form.price} onChange={handleChange} required className="w-full border px-3 py-2 rounded" />
        </div>
        <div>
          <label className="block font-medium">Pricing Model</label>
          <select name="pricingModel" value={form.pricingModel} onChange={handleChange} className="w-full border px-3 py-2 rounded">
            <option value="ONE_TIME">ONE_TIME</option>
            <option value="MONTHLY">MONTHLY</option>
            <option value="YEARLY">YEARLY</option>
          </select>
        </div>
        <div>
          <label className="block font-medium">Features (comma separated)</label>
          <input name="features" value={form.features} onChange={handleChange} className="w-full border px-3 py-2 rounded" />
        </div>
        <div className="flex items-center">
          <input name="popular" type="checkbox" checked={form.popular} onChange={handleChange} className="mr-2" />
          <label className="font-medium">Popular</label>
        </div>
        <div className="flex items-center">
          <input name="active" type="checkbox" checked={form.active} onChange={handleChange} className="mr-2" />
          <label className="font-medium">Active</label>
        </div>
        <div>
          <label className="block font-medium">Image URL</label>
          <input name="image" value={form.image} onChange={handleChange} className="w-full border px-3 py-2 rounded" />
        </div>
        <div>
          <label className="block font-medium">Configuration Options (JSON)</label>
          <textarea name="configurationOptions" value={form.configurationOptions} onChange={handleChange} className="w-full border px-3 py-2 rounded" />
        </div>
        {error && <div className="text-red-600">{error}</div>}
        <button type="submit" disabled={loading} className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 disabled:opacity-50">
          {loading ? "Saving..." : "Create Service"}
        </button>
      </form>
    </div>
  );
}
