import { NextRequest, NextResponse } from 'next/server'
import { prisma } from '@/lib/prisma'
import crypto from 'crypto'

export async function POST(request: NextRequest) {
  try {
    const { email } = await request.json()
    if (!email) {
      return NextResponse.json({ error: 'Email is required' }, { status: 400 })
    }
    const user = await prisma.user.findUnique({ where: { email } })
    if (!user) {
      // Always return success to avoid email enumeration
      return NextResponse.json({ success: true })
    }
    // Generate a secure token
    const resetToken = crypto.randomBytes(32).toString('hex')
    const resetTokenExpiry = new Date(Date.now() + 1000 * 60 * 30) // 30 min expiry
    await prisma.user.update({
      where: { email },
      data: { resetToken, resetTokenExpiry }
    })
    // For now, log the reset link to the console
    console.log(`Password reset link: http://localhost:3000/auth/reset-password?token=${resetToken}`)
    return NextResponse.json({ success: true })
  } catch (error) {
    return NextResponse.json({ error: 'Internal server error' }, { status: 500 })
  }
}
