import { NextRequest, NextResponse } from 'next/server'
import { prisma } from '@/lib/prisma'
import { getServerSession } from 'next-auth'
import { authOptions } from '@/lib/auth'

// GET /api/cms/content - List all content
export async function GET(request: NextRequest) {
  try {
    const session = await getServerSession(authOptions)
    if (!session?.user?.email) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    const { searchParams } = new URL(request.url)
    const type = searchParams.get('type')
    const status = searchParams.get('status')
    const category = searchParams.get('category')
    const page = parseInt(searchParams.get('page') || '1')
    const limit = parseInt(searchParams.get('limit') || '10')

    const where: any = {}
    if (type) where.type = type
    if (status) where.status = status
    if (category) where.category = category

    const content = await prisma.content.findMany({
      where,
      include: {
        author: {
          select: {
            name: true,
            email: true,
          }
        },
        _count: {
          select: {
            versions: true,
            comments: true,
          }
        }
      },
      orderBy: { updatedAt: 'desc' },
      skip: (page - 1) * limit,
      take: limit,
    })

    const total = await prisma.content.count({ where })

    return NextResponse.json({
      content,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit),
      }
    })
  } catch (error) {
    console.error('Error fetching content:', error)
    return NextResponse.json(
      { error: 'Failed to fetch content' },
      { status: 500 }
    )
  }
}

// POST /api/cms/content - Create new content
export async function POST(request: NextRequest) {
  try {
    const session = await getServerSession(authOptions)
    if (!session?.user?.email) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 })
    }

    const body = await request.json()
    const {
      title,
      content,
      excerpt,
      type,
      category,
      status,
      slug,
      metaTitle,
      metaDescription,
      tags,
      publishedAt
    } = body

    // Validate required fields
    if (!title || !content || !type) {
      return NextResponse.json(
        { error: 'Title, content, and type are required' },
        { status: 400 }
      )
    }

    // Check if slug already exists
    if (slug) {
      const existingContent = await prisma.content.findUnique({
        where: { slug }
      })
      if (existingContent) {
        return NextResponse.json(
          { error: 'Slug already exists' },
          { status: 400 }
        )
      }
    }

    // Create content
    const newContent = await prisma.content.create({
      data: {
        title,
        content,
        excerpt,
        type,
        category,
        status: status || 'DRAFT',
        slug: slug || title.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/(^-|-$)/g, ''),
        metaTitle,
        metaDescription,
        tags: tags || [],
        publishedAt: status === 'PUBLISHED' ? publishedAt || new Date() : null,
        authorId: session.user.id,
      },
      include: {
        author: {
          select: {
            name: true,
            email: true,
          }
        }
      }
    })

    // Create initial version
    await prisma.contentVersion.create({
      data: {
        contentId: newContent.id,
        version: 1,
        title: newContent.title,
        contentBody: newContent.content,
        changes: 'Initial version',
        createdById: session.user.id,
      }
    })

    return NextResponse.json(newContent, { status: 201 })
  } catch (error) {
    console.error('Error creating content:', error)
    return NextResponse.json(
      { error: 'Failed to create content' },
      { status: 500 }
    )
  }
}