import { NextRequest, NextResponse } from 'next/server'
import { getServerSession } from 'next-auth'
import { authOptions } from '@/lib/auth'
import { prisma } from '@/lib/prisma'

// GET /api/cms/pages - Get all website pages and their content
export async function GET(request: NextRequest) {
  try {
    const session = await getServerSession(authOptions)
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json(
        { error: 'Unauthorized' },
        { status: 401 }
      )
    }

    // Define all website pages with their routes and sections
    const websitePages = [
      {
        id: 'home',
        name: 'Home Page',
        route: '/',
        sections: [
          { id: 'hero', name: 'Hero Section', description: 'Main banner with headline and CTA' },
          { id: 'services', name: 'Services Overview', description: 'Featured services showcase' },
          { id: 'stats', name: 'Statistics', description: 'Company statistics and achievements' },
          { id: 'why-choose-us', name: 'Why Choose Us', description: 'Benefits and differentiators' },
          { id: 'contact-cta', name: 'Contact CTA', description: 'Call-to-action for contact' }
        ]
      },
      {
        id: 'services',
        name: 'Services Page',
        route: '/services',
        sections: [
          { id: 'services-hero', name: 'Services Hero', description: 'Services page banner' },
          { id: 'services-list', name: 'Services List', description: 'All available services' },
          { id: 'services-comparison', name: 'Service Comparison', description: 'Compare service packages' }
        ]
      },
      {
        id: 'about',
        name: 'About Us',
        route: '/about',
        sections: [
          { id: 'about-hero', name: 'About Hero', description: 'About page introduction' },
          { id: 'about-mission', name: 'Mission & Vision', description: 'Company mission and vision' },
          { id: 'about-team', name: 'Our Team', description: 'Team members showcase' }
        ]
      },
      {
        id: 'portfolio',
        name: 'Portfolio',
        route: '/portfolio',
        sections: [
          { id: 'portfolio-hero', name: 'Portfolio Hero', description: 'Portfolio page banner' },
          { id: 'portfolio-showcase', name: 'Projects Showcase', description: 'Featured projects' }
        ]
      },
      {
        id: 'blog',
        name: 'Blog',
        route: '/blog',
        sections: [
          { id: 'blog-hero', name: 'Blog Hero', description: 'Blog page banner' },
          { id: 'blog-posts', name: 'Blog Posts', description: 'Latest blog articles' }
        ]
      },
      {
        id: 'contact',
        name: 'Contact',
        route: '/contact',
        sections: [
          { id: 'contact-hero', name: 'Contact Hero', description: 'Contact page banner' },
          { id: 'contact-form', name: 'Contact Form', description: 'Contact form and information' }
        ]
      }
    ]

    // Fetch content for each page from database
    const pagesWithContent = await Promise.all(
      websitePages.map(async (page) => {
        // Get content for this page
        const pageContent = await prisma.content.findMany({
          where: {
            OR: [
              { slug: page.id },
              { slug: { startsWith: `${page.id}-` } },
              { category: page.name }
            ]
          },
          include: {
            author: {
              select: {
                id: true,
                name: true,
                email: true
              }
            },
            _count: {
              select: {
                versions: true,
                comments: true
              }
            }
          },
          orderBy: { updatedAt: 'desc' }
        })

        return {
          ...page,
          content: pageContent,
          hasContent: pageContent.length > 0
        }
      })
    )

    // Also get all services/packages
    const services = await prisma.service.findMany({
      orderBy: { createdAt: 'desc' }
    })

    return NextResponse.json({
      pages: pagesWithContent,
      services,
      totalPages: websitePages.length,
      totalContent: pagesWithContent.reduce((acc, page) => acc + page.content.length, 0)
    })
  } catch (error) {
    console.error('Error fetching pages:', error)
    return NextResponse.json(
      { error: 'Failed to fetch pages' },
      { status: 500 }
    )
  }
}

// POST /api/cms/pages - Create content for a page section
export async function POST(request: NextRequest) {
  try {
    const session = await getServerSession(authOptions)
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json(
        { error: 'Unauthorized' },
        { status: 401 }
      )
    }

    const body = await request.json()
    const { pageId, sectionId, title, content, type = 'PAGE', status = 'DRAFT' } = body

    // Create slug from pageId and sectionId
    const slug = `${pageId}-${sectionId}`

    // Check if content already exists
    const existingContent = await prisma.content.findUnique({
      where: { slug }
    })

    if (existingContent) {
      return NextResponse.json(
        { error: 'Content for this section already exists. Use PUT to update.' },
        { status: 400 }
      )
    }

    // Create new content
    const newContent = await prisma.content.create({
      data: {
        title: title || `${pageId} - ${sectionId}`,
        slug,
        type,
        category: pageId,
        status,
        content: content || '',
        tags: '[]', // Empty JSON array as string
        authorId: session.user.id,
        publishedAt: status === 'PUBLISHED' ? new Date() : null
      },
      include: {
        author: {
          select: {
            id: true,
            name: true,
            email: true
          }
        }
      }
    })

    return NextResponse.json({
      message: 'Content created successfully',
      content: newContent
    }, { status: 201 })
  } catch (error) {
    console.error('Error creating page content:', error)
    return NextResponse.json(
      { error: 'Failed to create content' },
      { status: 500 }
    )
  }
}
