import { NextRequest, NextResponse } from 'next/server'

const popularExtensions = ['.com', '.net', '.org', '.info', '.biz', '.co', '.io', '.app', '.dev', '.tech']

// Mock domain availability check - replace with actual domain API service
async function checkDomainAvailability(domain: string, extension: string): Promise<boolean> {
  // Simulate API call delay
  await new Promise(resolve => setTimeout(resolve, 100))
  
  // Mock logic - some domains are "taken"
  const combinedDomain = `${domain}${extension}`.toLowerCase()
  
  // Make common words less likely to be available
  const commonWords = ['google', 'facebook', 'amazon', 'microsoft', 'apple', 'test', 'example']
  const isCommon = commonWords.some(word => combinedDomain.includes(word))
  
  if (isCommon && extension === '.com') {
    return false // Common .com domains are usually taken
  }
  
  // Random availability for demo purposes
  // In production, replace with actual domain registrar API
  return Math.random() > 0.4
}

export async function POST(request: NextRequest) {
  try {
    const { domain } = await request.json()
    
    if (!domain || typeof domain !== 'string') {
      return NextResponse.json(
        { error: 'Domain name is required' },
        { status: 400 }
      )
    }
    
    // Clean the domain name
    const cleanDomain = domain.toLowerCase().trim().replace(/[^a-z0-9-]/g, '')
    
    if (cleanDomain.length < 2) {
      return NextResponse.json(
        { error: 'Domain name must be at least 2 characters long' },
        { status: 400 }
      )
    }
    
    // Check availability for each extension
    const results = await Promise.all(
      popularExtensions.map(async (extension) => {
        const available = await checkDomainAvailability(cleanDomain, extension)
        return {
          domain: `${cleanDomain}${extension}`,
          extension,
          available,
          price: getPrice(extension, available)
        }
      })
    )
    
    return NextResponse.json({
      domain: cleanDomain,
      results,
      timestamp: new Date().toISOString()
    })
    
  } catch (error) {
    console.error('Domain check error:', error)
    return NextResponse.json(
      { error: 'Failed to check domain availability' },
      { status: 500 }
    )
  }
}

function getPrice(extension: string, available: boolean): number | null {
  if (!available) return null
  
  // Mock pricing - replace with actual pricing from domain registrar
  const pricing: Record<string, number> = {
    '.com': 12.99,
    '.net': 14.99,
    '.org': 13.99,
    '.info': 9.99,
    '.biz': 11.99,
    '.co': 24.99,
    '.io': 39.99,
    '.app': 19.99,
    '.dev': 17.99,
    '.tech': 29.99
  }
  
  return pricing[extension] || 15.99
}