import { NextRequest, NextResponse } from 'next/server'
import { getServerSession } from 'next-auth'
import { authOptions } from '@/lib/auth'
import { prisma } from '@/lib/prisma'

// GET - Fetch all homepage sections (public)
export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const section = searchParams.get('section')

    const where: any = { enabled: true }
    if (section) {
      where.section = section
    }

    const sections = await prisma.homePageSection.findMany({
      where,
      orderBy: [
        { section: 'asc' },
        { order: 'asc' }
      ]
    })

    // Group by section for easier consumption
    const grouped = sections.reduce((acc: any, item) => {
      if (!acc[item.section]) {
        acc[item.section] = {}
      }
      acc[item.section][item.key] = {
        value: item.value,
        type: item.type,
        id: item.id
      }
      return acc
    }, {})

    return NextResponse.json({ 
      success: true, 
      data: section ? grouped[section] : grouped 
    })

  } catch (error: any) {
    console.error('Error fetching homepage content:', error)
    return NextResponse.json(
      { success: false, error: 'Failed to fetch homepage content' },
      { status: 500 }
    )
  }
}

// POST - Create new homepage content (admin only)
export async function POST(request: NextRequest) {
  try {
    const session = await getServerSession(authOptions)
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json(
        { success: false, error: 'Unauthorized' },
        { status: 401 }
      )
    }

    const body = await request.json()
    const { section, key, value, type = 'text', order = 0, enabled = true } = body

    if (!section || !key || value === undefined) {
      return NextResponse.json(
        { success: false, error: 'Missing required fields: section, key, value' },
        { status: 400 }
      )
    }

    const newSection = await prisma.homePageSection.create({
      data: {
        section,
        key,
        value,
        type,
        order,
        enabled
      }
    })

    return NextResponse.json({ 
      success: true, 
      data: newSection 
    })

  } catch (error: any) {
    console.error('Error creating homepage content:', error)
    
    if (error.code === 'P2002') {
      return NextResponse.json(
        { success: false, error: 'Content with this section and key already exists' },
        { status: 409 }
      )
    }
    
    return NextResponse.json(
      { success: false, error: 'Failed to create homepage content' },
      { status: 500 }
    )
  }
}

// PUT - Update homepage content (admin only)
export async function PUT(request: NextRequest) {
  try {
    const session = await getServerSession(authOptions)
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json(
        { success: false, error: 'Unauthorized' },
        { status: 401 }
      )
    }

    const body = await request.json()
    const { id, section, key, value, type, order, enabled } = body

    if (!id) {
      return NextResponse.json(
        { success: false, error: 'Missing required field: id' },
        { status: 400 }
      )
    }

    const updateData: any = { updatedAt: new Date() }
    if (section !== undefined) updateData.section = section
    if (key !== undefined) updateData.key = key
    if (value !== undefined) updateData.value = value
    if (type !== undefined) updateData.type = type
    if (order !== undefined) updateData.order = order
    if (enabled !== undefined) updateData.enabled = enabled

    const updated = await prisma.homePageSection.update({
      where: { id },
      data: updateData
    })

    return NextResponse.json({ 
      success: true, 
      data: updated 
    })

  } catch (error: any) {
    console.error('Error updating homepage content:', error)
    
    if (error.code === 'P2025') {
      return NextResponse.json(
        { success: false, error: 'Content not found' },
        { status: 404 }
      )
    }
    
    return NextResponse.json(
      { success: false, error: 'Failed to update homepage content' },
      { status: 500 }
    )
  }
}

// DELETE - Delete homepage content (admin only)
export async function DELETE(request: NextRequest) {
  try {
    const session = await getServerSession(authOptions)
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json(
        { success: false, error: 'Unauthorized' },
        { status: 401 }
      )
    }

    const { searchParams } = new URL(request.url)
    const id = searchParams.get('id')

    if (!id) {
      return NextResponse.json(
        { success: false, error: 'Missing required parameter: id' },
        { status: 400 }
      )
    }

    await prisma.homePageSection.delete({
      where: { id }
    })

    return NextResponse.json({ 
      success: true, 
      message: 'Content deleted successfully' 
    })

  } catch (error: any) {
    console.error('Error deleting homepage content:', error)
    
    if (error.code === 'P2025') {
      return NextResponse.json(
        { success: false, error: 'Content not found' },
        { status: 404 }
      )
    }
    
    return NextResponse.json(
      { success: false, error: 'Failed to delete homepage content' },
      { status: 500 }
    )
  }
}
