import { NextRequest, NextResponse } from 'next/server'
import { getServerSession } from 'next-auth'
import { authOptions } from '@/lib/auth'
import { prisma } from '@/lib/prisma'

export async function GET(request: NextRequest) {
  try {
    const session = await getServerSession(authOptions)
    
    if (!session?.user) {
      return NextResponse.json(
        { success: false, error: 'Unauthorized' },
        { status: 401 }
      )
    }

    const { searchParams } = new URL(request.url)
    const status = searchParams.get('status')
    const limit = parseInt(searchParams.get('limit') || '10')
    const offset = parseInt(searchParams.get('offset') || '0')

    const where: any = {
      userId: session.user.id,
    }

    if (status && status !== 'all') {
      where.status = status.toUpperCase()
    }

    const invoices = await prisma.invoice.findMany({
      where,
      include: {
        order: {
          include: {
            items: {
              include: {
                service: true,
              },
            },
          },
        },
        payments: true,
      },
      orderBy: {
        createdAt: 'desc',
      },
      take: limit,
      skip: offset,
    })

    const total = await prisma.invoice.count({ where })

    return NextResponse.json({
      success: true,
      invoices,
      pagination: {
        total,
        limit,
        offset,
        hasMore: total > offset + limit,
      },
    })
  } catch (error) {
    console.error('Invoice fetch error:', error)
    return NextResponse.json(
      { 
        success: false, 
        error: error instanceof Error ? error.message : 'Internal server error' 
      },
      { status: 500 }
    )
  }
}

export async function POST(request: NextRequest) {
  try {
    const session = await getServerSession(authOptions)
    
    if (!session?.user) {
      return NextResponse.json(
        { success: false, error: 'Unauthorized' },
        { status: 401 }
      )
    }

    const body = await request.json()
    const {
      orderId,
      description,
      amount,
      subtotal,
      taxRate = 0,
      discountAmount = 0,
      dueDate,
      billingName,
      billingEmail,
      billingAddress,
      notes,
      termsAndConditions,
    } = body

    if (!orderId || !description || !amount || !billingName || !billingEmail) {
      return NextResponse.json(
        { success: false, error: 'Missing required fields' },
        { status: 400 }
      )
    }

    // Verify the order exists and belongs to the user
    const order = await prisma.order.findFirst({
      where: {
        id: orderId,
        userId: session.user.id,
      },
    })

    if (!order) {
      return NextResponse.json(
        { success: false, error: 'Order not found' },
        { status: 404 }
      )
    }

    // Generate unique invoice number
    const invoiceCount = await prisma.invoice.count()
    const invoiceNumber = `INV-${String(invoiceCount + 1).padStart(6, '0')}`

    // Calculate tax amount
    const taxAmount = subtotal * (taxRate / 100)

    const invoice = await prisma.invoice.create({
      data: {
        invoiceNumber,
        description,
        amount,
        subtotal,
        taxAmount,
        taxRate,
        discountAmount,
        status: 'PENDING',
        dueDate: new Date(dueDate),
        billingName,
        billingEmail,
        billingAddress,
        notes,
        termsAndConditions,
        orderId,
        clientId: order.clientId,
        userId: session.user.id,
      },
      include: {
        order: {
          include: {
            items: {
              include: {
                service: true,
              },
            },
          },
        },
      },
    })

    return NextResponse.json({
      success: true,
      invoice,
    })
  } catch (error) {
    console.error('Invoice creation error:', error)
    return NextResponse.json(
      { 
        success: false, 
        error: error instanceof Error ? error.message : 'Internal server error' 
      },
      { status: 500 }
    )
  }
}