import { NextRequest, NextResponse } from 'next/server'
import { getServerSession } from 'next-auth'
import { authOptions } from '@/lib/auth'
import { Logger, RequestMonitor, ErrorTracker, HealthMonitor } from '@/lib/monitoring'
import { CacheManager, PerformanceMonitor, QueryOptimizer } from '@/lib/performance'

export async function GET(request: NextRequest) {
  try {
    const session = await getServerSession(authOptions)
    
    // Only allow admin users to access monitoring data
    if (!session?.user?.email || session.user.email !== 'sales@nuwair.com') {
      return NextResponse.json(
        { error: 'Unauthorized access to monitoring endpoint' },
        { status: 403 }
      )
    }

    const url = new URL(request.url)
    const action = url.searchParams.get('action')
    const timeRange = {
      start: url.searchParams.get('start') || new Date(Date.now() - 24 * 60 * 60 * 1000).toISOString(),
      end: url.searchParams.get('end') || new Date().toISOString(),
    }

    switch (action) {
      case 'health':
        const health = await HealthMonitor.checkHealth()
        return NextResponse.json(health)

      case 'logs':
        const level = url.searchParams.get('level') || undefined
        const limit = parseInt(url.searchParams.get('limit') || '100')
        const logs = Logger.getLogs({
          level: level as any,
          since: timeRange.start,
          limit,
        })
        return NextResponse.json({
          logs,
          stats: Logger.getStats(),
        })

      case 'requests':
        const requestMetrics = RequestMonitor.getMetrics(timeRange)
        const topEndpoints = RequestMonitor.getTopEndpoints(10)
        return NextResponse.json({
          metrics: requestMetrics,
          topEndpoints,
        })

      case 'errors':
        const userId = url.searchParams.get('userId') || undefined
        const errorLimit = parseInt(url.searchParams.get('limit') || '50')
        const errors = ErrorTracker.getErrors({
          since: timeRange.start,
          limit: errorLimit,
          userId,
        })
        const errorStats = ErrorTracker.getErrorStats()
        return NextResponse.json({
          errors,
          stats: errorStats,
        })

      case 'performance':
        const performanceMetrics = PerformanceMonitor.getAllMetrics()
        const slowQueries = QueryOptimizer.getSlowQueries()
        const cacheStats = CacheManager.getStats()
        return NextResponse.json({
          metrics: performanceMetrics,
          slowQueries,
          cache: cacheStats,
        })

      case 'clear':
        const clearType = url.searchParams.get('type')
        switch (clearType) {
          case 'logs':
            Logger.clearLogs()
            break
          case 'requests':
            RequestMonitor.clearMetrics()
            break
          case 'errors':
            ErrorTracker.clearErrors()
            break
          case 'performance':
            PerformanceMonitor.clearMetrics()
            QueryOptimizer.clearSlowQueries()
            break
          case 'cache':
            CacheManager.clear()
            break
          case 'all':
            Logger.clearLogs()
            RequestMonitor.clearMetrics()
            ErrorTracker.clearErrors()
            PerformanceMonitor.clearMetrics()
            QueryOptimizer.clearSlowQueries()
            CacheManager.clear()
            break
          default:
            return NextResponse.json(
              { error: 'Invalid clear type' },
              { status: 400 }
            )
        }
        return NextResponse.json({ success: true, cleared: clearType })

      case 'dashboard':
      default:
        const dashboardData = {
          health: await HealthMonitor.checkHealth(),
          requests: RequestMonitor.getMetrics(timeRange),
          errors: ErrorTracker.getErrorStats(),
          performance: PerformanceMonitor.getAllMetrics(),
          cache: CacheManager.getStats(),
          logs: Logger.getStats(),
          uptime: process.uptime(),
          memory: process.memoryUsage(),
          env: process.env.NODE_ENV,
          version: process.env.npm_package_version || '1.0.0',
        }
        return NextResponse.json(dashboardData)
    }
  } catch (error) {
    Logger.error('Monitoring endpoint error', error as Error)
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    )
  }
}

// Health check endpoint (public for load balancers)
export async function HEAD(request: NextRequest) {
  try {
    const health = await HealthMonitor.checkHealth()
    return new NextResponse(null, {
      status: health.status === 'healthy' ? 200 : 503,
      headers: {
        'Cache-Control': 'no-cache',
        'X-Health-Status': health.status,
      },
    })
  } catch (error) {
    return new NextResponse(null, { status: 503 })
  }
}