import { NextRequest, NextResponse } from 'next/server'
import { getServerSession } from 'next-auth'
import { authOptions } from '@/lib/auth'
import { prisma } from '@/lib/prisma'
import { z } from 'zod'

const createOrderSchema = z.object({
  items: z.array(z.object({
    serviceId: z.string(),
    name: z.string().optional(), // Service name from cart
    quantity: z.number().positive(),
    price: z.number().positive(),
    configuration: z.record(z.string(), z.any()).optional(),
    addons: z.record(z.string(), z.any()).optional(),
  })),
  billingInfo: z.object({
    email: z.string().email(),
    firstName: z.string().min(1),
    lastName: z.string().min(1),
    company: z.string().optional(),
    phone: z.string().min(1),
    address: z.string().min(1),
    city: z.string().min(1),
    state: z.string().min(1),
    zipCode: z.string().min(1),
    country: z.string().min(1),
  }),
  paymentMethod: z.enum(['CREDIT_CARD', 'BANK_TRANSFER', 'PAYPAL', 'CASH', 'JAZZ_CASH', 'CHEQUE']),
  paymentData: z.record(z.string(), z.any()),
  totalAmount: z.number().positive(),
  tax: z.number().min(0),
})

export async function POST(request: NextRequest) {
  try {
    console.log('=== Order Creation Started ===')
    const session = await getServerSession(authOptions)
    console.log('Session:', session?.user?.id ? 'User authenticated' : 'No session')
    
    const body = await request.json()
    console.log('Request body received:', JSON.stringify(body, null, 2))
    
    // Validate the request body
    const validatedData = createOrderSchema.parse(body)
    console.log('Data validation passed')

    // Create or find user for guest checkout
    let userId = session?.user?.id
    if (!userId) {
      console.log('Guest checkout - finding/creating user for email:', validatedData.billingInfo.email)
      // For guest checkout, create a user account or link to existing
      const existingUser = await prisma.user.findUnique({
        where: { email: validatedData.billingInfo.email }
      })

      if (existingUser) {
        userId = existingUser.id
        console.log('Found existing user:', userId)
      } else {
        console.log('Creating new user...')
        const newUser = await prisma.user.create({
          data: {
            email: validatedData.billingInfo.email,
            name: `${validatedData.billingInfo.firstName} ${validatedData.billingInfo.lastName}`,
            role: 'CLIENT'
          }
        })
        userId = newUser.id
        console.log('Created new user:', userId)
      }
    }

    // Create client profile if it doesn't exist
    console.log('Finding/creating client profile...')
    let client = await prisma.client.findFirst({
      where: { userId: userId }
    })

    if (!client) {
      console.log('Creating client profile...')
      client = await prisma.client.create({
        data: {
          userId: userId,
          companyName: validatedData.billingInfo.company || null,
          phone: validatedData.billingInfo.phone || null,
          address: validatedData.billingInfo.address || null,
          city: validatedData.billingInfo.city || null,
          country: validatedData.billingInfo.country || null,
          vatNumber: null
        }
      })
      console.log('Created client:', client.id)
    } else {
      console.log('Found existing client:', client.id)
      // Update client info if needed
      if (validatedData.billingInfo.company || validatedData.billingInfo.phone) {
        await prisma.client.update({
          where: { id: client.id },
          data: {
            companyName: validatedData.billingInfo.company || client.companyName,
            phone: validatedData.billingInfo.phone || client.phone,
            address: validatedData.billingInfo.address || client.address,
            city: validatedData.billingInfo.city || client.city,
            country: validatedData.billingInfo.country || client.country,
          }
        })
      }
    }

    // Generate order ID
    const orderNumber = `ORD-${Date.now()}`
    console.log('Generated order number:', orderNumber)

    // Create the order
    console.log('Creating order with items:', validatedData.items.length)
    const order = await prisma.order.create({
      data: {
        orderNumber,
        totalAmount: validatedData.totalAmount,
        tax: validatedData.tax,
        status: 'PENDING',
        billingAddress: JSON.stringify(validatedData.billingInfo),
        clientId: client.id,
        userId: userId,
        items: {
          create: validatedData.items.map(item => ({
            quantity: item.quantity,
            price: item.price,
            serviceName: item.name || 'Custom Service',
            serviceDescription: null,
            serviceId: null, // Don't use cart IDs as they're temporary
            configuration: item.configuration || {},
            addons: item.addons || {},
          }))
        }
      },
      include: {
        items: true
      }
    })
    console.log('Order created successfully:', order.id)

    // Create payment record
    const payment = await prisma.payment.create({
      data: {
        amount: validatedData.totalAmount,
        currency: 'USD',
        status: getInitialPaymentStatus(validatedData.paymentMethod),
        paymentMethod: validatedData.paymentMethod,
        orderId: order.id,
        clientId: client.id,
        userId: userId,
        ...getPaymentDetails(validatedData.paymentMethod, validatedData.paymentData)
      }
    })

    // Handle different payment methods
    await processPaymentMethod(validatedData.paymentMethod, payment.id, validatedData.paymentData)

    return NextResponse.json({
      success: true,
      orderId: order.orderNumber,
      order: {
        id: order.id,
        orderNumber: order.orderNumber,
        status: order.status,
        totalAmount: order.totalAmount,
        paymentMethod: validatedData.paymentMethod,
        paymentStatus: payment.status,
        items: order.items
      }
    })

  } catch (error) {
    console.error('Order creation error:', error)
    
    if (error instanceof z.ZodError) {
      return NextResponse.json({
        success: false,
        error: 'Invalid data provided',
        details: error.issues
      }, { status: 400 })
    }

    return NextResponse.json({
      success: false,
      error: 'Failed to create order'
    }, { status: 500 })
  }
}

export async function GET(request: NextRequest) {
  try {
    const session = await getServerSession(authOptions)
    
    if (!session?.user?.id) {
      return NextResponse.json({
        success: false,
        error: 'Authentication required'
      }, { status: 401 })
    }

    const url = new URL(request.url)
    const page = parseInt(url.searchParams.get('page') || '1')
    const limit = parseInt(url.searchParams.get('limit') || '10')
    const skip = (page - 1) * limit

    const orders = await prisma.order.findMany({
      where: { userId: session.user.id },
      include: {
        items: {
          include: {
            service: true
          }
        },
        payments: true
      },
      orderBy: { createdAt: 'desc' },
      take: limit,
      skip: skip
    })

    const totalOrders = await prisma.order.count({
      where: { userId: session.user.id }
    })

    return NextResponse.json({
      success: true,
      orders,
      pagination: {
        page,
        limit,
        total: totalOrders,
        pages: Math.ceil(totalOrders / limit)
      }
    })

  } catch (error) {
    console.error('Orders fetch error:', error)
    return NextResponse.json({
      success: false,
      error: 'Failed to fetch orders'
    }, { status: 500 })
  }
}

function getInitialPaymentStatus(paymentMethod: string) {
  switch (paymentMethod) {
    case 'CREDIT_CARD':
    case 'PAYPAL':
      return 'PROCESSING'
    case 'BANK_TRANSFER':
    case 'JAZZ_CASH':
    case 'CHEQUE':
      return 'PENDING'
    case 'CASH':
      return 'PENDING'
    default:
      return 'PENDING'
  }
}

function getPaymentDetails(paymentMethod: string, paymentData: any) {
  const details: any = {}

  switch (paymentMethod) {
    case 'BANK_TRANSFER':
      details.bankTransferDetails = {
        customerBank: paymentData.bankName,
        customerAccount: paymentData.accountNumber,
        customerRouting: paymentData.routingNumber
      }
      break
    case 'PAYPAL':
      details.paypalTransactionId = null // Will be updated after PayPal processing
      break
    case 'JAZZ_CASH':
      details.jazzCashDetails = {
        customerNumber: paymentData.jazzCashNumber
      }
      break
    case 'CHEQUE':
      details.chequeDetails = {
        chequeNumber: paymentData.chequeNumber,
        bankBranch: paymentData.bankBranch
      }
      break
    case 'CASH':
      details.notes = 'Cash payment - awaiting collection at office'
      break
  }

  return details
}

async function processPaymentMethod(paymentMethod: string, paymentId: string, paymentData: any) {
  // Here you would integrate with actual payment processors
  switch (paymentMethod) {
    case 'CREDIT_CARD':
      // Integrate with Stripe, Square, etc.
      // For now, simulate processing
      await simulatePaymentProcessing(paymentId, 'COMPLETED')
      break
    case 'PAYPAL':
      // Integrate with PayPal API
      await simulatePaymentProcessing(paymentId, 'COMPLETED')
      break
    case 'BANK_TRANSFER':
      // Notify admin about pending bank transfer
      await notifyAdminNewPayment(paymentId, 'Bank Transfer')
      break
    case 'JAZZ_CASH':
      // Notify admin about Jazz Cash payment
      await notifyAdminNewPayment(paymentId, 'Jazz Cash')
      break
    case 'CHEQUE':
      // Notify admin about cheque payment
      await notifyAdminNewPayment(paymentId, 'Cheque')
      break
    case 'CASH':
      // Notify admin about cash payment
      await notifyAdminNewPayment(paymentId, 'Cash')
      break
  }
}

async function simulatePaymentProcessing(paymentId: string, status: string) {
  // Simulate payment processing delay
  setTimeout(async () => {
    await prisma.payment.update({
      where: { id: paymentId },
      data: { status: status as any }
    })
  }, 3000)
}

async function notifyAdminNewPayment(paymentId: string, method: string) {
  // In a real application, you would send notifications here
  console.log(`New ${method} payment received: ${paymentId}`)
  // Could send email, webhook, or push notification to admin
}