import { NextRequest, NextResponse } from 'next/server'
import { getServerSession } from 'next-auth'
import { authOptions } from '@/lib/auth'
import { createPaymentIntent } from '@/lib/stripe'
import { prisma } from '@/lib/prisma'

export async function POST(request: NextRequest) {
  try {
    const session = await getServerSession(authOptions)
    
    if (!session?.user) {
      return NextResponse.json(
        { success: false, error: 'Unauthorized' },
        { status: 401 }
      )
    }

    const body = await request.json()
    const { orderId, amount, currency = 'usd' } = body

    if (!orderId || !amount) {
      return NextResponse.json(
        { success: false, error: 'Missing required fields' },
        { status: 400 }
      )
    }

    console.log('Received orderId:', orderId);
    console.log('Session userId:', session.user.id);

    // Verify the order exists and belongs to the user
    const order = await prisma.order.findFirst({
      where: {
        id: orderId,
        userId: session.user.id,
      },
      include: {
        client: true,
        user: true,
        items: {
          include: {
            service: true,
          },
        },
      },
    })

    if (!order) {
      return NextResponse.json(
        { success: false, error: 'Order not found' },
        { status: 404 }
      )
    }

    // Calculate order total from items
    const orderTotal = order.items.reduce((sum, item) => sum + (item.price * item.quantity), 0)

    // Create payment intent with Stripe
    const paymentResult = await createPaymentIntent(
      amount,
      currency,
      {
        orderId: order.id,
        userId: session.user.id,
        clientId: order.clientId,
        customerEmail: order.user.email || 'Unknown',
      }
    )

    if (!paymentResult.success) {
      return NextResponse.json(
        { success: false, error: paymentResult.error },
        { status: 500 }
      )
    }

    // Create payment record in database
    const payment = await prisma.payment.create({
      data: {
        amount,
        currency,
        status: 'PENDING',
        paymentMethod: 'CREDIT_CARD',
        stripePaymentIntentId: paymentResult.paymentIntentId,
        orderId: order.id,
        clientId: order.clientId,
        userId: session.user.id,
        notes: `Payment for order ${order.id}`,
      },
    })

    return NextResponse.json({
      success: true,
      clientSecret: paymentResult.clientSecret,
      paymentId: payment.id,
      order: {
        id: order.id,
        total: orderTotal,
        status: order.status,
      },
    })
  } catch (error) {
    console.error('Payment intent creation error:', error)
    return NextResponse.json(
      { 
        success: false, 
        error: error instanceof Error ? error.message : 'Internal server error' 
      },
      { status: 500 }
    )
  }
}