import { NextRequest, NextResponse } from 'next/server'
import Stripe from 'stripe'
import { stripe } from '@/lib/stripe'
import { prisma } from '@/lib/prisma'

const webhookSecret = process.env.STRIPE_WEBHOOK_SECRET!

export async function POST(request: NextRequest) {
  try {
    if (!stripe) {
      console.error('Stripe not initialized')
      return NextResponse.json(
        { error: 'Payment system not available' },
        { status: 500 }
      )
    }

    const body = await request.text()
    const signature = request.headers.get('stripe-signature')!

    let event: Stripe.Event

    try {
      event = stripe.webhooks.constructEvent(body, signature, webhookSecret)
    } catch (err) {
      console.error('Webhook signature verification failed:', err)
      return NextResponse.json(
        { error: 'Invalid signature' },
        { status: 400 }
      )
    }

    // Handle the event
    switch (event.type) {
      case 'payment_intent.succeeded':
        await handlePaymentSucceeded(event.data.object as Stripe.PaymentIntent)
        break

      case 'payment_intent.payment_failed':
        await handlePaymentFailed(event.data.object as Stripe.PaymentIntent)
        break

      case 'invoice.payment_succeeded':
        await handleInvoicePaymentSucceeded(event.data.object as Stripe.Invoice)
        break

      case 'invoice.payment_failed':
        await handleInvoicePaymentFailed(event.data.object as Stripe.Invoice)
        break

      default:
        console.log(`Unhandled event type: ${event.type}`)
    }

    return NextResponse.json({ received: true })
  } catch (error) {
    console.error('Webhook handler error:', error)
    return NextResponse.json(
      { error: 'Webhook handler failed' },
      { status: 500 }
    )
  }
}

async function handlePaymentSucceeded(paymentIntent: Stripe.PaymentIntent) {
  try {
    // Update payment status in database
    const payment = await prisma.payment.findFirst({
      where: {
        stripePaymentIntentId: paymentIntent.id,
      },
      include: {
        order: true,
      },
    })

    if (payment) {
      await prisma.payment.update({
        where: { id: payment.id },
        data: {
          status: 'COMPLETED',
          notes: `Payment completed via Stripe. Amount: ${paymentIntent.amount / 100} ${paymentIntent.currency.toUpperCase()}`,
        },
      })

      // Update order status if all payments are completed
      const orderPayments = await prisma.payment.findMany({
        where: { orderId: payment.orderId },
      })

      const allPaymentsCompleted = orderPayments.every(
        p => p.status === 'COMPLETED' || p.status === 'REFUNDED'
      )

      if (allPaymentsCompleted) {
        await prisma.order.update({
          where: { id: payment.orderId },
          data: { status: 'COMPLETED' },
        })
      }

      console.log(`Payment ${payment.id} completed successfully`)
    }
  } catch (error) {
    console.error('Error handling payment succeeded:', error)
  }
}

async function handlePaymentFailed(paymentIntent: Stripe.PaymentIntent) {
  try {
    const payment = await prisma.payment.findFirst({
      where: {
        stripePaymentIntentId: paymentIntent.id,
      },
    })

    if (payment) {
      await prisma.payment.update({
        where: { id: payment.id },
        data: {
          status: 'FAILED',
          notes: `Payment failed via Stripe. Last error: ${paymentIntent.last_payment_error?.message || 'Unknown error'}`,
        },
      })

      console.log(`Payment ${payment.id} failed`)
    }
  } catch (error) {
    console.error('Error handling payment failed:', error)
  }
}

async function handleInvoicePaymentSucceeded(invoice: Stripe.Invoice) {
  try {
    // Handle invoice payment success
    console.log(`Invoice ${invoice.id} payment succeeded`)
  } catch (error) {
    console.error('Error handling invoice payment succeeded:', error)
  }
}

async function handleInvoicePaymentFailed(invoice: Stripe.Invoice) {
  try {
    // Handle invoice payment failure
    console.log(`Invoice ${invoice.id} payment failed`)
  } catch (error) {
    console.error('Error handling invoice payment failed:', error)
  }
}