'use client'

import { useState, useEffect } from 'react'
import { useHydrationSafeCart } from '@/hooks/useHydrationSafeCart'
import { formatPrice } from '@/lib/utils'
import { motion } from 'framer-motion'
import { useSession, signIn } from 'next-auth/react'
import { useRouter } from 'next/navigation'
import Link from 'next/link'
import StripeProvider from '@/components/StripeProvider'
import ClientPaymentFormWrapper from '@/components/ClientPaymentFormWrapper'
import ClientPaymentSection from "@/components/ClientPaymentSection";
import { 
  CreditCardIcon, 
  ShieldCheckIcon, 
  TruckIcon,
  CheckCircleIcon,
  ArrowLeftIcon,
  UserIcon,
  BanknotesIcon,
  BuildingLibraryIcon,
  DevicePhoneMobileIcon,
  DocumentCheckIcon,
  EyeIcon,
  EyeSlashIcon
} from '@heroicons/react/24/outline'

const paymentMethods = [
  {
    id: 'CREDIT_CARD',
    name: 'Credit Card',
    icon: CreditCardIcon,
    description: 'Pay securely with Visa, MasterCard, or American Express'
  },
  {
    id: 'BANK_TRANSFER',
    name: 'Bank Transfer',
    icon: BuildingLibraryIcon,
    description: 'Direct bank transfer to our account'
  },
  {
    id: 'PAYPAL',
    name: 'PayPal',
    icon: CreditCardIcon,
    description: 'Pay with your PayPal account'
  },
  {
    id: 'CASH',
    name: 'Cash Payment',
    icon: BanknotesIcon,
    description: 'Pay in cash at our office'
  },
  {
    id: 'JAZZ_CASH',
    name: 'Jazz Cash',
    icon: DevicePhoneMobileIcon,
    description: 'Mobile wallet payment via Jazz Cash'
  },
  {
    id: 'CHEQUE',
    name: 'Bank Cheque',
    icon: DocumentCheckIcon,
    description: 'Pay by bank cheque'
  }
]

export default function Checkout() {
  const { items, getTotalPrice, clearCart } = useHydrationSafeCart()
  const { data: session } = useSession()
  const router = useRouter()
  const [isProcessing, setIsProcessing] = useState(false)
  const [orderComplete, setOrderComplete] = useState(false)
  const [orderId, setOrderId] = useState('')
  const [mounted, setMounted] = useState(false)
  const [currentStep, setCurrentStep] = useState(1) // Always start with step 1 to avoid hydration issues
  const [selectedPaymentMethod, setSelectedPaymentMethod] = useState('CREDIT_CARD')
  const [orderCreated, setOrderCreated] = useState(false)
  const [currentOrderId, setCurrentOrderId] = useState('')
  const [paymentSuccess, setPaymentSuccess] = useState(false)
  const [showCreateAccount, setShowCreateAccount] = useState(false)
  const [showPassword, setShowPassword] = useState(false)

  // Authentication form
  const [authData, setAuthData] = useState({
    email: '',
    password: '',
    firstName: '',
    lastName: '',
    confirmPassword: '',
    agreeToTerms: false
  })

  // Main form data - Initialize with empty values to avoid hydration issues
  const [formData, setFormData] = useState({
    // Billing Information
    email: '',
    firstName: '',
    lastName: '',
    company: '',
    phone: '',
    
    // Billing Address
    address: '',
    city: '',
    state: '',
    zipCode: '',
    country: 'United States',
    
    // Credit Card Information
    cardNumber: '',
    expiryDate: '',
    cvv: '',
    cardName: '',
    
    // Bank Transfer Information
    bankName: '',
    accountNumber: '',
    routingNumber: '',
    
    // PayPal Information
    paypalEmail: '',
    
    // Jazz Cash Information
    jazzCashNumber: '',
    
    // Cheque Information
    chequeNumber: '',
    bankBranch: '',
    
    // Additional Options
    saveInfo: false,
    newsletter: false,
  })

  const totalPrice = getTotalPrice()
  const tax = totalPrice * 0.08 // 8% tax
  const finalTotal = totalPrice + tax

  useEffect(() => {
    setMounted(true)
  }, [])

  useEffect(() => {
    // Only redirect if mounted AND cart is truly empty AND not completing an order
    // Add a small delay to ensure cart state is fully hydrated
    if (mounted && items.length === 0 && !orderComplete) {
      const timer = setTimeout(() => {
        console.log('Cart is empty, redirecting to services')
        router.push('/services')
      }, 1000) // Increased delay to ensure proper hydration
      return () => clearTimeout(timer)
    }
  }, [items, orderComplete, router, mounted])

  useEffect(() => {
    if (mounted && session?.user && currentStep === 1) {
      setCurrentStep(2)
      setFormData(prev => ({
        ...prev,
        email: session.user?.email || '',
        firstName: session.user?.name?.split(' ')[0] || '',
        lastName: session.user?.name?.split(' ').slice(1).join(' ') || '',
      }))
    }
  }, [mounted, session, currentStep])

  if (!mounted) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-600"></div>
      </div>
    )
  }

  // Show loading while cart is hydrating, but only for a short time
  if (mounted && items.length === 0 && !orderComplete) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Loading your cart...</p>
        </div>
      </div>
    )
  }

  const handleAuthInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value, type, checked } = e.target
    setAuthData(prev => ({
      ...prev,
      [name]: type === 'checkbox' ? checked : value
    }))
  }

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value, type } = e.target
    const checked = (e.target as HTMLInputElement).checked
    
    setFormData(prev => ({
      ...prev,
      [name]: type === 'checkbox' ? checked : value
    }))
  }

  const handleCreateAccount = async (e: React.FormEvent) => {
    e.preventDefault()
    if (authData.password !== authData.confirmPassword) {
      alert('Passwords do not match!')
      return
    }

    try {
      const response = await fetch('/api/auth/register', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          email: authData.email,
          password: authData.password,
          name: `${authData.firstName} ${authData.lastName}`,
        })
      })

      if (response.ok) {
        // Auto sign in after registration
        const result = await signIn('credentials', {
          email: authData.email,
          password: authData.password,
          redirect: false
        })
        
        if (result?.ok) {
          setCurrentStep(2)
        }
      } else {
        const error = await response.json()
        alert(error.message || 'Registration failed')
      }
    } catch (error) {
      alert('Registration failed. Please try again.')
    }
  }

  const handleLogin = async (e: React.FormEvent) => {
    e.preventDefault()
    
    const result = await signIn('credentials', {
      email: authData.email,
      password: authData.password,
      redirect: false
    })
    
    if (result?.ok) {
      setCurrentStep(2)
    } else {
      alert('Login failed. Please check your credentials.')
    }
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsProcessing(true)

    try {
      // For credit card payments, create order first and then show Stripe form
      if (selectedPaymentMethod === 'CREDIT_CARD') {
        const orderData = {
          items: items.map(item => ({
            serviceId: item.id,
            name: item.name, // Include service name
            quantity: item.quantity,
            price: item.price,
            configuration: {},
            addons: {}
          })),
          billingInfo: {
            email: formData.email,
            firstName: formData.firstName,
            lastName: formData.lastName,
            company: formData.company,
            phone: formData.phone,
            address: formData.address,
            city: formData.city,
            state: formData.state,
            zipCode: formData.zipCode,
            country: formData.country
          },
          paymentMethod: selectedPaymentMethod,
          paymentData: {}, // Add required paymentData field
          totalAmount: finalTotal,
          tax: tax
        }

        const response = await fetch('/api/orders', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(orderData)
        })

        if (response.ok) {
          const result = await response.json()
          setCurrentOrderId(result.order.id)
          setOrderCreated(true)
          // Don't process payment here - let the Stripe form handle it
          setIsProcessing(false)
          return
        } else {
          throw new Error('Failed to create order')
        }
      }

      // For other payment methods, create order with payment data
      const orderData = {
        items: items.map(item => ({
          serviceId: item.id,
          name: item.name, // Include service name
          quantity: item.quantity,
          price: item.price,
          configuration: {},
          addons: {}
        })),
        billingInfo: {
          email: formData.email,
          firstName: formData.firstName,
          lastName: formData.lastName,
          company: formData.company,
          phone: formData.phone,
          address: formData.address,
          city: formData.city,
          state: formData.state,
          zipCode: formData.zipCode,
          country: formData.country
        },
        paymentMethod: selectedPaymentMethod,
        paymentData: getPaymentData(),
        totalAmount: finalTotal,
        tax: tax
      }

      const response = await fetch('/api/orders', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(orderData)
      })

      const result = await response.json()

      if (response.ok && result.success) {
        setOrderId(result.orderId)
        setOrderComplete(true)
        clearCart()
        
        setTimeout(() => {
          router.push(`/dashboard/orders/${result.orderId}`)
        }, 3000)
      } else {
        const errorMsg = result.error || 'Order creation failed'
        const errorDetails = result.details ? JSON.stringify(result.details) : ''
        console.error('Order creation failed:', errorMsg, errorDetails)
        throw new Error(errorMsg)
      }
    } catch (error) {
      console.error('Payment failed:', error)
      const errorMessage = error instanceof Error ? error.message : 'Payment processing failed'
      alert(`${errorMessage}. Please try again or contact support.`)
    } finally {
      setIsProcessing(false)
    }
  }

  const handlePaymentSuccess = (paymentResult: any) => {
    setPaymentSuccess(true)
    clearCart()
    // Redirect to success page after a short delay
    setTimeout(() => {
      router.push(`/dashboard/orders/${currentOrderId}`)
    }, 2000)
  }

  const handlePaymentError = (error: string) => {
    console.error('Payment error:', error)
    alert(`Payment failed: ${error}`)
  }

  const getPaymentData = () => {
    switch (selectedPaymentMethod) {
      case 'CREDIT_CARD':
        return {
          cardNumber: formData.cardNumber,
          expiryDate: formData.expiryDate,
          cvv: formData.cvv,
          cardName: formData.cardName
        }
      case 'BANK_TRANSFER':
        return {
          bankName: formData.bankName,
          accountNumber: formData.accountNumber,
          routingNumber: formData.routingNumber
        }
      case 'PAYPAL':
        return {
          paypalEmail: formData.paypalEmail
        }
      case 'JAZZ_CASH':
        return {
          jazzCashNumber: formData.jazzCashNumber
        }
      case 'CHEQUE':
        return {
          chequeNumber: formData.chequeNumber,
          bankBranch: formData.bankBranch
        }
      default:
        return {}
    }
  }

  // Step 1: Authentication
  if (currentStep === 1) {
    return (
      <div className="min-h-screen bg-gray-50 py-12">
        <div className="max-w-md mx-auto">
          <div className="bg-white rounded-lg shadow-sm p-8">
            <div className="text-center mb-8">
              <h1 className="text-2xl font-bold text-gray-900 mb-2">
                {showCreateAccount ? 'Create Your Account' : 'Sign In to Continue'}
              </h1>
              <p className="text-gray-600">
                {showCreateAccount 
                  ? 'Create an account to track your orders' 
                  : 'Sign in to your account to proceed with checkout'
                }
              </p>
            </div>

            <form onSubmit={showCreateAccount ? handleCreateAccount : handleLogin}>
              {showCreateAccount && (
                <div className="grid grid-cols-2 gap-4 mb-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      First Name *
                    </label>
                    <input
                      type="text"
                      name="firstName"
                      required
                      value={authData.firstName}
                      onChange={handleAuthInputChange}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Last Name *
                    </label>
                    <input
                      type="text"
                      name="lastName"
                      required
                      value={authData.lastName}
                      onChange={handleAuthInputChange}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                    />
                  </div>
                </div>
              )}

              <div className="mb-4">
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Email Address *
                </label>
                <input
                  type="email"
                  name="email"
                  required
                  value={authData.email}
                  onChange={handleAuthInputChange}
                  className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                />
              </div>

              <div className="mb-4">
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Password *
                </label>
                <div className="relative">
                  <input
                    type={showPassword ? "text" : "password"}
                    name="password"
                    required
                    value={authData.password}
                    onChange={handleAuthInputChange}
                    className="w-full px-3 py-2 pr-10 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                  />
                  <button
                    type="button"
                    onClick={() => setShowPassword(!showPassword)}
                    className="absolute inset-y-0 right-0 pr-3 flex items-center"
                  >
                    {showPassword ? (
                      <EyeSlashIcon className="h-4 w-4 text-gray-400" />
                    ) : (
                      <EyeIcon className="h-4 w-4 text-gray-400" />
                    )}
                  </button>
                </div>
              </div>

              {showCreateAccount && (
                <>
                  <div className="mb-4">
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Confirm Password *
                    </label>
                    <input
                      type="password"
                      name="confirmPassword"
                      required
                      value={authData.confirmPassword}
                      onChange={handleAuthInputChange}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                    />
                  </div>
                  <div className="mb-6">
                    <label className="flex items-center">
                      <input
                        type="checkbox"
                        name="agreeToTerms"
                        required
                        checked={authData.agreeToTerms}
                        onChange={handleAuthInputChange}
                        className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                      />
                      <span className="ml-2 text-sm text-gray-600">
                        I agree to the <Link href="/terms" className="text-blue-600 hover:text-blue-700">Terms of Service</Link> and <Link href="/privacy" className="text-blue-600 hover:text-blue-700">Privacy Policy</Link>
                      </span>
                    </label>
                  </div>
                </>
              )}

              <button
                type="submit"
                className="w-full bg-blue-600 text-white py-2 px-4 rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition-colors"
              >
                {showCreateAccount ? 'Create Account' : 'Sign In'}
              </button>
            </form>

            <div className="mt-6 text-center">
              <button
                onClick={() => setShowCreateAccount(!showCreateAccount)}
                className="text-blue-600 hover:text-blue-700 text-sm"
              >
                {showCreateAccount 
                  ? 'Already have an account? Sign in' 
                  : "Don't have an account? Create one"
                }
              </button>
            </div>

            <div className="mt-6">
              <div className="relative">
                <div className="absolute inset-0 flex items-center">
                  <div className="w-full border-t border-gray-300" />
                </div>
                <div className="relative flex justify-center text-sm">
                  <span className="px-2 bg-white text-gray-500">Or continue with</span>
                </div>
              </div>

              <div className="mt-6 grid grid-cols-2 gap-3">
                <button
                  onClick={() => signIn('google')}
                  className="w-full inline-flex justify-center py-2 px-4 border border-gray-300 rounded-md shadow-sm bg-white text-sm font-medium text-gray-500 hover:bg-gray-50"
                >
                  Google
                </button>
                <button
                  onClick={() => signIn('github')}
                  className="w-full inline-flex justify-center py-2 px-4 border border-gray-300 rounded-md shadow-sm bg-white text-sm font-medium text-gray-500 hover:bg-gray-50"
                >
                  GitHub
                </button>
              </div>
            </div>

            <div className="mt-6 text-center">
              <button
                onClick={() => setCurrentStep(2)}
                className="text-sm text-gray-500 hover:text-gray-700"
              >
                Continue as guest
              </button>
            </div>
          </div>
        </div>
      </div>
    )
  }

  // Order complete state
  if (orderComplete) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <motion.div
          initial={{ opacity: 0, scale: 0.8 }}
          animate={{ opacity: 1, scale: 1 }}
          transition={{ duration: 0.5 }}
          className="bg-white rounded-lg shadow-lg p-8 max-w-md w-full text-center"
        >
          <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
            <CheckCircleIcon className="w-8 h-8 text-green-600" />
          </div>
          <h2 className="text-2xl font-bold text-gray-900 mb-2">Order Placed Successfully!</h2>
          <p className="text-gray-600 mb-6">
            Thank you for your order. Your order ID is <strong>{orderId}</strong>.
            You will receive a confirmation email shortly.
          </p>
          <div className="flex flex-col space-y-3">
            <button
              onClick={() => router.push(`/dashboard/orders/${orderId}`)}
              className="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 transition-colors"
            >
              View Order Details
            </button>
            <button
              onClick={() => router.push('/')}
              className="text-gray-600 hover:text-gray-800 transition-colors"
            >
              Continue Shopping
            </button>
          </div>
        </motion.div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Progress Steps */}
        <div className="mb-8">
          <div className="flex items-center justify-center">
            <div className="flex items-center space-x-4">
              <div className={`flex items-center justify-center w-8 h-8 rounded-full ${currentStep >= 1 ? 'bg-blue-600 text-white' : 'bg-gray-300 text-gray-600'}`}>
                1
              </div>
              <div className={`w-16 h-1 ${currentStep >= 2 ? 'bg-blue-600' : 'bg-gray-300'}`}></div>
              <div className={`flex items-center justify-center w-8 h-8 rounded-full ${currentStep >= 2 ? 'bg-blue-600 text-white' : 'bg-gray-300 text-gray-600'}`}>
                2
              </div>
              <div className={`w-16 h-1 ${currentStep >= 3 ? 'bg-blue-600' : 'bg-gray-300'}`}></div>
              <div className={`flex items-center justify-center w-8 h-8 rounded-full ${currentStep >= 3 ? 'bg-blue-600 text-white' : 'bg-gray-300 text-gray-600'}`}>
                3
              </div>
            </div>
          </div>
          <div className="flex justify-center mt-2 space-x-8">
            <span className={`text-sm ${currentStep >= 1 ? 'text-blue-600' : 'text-gray-500'}`}>Account</span>
            <span className={`text-sm ${currentStep >= 2 ? 'text-blue-600' : 'text-gray-500'}`}>Details</span>
            <span className={`text-sm ${currentStep >= 3 ? 'text-blue-600' : 'text-gray-500'}`}>Payment</span>
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Main Content */}
          <div className="lg:col-span-2">
            <div className="bg-white rounded-lg shadow-sm p-6">
              <div className="flex items-center justify-between mb-6">
                <h1 className="text-2xl font-bold text-gray-900">Checkout</h1>
                <Link
                  href="/services"
                  className="inline-flex items-center text-blue-600 hover:text-blue-700"
                >
                  <ArrowLeftIcon className="w-4 h-4 mr-2" />
                  Back to Services
                </Link>
              </div>

              <form onSubmit={handleSubmit}>
                {/* Step 2: Billing Information */}
                {currentStep === 2 && (
                  <motion.div
                    initial={{ opacity: 0, x: 20 }}
                    animate={{ opacity: 1, x: 0 }}
                    transition={{ duration: 0.3 }}
                  >
                    <h2 className="text-xl font-semibold text-gray-900 mb-6">Billing Information</h2>
                    
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          First Name *
                        </label>
                        <input
                          type="text"
                          name="firstName"
                          required
                          value={formData.firstName}
                          onChange={handleInputChange}
                          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          Last Name *
                        </label>
                        <input
                          type="text"
                          name="lastName"
                          required
                          value={formData.lastName}
                          onChange={handleInputChange}
                          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        />
                      </div>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          Email Address *
                        </label>
                        <input
                          type="email"
                          name="email"
                          required
                          value={formData.email}
                          onChange={handleInputChange}
                          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          Phone Number *
                        </label>
                        <input
                          type="tel"
                          name="phone"
                          required
                          value={formData.phone}
                          onChange={handleInputChange}
                          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        />
                      </div>
                    </div>

                    <div className="mb-4">
                      <label className="block text-sm font-medium text-gray-700 mb-1">
                        Company (Optional)
                      </label>
                      <input
                        type="text"
                        name="company"
                        value={formData.company}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                      />
                    </div>

                    <div className="mb-4">
                      <label className="block text-sm font-medium text-gray-700 mb-1">
                        Address *
                      </label>
                      <input
                        type="text"
                        name="address"
                        required
                        value={formData.address}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                      />
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          City *
                        </label>
                        <input
                          type="text"
                          name="city"
                          required
                          value={formData.city}
                          onChange={handleInputChange}
                          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          State *
                        </label>
                        <input
                          type="text"
                          name="state"
                          required
                          value={formData.state}
                          onChange={handleInputChange}
                          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          ZIP Code *
                        </label>
                        <input
                          type="text"
                          name="zipCode"
                          required
                          value={formData.zipCode}
                          onChange={handleInputChange}
                          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        />
                      </div>
                    </div>

                    <div className="flex justify-end">
                      <button
                        type="button"
                        onClick={() => setCurrentStep(3)}
                        className="bg-blue-600 text-white px-6 py-2 rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition-colors"
                      >
                        Continue to Payment
                      </button>
                    </div>
                  </motion.div>
                )}

                {/* Step 3: Payment Method */}
                {currentStep === 3 && (
                  <motion.div
                    initial={{ opacity: 0, x: 20 }}
                    animate={{ opacity: 1, x: 0 }}
                    transition={{ duration: 0.3 }}
                  >
                    <h2 className="text-xl font-semibold text-gray-900 mb-6">Payment Method</h2>
                    
                    {/* Payment Method Selection */}
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
                      {paymentMethods.map((method) => {
                        const Icon = method.icon
                        return (
                          <div
                            key={method.id}
                            onClick={() => setSelectedPaymentMethod(method.id)}
                            className={`border rounded-lg p-4 cursor-pointer transition-colors ${
                              selectedPaymentMethod === method.id
                                ? 'border-blue-600 bg-blue-50'
                                : 'border-gray-300 hover:border-gray-400'
                            }`}
                          >
                            <div className="flex items-start space-x-3">
                              <input
                                type="radio"
                                name="paymentMethod"
                                value={method.id}
                                checked={selectedPaymentMethod === method.id}
                                onChange={(e) => setSelectedPaymentMethod(e.target.value)}
                                className="mt-1"
                              />
                              <div className="flex-1">
                                <div className="flex items-center space-x-2 mb-1">
                                  <Icon className="w-5 h-5 text-gray-600" />
                                  <span className="font-medium text-gray-900">{method.name}</span>
                                </div>
                                <p className="text-sm text-gray-600">{method.description}</p>
                              </div>
                            </div>
                          </div>
                        )
                      })}
                    </div>

                    {/* Payment Method Details */}
                    <div className="mb-6">
                      {selectedPaymentMethod === 'CREDIT_CARD' && (
                        <div className="space-y-4">
                          <h3 className="font-medium text-gray-900">Secure Credit Card Payment</h3>
                          {orderCreated && currentOrderId ? (
                            <StripeProvider>
                              <ClientPaymentSection
                                amount={finalTotal}
                                currency="USD"
                                orderId={currentOrderId}
                                onSuccess={handlePaymentSuccess}
                                onError={handlePaymentError}
                              />
                            </StripeProvider>
                          ) : (
                            <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                              <p className="text-blue-800 text-sm">
                                Please complete your billing information and click "Place Order" to proceed with payment.
                              </p>
                            </div>
                          )}
                        </div>
                      )}

                      {selectedPaymentMethod === 'BANK_TRANSFER' && (
                        <div className="space-y-4">
                          <h3 className="font-medium text-gray-900">Bank Transfer Details</h3>
                          <div className="bg-blue-50 border border-blue-200 rounded-md p-4 mb-4">
                            <h4 className="font-medium text-blue-900 mb-2">Our Bank Details:</h4>
                            <div className="text-sm text-blue-800">
                              <p><strong>Bank Name:</strong> Nuwair Systems Bank</p>
                              <p><strong>Account Number:</strong> 1234567890</p>
                              <p><strong>Routing Number:</strong> 021000021</p>
                              <p><strong>Swift Code:</strong> NUWABANKXXX</p>
                            </div>
                          </div>
                          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                Your Bank Name *
                              </label>
                              <input
                                type="text"
                                name="bankName"
                                required
                                value={formData.bankName}
                                onChange={handleInputChange}
                                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                              />
                            </div>
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                Your Account Number *
                              </label>
                              <input
                                type="text"
                                name="accountNumber"
                                required
                                value={formData.accountNumber}
                                onChange={handleInputChange}
                                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                              />
                            </div>
                          </div>
                        </div>
                      )}

                      {selectedPaymentMethod === 'PAYPAL' && (
                        <div className="space-y-4">
                          <h3 className="font-medium text-gray-900">PayPal Details</h3>
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-1">
                              PayPal Email *
                            </label>
                            <input
                              type="email"
                              name="paypalEmail"
                              required
                              value={formData.paypalEmail}
                              onChange={handleInputChange}
                              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                            />
                          </div>
                        </div>
                      )}

                      {selectedPaymentMethod === 'CASH' && (
                        <div className="bg-yellow-50 border border-yellow-200 rounded-md p-4">
                          <h3 className="font-medium text-yellow-900 mb-2">Cash Payment Instructions</h3>
                          <div className="text-sm text-yellow-800">
                            <p className="mb-2">Please visit our office to complete the cash payment:</p>
                            <p><strong>Address:</strong> 123 Business Street, Tech City, TC 12345</p>
                            <p><strong>Hours:</strong> Monday - Friday, 9:00 AM - 6:00 PM</p>
                            <p><strong>Phone:</strong> +1 (555) 123-4567</p>
                          </div>
                        </div>
                      )}

                      {selectedPaymentMethod === 'JAZZ_CASH' && (
                        <div className="space-y-4">
                          <h3 className="font-medium text-gray-900">Jazz Cash Details</h3>
                          <div className="bg-purple-50 border border-purple-200 rounded-md p-4 mb-4">
                            <h4 className="font-medium text-purple-900 mb-2">Jazz Cash Instructions:</h4>
                            <div className="text-sm text-purple-800">
                              <p>Send payment to: <strong>03XX-XXXXXXX</strong></p>
                              <p>Reference: Your order number will be provided after confirmation</p>
                            </div>
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-1">
                              Your Jazz Cash Number *
                            </label>
                            <input
                              type="text"
                              name="jazzCashNumber"
                              required
                              placeholder="03XX-XXXXXXX"
                              value={formData.jazzCashNumber}
                              onChange={handleInputChange}
                              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                            />
                          </div>
                        </div>
                      )}

                      {selectedPaymentMethod === 'CHEQUE' && (
                        <div className="space-y-4">
                          <h3 className="font-medium text-gray-900">Cheque Details</h3>
                          <div className="bg-gray-50 border border-gray-200 rounded-md p-4 mb-4">
                            <h4 className="font-medium text-gray-900 mb-2">Cheque Instructions:</h4>
                            <div className="text-sm text-gray-700">
                              <p>Make cheque payable to: <strong>Nuwair Systems LLC</strong></p>
                              <p>Mail to: 123 Business Street, Tech City, TC 12345</p>
                            </div>
                          </div>
                          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                Cheque Number *
                              </label>
                              <input
                                type="text"
                                name="chequeNumber"
                                required
                                value={formData.chequeNumber}
                                onChange={handleInputChange}
                                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                              />
                            </div>
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">
                                Bank Branch *
                              </label>
                              <input
                                type="text"
                                name="bankBranch"
                                required
                                value={formData.bankBranch}
                                onChange={handleInputChange}
                                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                              />
                            </div>
                          </div>
                        </div>
                      )}
                    </div>

                    <div className="flex justify-between">
                      <button
                        type="button"
                        onClick={() => setCurrentStep(2)}
                        className="text-gray-600 hover:text-gray-800 px-6 py-2 rounded-md border border-gray-300 hover:bg-gray-50 transition-colors"
                      >
                        Back to Details
                      </button>
                      <button
                        type="submit"
                        disabled={isProcessing}
                        className="bg-blue-600 text-white px-6 py-2 rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                      >
                        {isProcessing ? 'Processing...' : `Pay ${formatPrice(finalTotal)}`}
                      </button>
                    </div>
                  </motion.div>
                )}
              </form>
            </div>
          </div>

          {/* Order Summary */}
          <div className="lg:col-span-1">
            <div className="bg-white rounded-lg shadow-sm p-6 sticky top-8">
              <h2 className="text-lg font-semibold text-gray-900 mb-4">Order Summary</h2>
              
              <div className="space-y-4 mb-6">
                {items.map((item) => (
                  <div key={item.id} className="flex justify-between">
                    <div className="flex-1">
                      <h3 className="text-sm font-medium text-gray-900">{item.name}</h3>
                      <p className="text-sm text-gray-500">Qty: {item.quantity}</p>
                    </div>
                    <div className="text-sm font-medium text-gray-900">
                      {formatPrice(item.price * item.quantity)}
                    </div>
                  </div>
                ))}
              </div>

              <div className="border-t border-gray-200 pt-4 space-y-2">
                <div className="flex justify-between text-sm">
                  <span className="text-gray-600">Subtotal</span>
                  <span className="text-gray-900">{formatPrice(totalPrice)}</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span className="text-gray-600">Tax (8%)</span>
                  <span className="text-gray-900">{formatPrice(tax)}</span>
                </div>
                <div className="flex justify-between text-lg font-semibold border-t border-gray-200 pt-2">
                  <span className="text-gray-900">Total</span>
                  <span className="text-gray-900">{formatPrice(finalTotal)}</span>
                </div>
              </div>

              <div className="mt-6 space-y-2">
                <div className="flex items-center text-sm text-gray-600">
                  <ShieldCheckIcon className="w-4 h-4 mr-2" />
                  Secure SSL encryption
                </div>
                <div className="flex items-center text-sm text-gray-600">
                  <TruckIcon className="w-4 h-4 mr-2" />
                  Support included
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}