'use client'

import { useState, useEffect } from 'react'
import { useSession } from 'next-auth/react'
import { motion } from 'framer-motion'
import {
  ServerIcon,
  GlobeAltIcon,
  CreditCardIcon,
  TicketIcon,
  ChartBarIcon,
  DocumentTextIcon,
  CogIcon,
  UserIcon, 
  ShoppingBagIcon, 
  ChatBubbleLeftIcon,
  CalendarIcon
} from '@heroicons/react/24/outline'

interface Service {
  id: string
  name: string
  type: string
  status: 'active' | 'suspended' | 'pending'
  expiryDate: string
  price: number
}

interface Invoice {
  id: string
  amount: number
  status: 'paid' | 'pending' | 'overdue'
  dueDate: string
  description: string
}

interface Ticket {
  id: string
  subject: string
  status: 'open' | 'in-progress' | 'resolved' | 'closed'
  priority: 'low' | 'medium' | 'high' | 'urgent'
  createdAt: string
}

const myOrders = [
  { 
    id: 1, 
    service: 'Business Hosting Pro', 
    status: 'Active', 
    progress: 100,
    startDate: '2024-10-01',
    expectedDelivery: '2025-10-01',
    amount: '$24.99'
  },
  { 
    id: 2, 
    service: 'Domain Registration - nuwair.com', 
    status: 'Active', 
    progress: 100,
    startDate: '2024-10-29',
    expectedDelivery: '2025-10-29',
    amount: '$12.99'
  },
  { 
    id: 3, 
    service: 'SSL Certificate', 
    status: 'Active', 
    progress: 100,
    startDate: '2024-10-01',
    expectedDelivery: '2025-10-01',
    amount: '$15.00'
  },
]

const supportTickets = [
  { id: 1, subject: 'SSL Certificate Setup', status: 'Resolved', priority: 'Medium', date: '2024-10-25' },
  { id: 2, subject: 'Domain DNS Configuration', status: 'Resolved', priority: 'High', date: '2024-10-29' },
  { id: 3, subject: 'Email Setup Assistance', status: 'Open', priority: 'Low', date: '2024-10-30' },
]

export default function ClientDashboard() {
  const { data: session } = useSession()
  const [activeTab, setActiveTab] = useState('overview')
  const [services, setServices] = useState<Service[]>([])
  const [invoices, setInvoices] = useState<Invoice[]>([])
  const [tickets, setTickets] = useState<Ticket[]>([])
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    if (session?.user) {
      fetchDashboardData()
    }
  }, [session])

  const fetchDashboardData = async () => {
    try {
      setLoading(true)
      // Fetch services, invoices, and tickets
      setServices([
        {
          id: '1',
          name: 'Business Hosting Pro',
          type: 'hosting',
          status: 'active',
          expiryDate: '2025-10-01',
          price: 24.99
        },
        {
          id: '2', 
          name: 'nuwair.com',
          type: 'domain',
          status: 'active',
          expiryDate: '2025-10-29',
          price: 12.99
        },
        {
          id: '3',
          name: 'SSL Certificate',
          type: 'ssl',
          status: 'active',
          expiryDate: '2025-10-01',
          price: 15.00
        }
      ])
      
      setInvoices([
        {
          id: 'INV-001',
          amount: 24.99,
          status: 'paid',
          dueDate: '2025-10-01',
          description: 'Business Hosting Pro - Monthly'
        },
        {
          id: 'INV-002',
          amount: 12.99,
          status: 'paid',
          dueDate: '2025-10-29',
          description: 'Domain Registration - nuwair.com'
        },
        {
          id: 'INV-003',
          amount: 15.00,
          status: 'pending',
          dueDate: '2025-11-01',
          description: 'SSL Certificate Renewal'
        }
      ])
      
      setTickets([
        {
          id: 'TKT-001',
          subject: 'SSL Certificate Setup',
          status: 'resolved',
          priority: 'medium',
          createdAt: '2025-10-25'
        },
        {
          id: 'TKT-002',
          subject: 'Email Setup Assistance',
          status: 'open',
          priority: 'low',
          createdAt: '2025-10-30'
        }
      ])
    } catch (error) {
      console.error('Error fetching dashboard data:', error)
    } finally {
      setLoading(false)
    }
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active':
      case 'paid':
      case 'resolved':
      case 'Active':
      case 'Completed':
        return 'text-green-600 bg-green-100'
      case 'pending':
      case 'in-progress':
      case 'In Progress':
      case 'Open':
        return 'text-yellow-600 bg-yellow-100'
      case 'suspended':
      case 'overdue':
      case 'urgent':
        return 'text-red-600 bg-red-100'
      case 'Under Review':
        return 'text-blue-600 bg-blue-100'
      default:
        return 'text-gray-600 bg-gray-100'
    }
  }

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'urgent':
      case 'High':
        return 'text-red-600 bg-red-100'
      case 'high':
        return 'text-orange-600 bg-orange-100'
      case 'medium':
      case 'Medium':
        return 'text-yellow-600 bg-yellow-100'
      case 'low':
      case 'Low':
        return 'text-green-600 bg-green-100'
      default:
        return 'text-gray-600 bg-gray-100'
    }
  }

  if (!session?.user) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-50">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-gray-900 mb-4">Access Denied</h1>
          <p className="text-gray-600 mb-8">Please sign in to access your client dashboard.</p>
          <a
            href="/auth/signin"
            className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            Sign In
          </a>
        </div>
      </div>
    )
  }

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-50">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto"></div>
          <p className="text-gray-600 mt-4">Loading your dashboard...</p>
        </div>
      </div>
    )
  }

  const navigation = [
    { name: 'Overview', id: 'overview', icon: UserIcon },
    { name: 'My Services', id: 'services', icon: ServerIcon },
    { name: 'My Orders', id: 'orders', icon: ShoppingBagIcon },
    { name: 'Support Tickets', id: 'support', icon: ChatBubbleLeftIcon },
    { name: 'Invoices', id: 'invoices', icon: DocumentTextIcon },
    { name: 'Payment Methods', id: 'payments', icon: CreditCardIcon },
    { name: 'Account Settings', id: 'settings', icon: CogIcon },
  ]

  const renderOverview = () => (
    <div className="space-y-8">
      {/* Welcome Section */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        className="bg-gradient-to-br from-blue-600 to-purple-600 rounded-lg p-8 text-white"
      >
        <h2 className="text-2xl font-bold mb-2">Welcome back, {session?.user?.name || session?.user?.email}!</h2>
        <p className="text-blue-100">Here's what's happening with your services and account</p>
      </motion.div>

      {/* Quick Stats */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.1 }}
        className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6"
      >
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center">
            <div className="p-3 bg-blue-100 rounded-lg">
              <ServerIcon className="w-6 h-6 text-blue-600" />
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">Active Services</p>
              <p className="text-2xl font-bold text-gray-900">
                {services.filter(s => s.status === 'active').length}
              </p>
            </div>
          </div>
        </div>

        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center">
            <div className="p-3 bg-green-100 rounded-lg">
              <CreditCardIcon className="w-6 h-6 text-green-600" />
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">Paid Invoices</p>
              <p className="text-2xl font-bold text-gray-900">
                {invoices.filter(i => i.status === 'paid').length}
              </p>
            </div>
          </div>
        </div>

        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center">
            <div className="p-3 bg-yellow-100 rounded-lg">
              <TicketIcon className="w-6 h-6 text-yellow-600" />
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">Open Tickets</p>
              <p className="text-2xl font-bold text-gray-900">
                {tickets.filter(t => ['open', 'in-progress'].includes(t.status)).length}
              </p>
            </div>
          </div>
        </div>

        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center">
            <div className="p-3 bg-purple-100 rounded-lg">
              <ChartBarIcon className="w-6 h-6 text-purple-600" />
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">Total Spent</p>
              <p className="text-2xl font-bold text-gray-900">
                ${invoices.filter(i => i.status === 'paid').reduce((sum, i) => sum + i.amount, 0).toFixed(2)}
              </p>
            </div>
          </div>
        </div>
      </motion.div>

      {/* Services and Recent Activity */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
        {/* Active Services */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.2 }}
          className="bg-white rounded-lg shadow-sm border border-gray-200"
        >
          <div className="p-6 border-b border-gray-200">
            <h3 className="text-lg font-semibold text-gray-900 flex items-center">
              <ServerIcon className="w-5 h-5 mr-2" />
              Your Services
            </h3>
          </div>
          <div className="p-6">
            <div className="space-y-4">
              {services.slice(0, 3).map((service) => (
                <div
                  key={service.id}
                  className="flex items-center justify-between p-4 bg-gray-50 rounded-lg"
                >
                  <div className="flex items-center">
                    <div className="p-2 bg-blue-100 rounded-lg mr-3">
                      {service.type === 'hosting' ? (
                        <ServerIcon className="w-4 h-4 text-blue-600" />
                      ) : service.type === 'domain' ? (
                        <GlobeAltIcon className="w-4 h-4 text-blue-600" />
                      ) : (
                        <DocumentTextIcon className="w-4 h-4 text-blue-600" />
                      )}
                    </div>
                    <div>
                      <h4 className="text-sm font-medium text-gray-900">{service.name}</h4>
                      <p className="text-xs text-gray-500">Expires: {service.expiryDate}</p>
                    </div>
                  </div>
                  <div className="text-right">
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(service.status)}`}>
                      {service.status}
                    </span>
                    <p className="text-sm font-medium text-gray-900 mt-1">${service.price}</p>
                  </div>
                </div>
              ))}
            </div>
            <div className="mt-4">
              <button
                onClick={() => setActiveTab('services')}
                className="w-full text-center text-blue-600 hover:text-blue-700 text-sm font-medium"
              >
                View All Services →
              </button>
            </div>
          </div>
        </motion.div>

        {/* Recent Invoices */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.3 }}
          className="bg-white rounded-lg shadow-sm border border-gray-200"
        >
          <div className="p-6 border-b border-gray-200">
            <h3 className="text-lg font-semibold text-gray-900 flex items-center">
              <CreditCardIcon className="w-5 h-5 mr-2" />
              Recent Invoices
            </h3>
          </div>
          <div className="p-6">
            <div className="space-y-4">
              {invoices.slice(0, 3).map((invoice) => (
                <div
                  key={invoice.id}
                  className="flex items-center justify-between p-4 bg-gray-50 rounded-lg"
                >
                  <div>
                    <h4 className="text-sm font-medium text-gray-900">{invoice.id}</h4>
                    <p className="text-xs text-gray-500">{invoice.description}</p>
                    <p className="text-xs text-gray-500">Due: {invoice.dueDate}</p>
                  </div>
                  <div className="text-right">
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(invoice.status)}`}>
                      {invoice.status}
                    </span>
                    <p className="text-sm font-medium text-gray-900 mt-1">${invoice.amount}</p>
                  </div>
                </div>
              ))}
            </div>
            <div className="mt-4">
              <button
                onClick={() => setActiveTab('invoices')}
                className="w-full text-center text-blue-600 hover:text-blue-700 text-sm font-medium"
              >
                View All Invoices →
              </button>
            </div>
          </div>
        </motion.div>
      </div>

      {/* Support Tickets */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.4 }}
        className="bg-white rounded-lg shadow-sm border border-gray-200"
      >
        <div className="p-6 border-b border-gray-200">
          <div className="flex items-center justify-between">
            <h3 className="text-lg font-semibold text-gray-900 flex items-center">
              <TicketIcon className="w-5 h-5 mr-2" />
              Support Tickets
            </h3>
            <a
              href="/dashboard/tickets"
              className="text-blue-600 hover:text-blue-700 text-sm font-medium"
            >
              View All Tickets
            </a>
          </div>
        </div>
        <div className="p-6">
          {tickets.length === 0 ? (
            <div className="text-center py-8">
              <TicketIcon className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <p className="text-gray-600">No support tickets yet</p>
              <a
                href="/dashboard/tickets"
                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors mt-4"
              >
                Create Your First Ticket
              </a>
            </div>
          ) : (
            <div className="space-y-4">
              {tickets.map((ticket) => (
                <div
                  key={ticket.id}
                  className="flex items-center justify-between p-4 bg-gray-50 rounded-lg"
                >
                  <div>
                    <h4 className="text-sm font-medium text-gray-900">{ticket.subject}</h4>
                    <p className="text-xs text-gray-500">Ticket #{ticket.id}</p>
                    <p className="text-xs text-gray-500">Created: {ticket.createdAt}</p>
                  </div>
                  <div className="text-right space-y-1">
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(ticket.status)}`}>
                      {ticket.status}
                    </span>
                    <br />
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getPriorityColor(ticket.priority)}`}>
                      {ticket.priority}
                    </span>
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>
      </motion.div>

      {/* Quick Actions */}
      <motion.div
        initial={{ opacity: 0, y: 20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.5 }}
        className="bg-gradient-to-br from-blue-600 to-purple-600 rounded-lg p-8 text-white"
      >
        <h3 className="text-xl font-bold mb-6">Quick Actions</h3>
        <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
          <a
            href="/services"
            className="flex items-center p-4 bg-white/10 rounded-lg hover:bg-white/20 transition-colors"
          >
            <ServerIcon className="w-6 h-6 mr-3" />
            <span className="font-medium">Order New Service</span>
          </a>
          <a
            href="/payments"
            className="flex items-center p-4 bg-white/10 rounded-lg hover:bg-white/20 transition-colors"
          >
            <CreditCardIcon className="w-6 h-6 mr-3" />
            <span className="font-medium">Make Payment</span>
          </a>
          <a
            href="/dashboard/communication"
            className="flex items-center p-4 bg-white/10 rounded-lg hover:bg-white/20 transition-colors"
          >
            <ChatBubbleLeftIcon className="w-6 h-6 mr-3" />
            <span className="font-medium">Project Communication</span>
          </a>
          <a
            href="/dashboard/tickets"
            className="flex items-center p-4 bg-white/10 rounded-lg hover:bg-white/20 transition-colors"
          >
            <TicketIcon className="w-6 h-6 mr-3" />
            <span className="font-medium">Support Tickets</span>
          </a>
          <button
            onClick={() => setActiveTab('settings')}
            className="flex items-center p-4 bg-white/10 rounded-lg hover:bg-white/20 transition-colors"
          >
            <CogIcon className="w-6 h-6 mr-3" />
            <span className="font-medium">Account Settings</span>
          </button>
        </div>
      </motion.div>
    </div>
  )

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Navigation Header */}
      <div className="bg-white shadow">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <h1 className="text-xl font-semibold text-gray-900">Client Portal</h1>
            </div>
            <div className="flex items-center space-x-4">
              <span className="text-sm text-gray-600">
                Welcome, {session?.user?.name || session?.user?.email}
              </span>
              <a
                href="/api/auth/signout"
                className="text-sm text-gray-600 hover:text-gray-900"
              >
                Sign Out
              </a>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div className="flex flex-col lg:flex-row gap-8">
          {/* Sidebar */}
          <div className="lg:w-64 flex-shrink-0">
            <nav className="space-y-1">
              {navigation.map((item) => (
                <button
                  key={item.id}
                  onClick={() => setActiveTab(item.id)}
                  className={`w-full flex items-center px-3 py-2 text-sm font-medium rounded-md transition-colors ${
                    activeTab === item.id
                      ? 'bg-blue-100 text-blue-700'
                      : 'text-gray-600 hover:bg-gray-100 hover:text-gray-900'
                  }`}
                >
                  <item.icon className="mr-3 h-5 w-5" />
                  {item.name}
                </button>
              ))}
            </nav>
          </div>

          {/* Main Content */}
          <div className="flex-1">
            {activeTab === 'overview' && renderOverview()}

            {activeTab === 'services' && (
              <div>
                <h2 className="text-2xl font-bold text-gray-900 mb-6">My Services</h2>
                <div className="space-y-6">
                  {services.map((service) => (
                    <div key={service.id} className="bg-white rounded-lg shadow p-6">
                      <div className="flex justify-between items-start mb-4">
                        <div className="flex items-center">
                          <div className="p-3 bg-blue-100 rounded-lg mr-4">
                            {service.type === 'hosting' ? (
                              <ServerIcon className="w-6 h-6 text-blue-600" />
                            ) : service.type === 'domain' ? (
                              <GlobeAltIcon className="w-6 h-6 text-blue-600" />
                            ) : (
                              <DocumentTextIcon className="w-6 h-6 text-blue-600" />
                            )}
                          </div>
                          <div>
                            <h3 className="text-lg font-medium text-gray-900">{service.name}</h3>
                            <p className="text-sm text-gray-500 capitalize">{service.type} Service</p>
                          </div>
                        </div>
                        <div className="text-right">
                          <p className="text-lg font-semibold text-gray-900">${service.price}/month</p>
                          <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(service.status)}`}>
                            {service.status}
                          </span>
                        </div>
                      </div>
                      
                      <div className="grid grid-cols-2 gap-4 text-sm">
                        <div>
                          <span className="text-gray-500">Service ID:</span>
                          <p className="font-medium">{service.id}</p>
                        </div>
                        <div>
                          <span className="text-gray-500">Expiry Date:</span>
                          <p className="font-medium">{service.expiryDate}</p>
                        </div>
                      </div>
                      
                      <div className="mt-4 flex space-x-2">
                        <button className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                          Manage
                        </button>
                        <button className="px-4 py-2 bg-gray-200 text-gray-700 rounded-lg hover:bg-gray-300 transition-colors">
                          Renew
                        </button>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {activeTab === 'orders' && (
              <div>
                <h2 className="text-2xl font-bold text-gray-900 mb-6">My Orders</h2>
                <div className="space-y-6">
                  {myOrders.map((order) => (
                    <div key={order.id} className="bg-white rounded-lg shadow p-6">
                      <div className="flex justify-between items-start mb-4">
                        <div>
                          <h3 className="text-lg font-medium text-gray-900">{order.service}</h3>
                          <p className="text-sm text-gray-500">Order #{order.id}</p>
                        </div>
                        <div className="text-right">
                          <p className="text-lg font-semibold text-gray-900">{order.amount}</p>
                          <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(order.status)}`}>
                            {order.status}
                          </span>
                        </div>
                      </div>
                      
                      <div className="mb-4">
                        <div className="flex justify-between text-sm text-gray-600 mb-1">
                          <span>Progress</span>
                          <span>{order.progress}%</span>
                        </div>
                        <div className="w-full bg-gray-200 rounded-full h-2">
                          <div 
                            className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                            style={{ width: `${order.progress}%` }}
                          ></div>
                        </div>
                      </div>
                      
                      <div className="grid grid-cols-2 gap-4 text-sm">
                        <div>
                          <span className="text-gray-500">Start Date:</span>
                          <p className="font-medium">{order.startDate}</p>
                        </div>
                        <div>
                          <span className="text-gray-500">Expected Delivery:</span>
                          <p className="font-medium">{order.expectedDelivery}</p>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {activeTab === 'support' && (
              <div>
                <div className="flex justify-between items-center mb-6">
                  <h2 className="text-2xl font-bold text-gray-900">Support Tickets</h2>
                  <button className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                    Create New Ticket
                  </button>
                </div>
                <div className="bg-white shadow rounded-lg">
                  <div className="overflow-x-auto">
                    <table className="min-w-full divide-y divide-gray-200">
                      <thead className="bg-gray-50">
                        <tr>
                          <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Subject
                          </th>
                          <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Priority
                          </th>
                          <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Status
                          </th>
                          <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                            Date
                          </th>
                        </tr>
                      </thead>
                      <tbody className="bg-white divide-y divide-gray-200">
                        {supportTickets.map((ticket) => (
                          <tr key={ticket.id}>
                            <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                              {ticket.subject}
                            </td>
                            <td className="px-6 py-4 whitespace-nowrap">
                              <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getPriorityColor(ticket.priority)}`}>
                                {ticket.priority}
                              </span>
                            </td>
                            <td className="px-6 py-4 whitespace-nowrap">
                              <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(ticket.status)}`}>
                                {ticket.status}
                              </span>
                            </td>
                            <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                              {ticket.date}
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                </div>
              </div>
            )}

            {activeTab === 'invoices' && (
              <div>
                <h2 className="text-2xl font-bold text-gray-900 mb-6">Invoices & Billing</h2>
                <div className="space-y-6">
                  {invoices.map((invoice) => (
                    <div key={invoice.id} className="bg-white rounded-lg shadow p-6">
                      <div className="flex justify-between items-start mb-4">
                        <div>
                          <h3 className="text-lg font-medium text-gray-900">{invoice.id}</h3>
                          <p className="text-sm text-gray-500">{invoice.description}</p>
                        </div>
                        <div className="text-right">
                          <p className="text-lg font-semibold text-gray-900">${invoice.amount}</p>
                          <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(invoice.status)}`}>
                            {invoice.status}
                          </span>
                        </div>
                      </div>
                      
                      <div className="grid grid-cols-2 gap-4 text-sm">
                        <div>
                          <span className="text-gray-500">Invoice ID:</span>
                          <p className="font-medium">{invoice.id}</p>
                        </div>
                        <div>
                          <span className="text-gray-500">Due Date:</span>
                          <p className="font-medium">{invoice.dueDate}</p>
                        </div>
                      </div>
                      
                      <div className="mt-4 flex space-x-2">
                        <button className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                          View Invoice
                        </button>
                        {invoice.status === 'pending' && (
                          <button className="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors">
                            Pay Now
                          </button>
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {activeTab === 'payments' && (
              <div className="bg-white shadow rounded-lg p-6">
                <h2 className="text-2xl font-bold text-gray-900 mb-6">Payment & Billing</h2>
                
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-8">
                  <a
                    href="/payments"
                    className="block p-6 bg-gradient-to-br from-blue-50 to-blue-100 border border-blue-200 rounded-lg hover:shadow-lg transition-all"
                  >
                    <CreditCardIcon className="w-8 h-8 text-blue-600 mb-3" />
                    <h3 className="font-semibold text-gray-900 mb-2">Payment Gateway</h3>
                    <p className="text-sm text-gray-600">
                      Make payments with multiple options including local Pakistani methods
                    </p>
                  </a>

                  <a
                    href="/subscriptions"
                    className="block p-6 bg-gradient-to-br from-green-50 to-green-100 border border-green-200 rounded-lg hover:shadow-lg transition-all"
                  >
                    <CalendarIcon className="w-8 h-8 text-green-600 mb-3" />
                    <h3 className="font-semibold text-gray-900 mb-2">Subscriptions</h3>
                    <p className="text-sm text-gray-600">
                      Manage recurring payments and subscription plans
                    </p>
                  </a>

                  <a
                    href="/invoices"
                    className="block p-6 bg-gradient-to-br from-purple-50 to-purple-100 border border-purple-200 rounded-lg hover:shadow-lg transition-all"
                  >
                    <DocumentTextIcon className="w-8 h-8 text-purple-600 mb-3" />
                    <h3 className="font-semibold text-gray-900 mb-2">Invoice Management</h3>
                    <p className="text-sm text-gray-600">
                      View, manage, and download your invoices and billing history
                    </p>
                  </a>
                </div>

                <div className="bg-gray-50 rounded-lg p-6">
                  <h3 className="font-medium text-gray-900 mb-4">Quick Payment Options</h3>
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                    <div className="flex items-center space-x-2 p-3 bg-white rounded-lg border">
                      <span className="text-lg">💳</span>
                      <span className="text-sm font-medium">Credit Card</span>
                    </div>
                    <div className="flex items-center space-x-2 p-3 bg-white rounded-lg border">
                      <span className="text-lg">📱</span>
                      <span className="text-sm font-medium">Easypaisa</span>
                    </div>
                    <div className="flex items-center space-x-2 p-3 bg-white rounded-lg border">
                      <span className="text-lg">💰</span>
                      <span className="text-sm font-medium">JazzCash</span>
                    </div>
                    <div className="flex items-center space-x-2 p-3 bg-white rounded-lg border">
                      <span className="text-lg">🏦</span>
                      <span className="text-sm font-medium">Bank Transfer</span>
                    </div>
                  </div>
                </div>
              </div>
            )}

            {activeTab === 'settings' && (
              <div className="bg-white shadow rounded-lg p-6">
                <h2 className="text-2xl font-bold text-gray-900 mb-6">Account Settings</h2>
                <div className="space-y-6">
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Full Name</label>
                    <input
                      type="text"
                      className="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500"
                      placeholder={session?.user?.name || "Enter your name"}
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Email Address</label>
                    <input
                      type="email"
                      className="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500"
                      placeholder={session?.user?.email || "Enter your email"}
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Phone Number</label>
                    <input
                      type="tel"
                      className="mt-1 block w-full border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500"
                      placeholder="Enter your phone number"
                    />
                  </div>
                  <div>
                    <button className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                      Save Changes
                    </button>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  )
}