'use client'

import { useState, useEffect } from 'react'
import { useSession } from 'next-auth/react'
import { motion } from 'framer-motion'
import {
  TicketIcon,
  PaperClipIcon,
  ExclamationTriangleIcon,
  CheckCircleIcon,
  ArrowLeftIcon
} from '@heroicons/react/24/outline'
import Link from 'next/link'

export default function NewTicket() {
  const { data: session } = useSession()
  const [formData, setFormData] = useState({
    subject: '',
    priority: 'medium',
    category: 'general',
    description: '',
    attachments: []
  })
  const [isSubmitting, setIsSubmitting] = useState(false)
  const [submitted, setSubmitted] = useState(false)
  const [ticketId, setTicketId] = useState('')

  // Generate ticket ID on client side to avoid hydration mismatch
  useEffect(() => {
    if (submitted && !ticketId) {
      const id = `TKT-${Date.now().toString().slice(-6)}`
      setTicketId(id)
    }
  }, [submitted, ticketId])

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsSubmitting(true)

    try {
      // Mock API call - replace with actual API endpoint
      await new Promise(resolve => setTimeout(resolve, 2000))
      setSubmitted(true)
    } catch (error) {
      console.error('Error submitting ticket:', error)
    } finally {
      setIsSubmitting(false)
    }
  }

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>) => {
    const { name, value } = e.target
    setFormData(prev => ({
      ...prev,
      [name]: value
    }))
  }

  if (!session?.user) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-50">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-gray-900 mb-4">Access Denied</h1>
          <p className="text-gray-600 mb-8">Please sign in to create a support ticket.</p>
          <Link
            href="/auth/signin"
            className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            Sign In
          </Link>
        </div>
      </div>
    )
  }

  if (submitted) {
    return (
      <div className="min-h-screen bg-gray-50">
        <div className="max-w-3xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="bg-white rounded-lg shadow-sm p-8 text-center"
          >
            <CheckCircleIcon className="w-16 h-16 text-green-600 mx-auto mb-6" />
            <h1 className="text-2xl font-bold text-gray-900 mb-4">Ticket Submitted Successfully!</h1>
            <p className="text-gray-600 mb-8">
              Your support ticket has been created and assigned a ticket number. 
              Our support team will review your request and respond within 24 hours.
            </p>
            <div className="bg-green-50 rounded-lg p-4 mb-8">
              <p className="text-sm text-green-700">
                <strong>Ticket #{ticketId || 'Generating...'}</strong> - {formData.subject}
              </p>
            </div>
            <div className="flex justify-center space-x-4">
              <Link
                href="/dashboard"
                className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
              >
                Back to Dashboard
              </Link>
              <button
                onClick={() => {
                  setSubmitted(false)
                  setFormData({
                    subject: '',
                    priority: 'medium',
                    category: 'general',
                    description: '',
                    attachments: []
                  })
                }}
                className="px-6 py-3 bg-gray-200 text-gray-700 rounded-lg hover:bg-gray-300 transition-colors"
              >
                Create Another Ticket
              </button>
            </div>
          </motion.div>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-3xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Header */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          className="mb-8"
        >
          <Link
            href="/dashboard"
            className="inline-flex items-center text-blue-600 hover:text-blue-700 mb-4"
          >
            <ArrowLeftIcon className="w-4 h-4 mr-2" />
            Back to Dashboard
          </Link>
          <h1 className="text-3xl font-bold text-gray-900 mb-2">Create Support Ticket</h1>
          <p className="text-gray-600">
            Tell us about your issue and our support team will help you resolve it
          </p>
        </motion.div>

        {/* Form */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.1 }}
          className="bg-white rounded-lg shadow-sm"
        >
          <form onSubmit={handleSubmit} className="p-8">
            <div className="space-y-6">
              {/* Subject */}
              <div>
                <label htmlFor="subject" className="block text-sm font-medium text-gray-700 mb-2">
                  Subject *
                </label>
                <input
                  type="text"
                  id="subject"
                  name="subject"
                  required
                  value={formData.subject}
                  onChange={handleInputChange}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  placeholder="Brief description of your issue"
                />
              </div>

              {/* Priority and Category Row */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div>
                  <label htmlFor="priority" className="block text-sm font-medium text-gray-700 mb-2">
                    Priority
                  </label>
                  <select
                    id="priority"
                    name="priority"
                    value={formData.priority}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  >
                    <option value="low">Low - General questions</option>
                    <option value="medium">Medium - Service issues</option>
                    <option value="high">High - Business impact</option>
                    <option value="urgent">Urgent - Critical system down</option>
                  </select>
                </div>

                <div>
                  <label htmlFor="category" className="block text-sm font-medium text-gray-700 mb-2">
                    Category
                  </label>
                  <select
                    id="category"
                    name="category"
                    value={formData.category}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  >
                    <option value="general">General Support</option>
                    <option value="hosting">Hosting Issues</option>
                    <option value="domain">Domain Management</option>
                    <option value="email">Email Configuration</option>
                    <option value="ssl">SSL Certificates</option>
                    <option value="billing">Billing & Payments</option>
                    <option value="technical">Technical Issues</option>
                  </select>
                </div>
              </div>

              {/* Description */}
              <div>
                <label htmlFor="description" className="block text-sm font-medium text-gray-700 mb-2">
                  Description *
                </label>
                <textarea
                  id="description"
                  name="description"
                  required
                  rows={6}
                  value={formData.description}
                  onChange={handleInputChange}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  placeholder="Please provide detailed information about your issue, including any error messages, steps to reproduce, and what you've already tried..."
                />
              </div>

              {/* Attachments */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Attachments (Optional)
                </label>
                <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                  <PaperClipIcon className="w-8 h-8 text-gray-400 mx-auto mb-2" />
                  <p className="text-sm text-gray-600 mb-2">
                    Drag and drop files here, or click to browse
                  </p>
                  <input
                    type="file"
                    multiple
                    accept=".jpg,.jpeg,.png,.gif,.pdf,.doc,.docx,.txt"
                    className="hidden"
                    id="file-upload"
                  />
                  <label
                    htmlFor="file-upload"
                    className="cursor-pointer text-blue-600 hover:text-blue-700 text-sm font-medium"
                  >
                    Choose Files
                  </label>
                  <p className="text-xs text-gray-500 mt-1">
                    Supported formats: JPG, PNG, GIF, PDF, DOC, TXT (Max 10MB)
                  </p>
                </div>
              </div>

              {/* Priority Warning */}
              {formData.priority === 'urgent' && (
                <motion.div
                  initial={{ opacity: 0, y: 10 }}
                  animate={{ opacity: 1, y: 0 }}
                  className="bg-red-50 border border-red-200 rounded-lg p-4 flex items-start"
                >
                  <ExclamationTriangleIcon className="w-5 h-5 text-red-600 mr-3 mt-0.5" />
                  <div>
                    <h4 className="text-sm font-medium text-red-800">Urgent Priority Selected</h4>
                    <p className="text-sm text-red-700 mt-1">
                      Please only use urgent priority for critical system outages or security issues. 
                      For general questions, please select a lower priority.
                    </p>
                  </div>
                </motion.div>
              )}

              {/* Contact Information */}
              <div className="bg-blue-50 rounded-lg p-4">
                <h4 className="text-sm font-medium text-blue-800 mb-2">Contact Information</h4>
                <p className="text-sm text-blue-700">
                  Ticket will be created for: <strong>{session.user.email}</strong>
                </p>
                <p className="text-sm text-blue-700 mt-1">
                  Response will be sent to this email address. Make sure it's accessible.
                </p>
              </div>
            </div>

            {/* Submit Button */}
            <div className="mt-8 flex justify-end space-x-4">
              <Link
                href="/dashboard"
                className="px-6 py-3 bg-gray-200 text-gray-700 rounded-lg hover:bg-gray-300 transition-colors"
              >
                Cancel
              </Link>
              <button
                type="submit"
                disabled={isSubmitting || !formData.subject || !formData.description}
                className={`px-6 py-3 rounded-lg transition-colors flex items-center ${
                  isSubmitting || !formData.subject || !formData.description
                    ? 'bg-gray-300 text-gray-500 cursor-not-allowed'
                    : 'bg-blue-600 text-white hover:bg-blue-700'
                }`}
              >
                {isSubmitting ? (
                  <>
                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                    Submitting...
                  </>
                ) : (
                  <>
                    <TicketIcon className="w-4 h-4 mr-2" />
                    Submit Ticket
                  </>
                )}
              </button>
            </div>
          </form>
        </motion.div>

        {/* Help Section */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.2 }}
          className="mt-8 bg-white rounded-lg shadow-sm p-6"
        >
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Need Help Faster?</h3>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
              <h4 className="font-medium text-gray-900 mb-2">📚 Knowledge Base</h4>
              <p className="text-sm text-gray-600 mb-2">
                Find answers to common questions in our comprehensive knowledge base.
              </p>
              <a href="#" className="text-blue-600 hover:text-blue-700 text-sm font-medium">
                Browse Articles →
              </a>
            </div>
            <div>
              <h4 className="font-medium text-gray-900 mb-2">💬 Live Chat</h4>
              <p className="text-sm text-gray-600 mb-2">
                Chat with our support team in real-time during business hours.
              </p>
              <a href="#" className="text-blue-600 hover:text-blue-700 text-sm font-medium">
                Start Chat →
              </a>
            </div>
          </div>
        </motion.div>
      </div>
    </div>
  )
}