import React from "react";
import type { Metadata } from "next";
import { Inter, Poppins } from "next/font/google";
import "./globals.css";
import { Providers } from "./providers";
import Header from "@/components/Header";
import Footer from "@/components/Footer";
import EnhancedShoppingCart from "@/components/EnhancedShoppingCart";
import LiveChatWidget from "@/components/LiveChatWidget";
import HydrationBoundary from "@/components/HydrationBoundary";

const inter = Inter({ 
  subsets: ["latin"],
  variable: "--font-inter",
});

const poppins = Poppins({ 
  subsets: ["latin"],
  weight: ["300", "400", "500", "600", "700", "800"],
  variable: "--font-poppins",
});

export const metadata: Metadata = {
  title: {
    default: "Nuwair Systems - Professional IT Services & Web Hosting in Pakistan",
    template: "%s | Nuwair Systems"
  },
  description: "Professional IT services including domain registration, web hosting, digital marketing, software development, Google Ads, SEO services, and web development in Pakistan. Trusted by 500+ businesses.",
  keywords: ["IT services Pakistan", "web hosting Pakistan", "domain registration", "digital marketing", "software development", "SEO Pakistan", "Google Ads", "Lahore IT company"],
  authors: [{ name: "Nuwair Systems" }],
  creator: "Nuwair Systems",
  publisher: "Nuwair Systems",
  icons: {
    icon: '/favicon.ico',
    shortcut: '/favicon.ico',
    apple: '/favicon.ico',
  },
  formatDetection: {
    email: false,
    address: false,
    telephone: false,
  },
  metadataBase: new URL(process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'),
  alternates: {
    canonical: '/',
  },
  openGraph: {
    type: 'website',
    locale: 'en_US',
    url: '/',
    title: 'Nuwair Systems - Professional IT Services & Solutions',
    description: 'Professional IT services including domain registration, web hosting, digital marketing, software development, and more.',
    siteName: 'Nuwair Systems',
    images: [
      {
        url: '/og-image.jpg',
        width: 1200,
        height: 630,
        alt: 'Nuwair Systems',
      },
    ],
  },
  twitter: {
    card: 'summary_large_image',
    title: 'Nuwair Systems - Professional IT Services & Solutions',
    description: 'Professional IT services including domain registration, web hosting, digital marketing, software development, and more.',
    images: ['/og-image.jpg'],
    creator: '@nuwairsystems',
  },
  robots: {
    index: true,
    follow: true,
    googleBot: {
      index: true,
      follow: true,
      'max-video-preview': -1,
      'max-image-preview': 'large',
      'max-snippet': -1,
    },
  },
  verification: {
    google: process.env.GOOGLE_SITE_VERIFICATION,
    yandex: process.env.YANDEX_VERIFICATION,
  },
};

export default function RootLayout({
  children,
}: {
  children: React.ReactNode;
}) {
  return (
    <html lang="en" className={`${inter.variable} ${poppins.variable}`} suppressHydrationWarning>
      <head>
        <script
          dangerouslySetInnerHTML={{
            __html: `
              // Aggressive browser extension attribute removal
              (function() {
                const attrs = ['bis_skin_checked', 'bis_register', 'bis_id', 'bis_size', 'bis_frame_id'];
                
                const cleanAll = () => {
                  attrs.forEach(attr => {
                    document.querySelectorAll('[' + attr + ']').forEach(el => {
                      el.removeAttribute(attr);
                    });
                  });
                };
                
                // Run immediately
                cleanAll();
                
                // Run multiple times during page load
                setTimeout(cleanAll, 0);
                setTimeout(cleanAll, 10);
                setTimeout(cleanAll, 50);
                setTimeout(cleanAll, 100);
                
                // Run on DOM ready
                if (document.readyState === 'loading') {
                  document.addEventListener('DOMContentLoaded', cleanAll);
                } else {
                  cleanAll();
                }
                
                // Use MutationObserver to catch extensions adding attributes
                const observer = new MutationObserver((mutations) => {
                  mutations.forEach((mutation) => {
                    if (mutation.type === 'attributes' && attrs.includes(mutation.attributeName)) {
                      mutation.target.removeAttribute(mutation.attributeName);
                    }
                  });
                });
                
                // Start observing after a short delay
                setTimeout(() => {
                  observer.observe(document.documentElement, {
                    attributes: true,
                    attributeFilter: attrs,
                    subtree: true
                  });
                }, 0);
              })();
            `,
          }}
        />
      </head>
      <body className="font-sans antialiased" suppressHydrationWarning>
        <Providers>
          <div className="min-h-screen flex">
            <div className="flex-1 flex flex-col">
              <Header />
              <main className="flex-1" suppressHydrationWarning>
                {children}
              </main>
              <Footer />
            </div>
          </div>
          <HydrationBoundary>
            <EnhancedShoppingCart />
            <LiveChatWidget />
          </HydrationBoundary>
        </Providers>
      </body>
    </html>
  );
}
