'use client'

import { useState } from 'react'
import Link from 'next/link'
import { motion } from 'framer-motion'
import { 
  CheckCircleIcon, 
  ClockIcon, 
  CurrencyDollarIcon,
  ArrowLeftIcon,
  PencilIcon,
  DocumentTextIcon,
  GlobeAltIcon,
  UserGroupIcon,
  EnvelopeIcon,
  MagnifyingGlassIcon,
  ChatBubbleLeftRightIcon,
  NewspaperIcon,
  BookOpenIcon,
  SpeakerWaveIcon,
  CameraIcon,
  TagIcon
} from '@heroicons/react/24/outline'
import { useCartStore } from '@/store/cart'

export default function ContentWritingPage() {
  const addToCart = useCartStore((state) => state.addToCart)
  const openCart = useCartStore((state) => state.openCart)

  const handleAddToCart = () => {
    addToCart({
      id: 5,
      title: 'Content Writing',
      price: 500,
      quantity: 1
    })
    
    setTimeout(() => {
      openCart()
    }, 100)
  }

  const mainServices = [
    {
      icon: DocumentTextIcon,
      title: 'Website Content',
      description: 'Compelling website copy that converts visitors into customers with clear messaging and strong calls-to-action.'
    },
    {
      icon: NewspaperIcon,
      title: 'Blog Writing',
      description: 'SEO-optimized blog posts that drive traffic, establish authority, and engage your target audience.'
    },
    {
      icon: MagnifyingGlassIcon,
      title: 'SEO Content',
      description: 'Search engine optimized content that ranks well and drives organic traffic to your website.'
    },
    {
      icon: EnvelopeIcon,
      title: 'Email Marketing',
      description: 'Persuasive email campaigns that nurture leads and drive conversions through strategic messaging.'
    }
  ]

  const contentTypes = [
    { name: 'Website Copy', icon: GlobeAltIcon, description: 'Homepage, about, services pages' },
    { name: 'Blog Articles', icon: NewspaperIcon, description: 'SEO-optimized blog content' },
    { name: 'Product Descriptions', icon: TagIcon, description: 'E-commerce product copy' },
    { name: 'Email Campaigns', icon: EnvelopeIcon, description: 'Marketing email sequences' },
    { name: 'Social Media', icon: UserGroupIcon, description: 'Posts and captions' },
    { name: 'Technical Writing', icon: DocumentTextIcon, description: 'Manuals and documentation' }
  ]

  const packages = [
    {
      name: 'Basic Content',
      price: 300,
      description: 'Essential content for small businesses',
      features: [
        'Up to 5 web pages',
        '2 blog posts per month',
        'Basic SEO optimization',
        'Content strategy consultation',
        'Email support',
        '48-hour turnaround'
      ],
      popular: false
    },
    {
      name: 'Professional Content',
      price: 750,
      description: 'Comprehensive content for growing businesses',
      features: [
        'Up to 15 web pages',
        '8 blog posts per month',
        'Advanced SEO optimization',
        'Content calendar planning',
        'Social media content',
        'Email campaign copy',
        '24-hour turnaround'
      ],
      popular: true
    },
    {
      name: 'Enterprise Content',
      price: 1500,
      description: 'Full content marketing for large organizations',
      features: [
        'Unlimited web pages',
        '20+ blog posts per month',
        'Technical documentation',
        'Content marketing strategy',
        'Multi-channel content',
        'Dedicated content manager',
        'Same-day turnaround'
      ],
      popular: false
    }
  ]

  const industries = [
    'Technology & Software',
    'Healthcare & Medical',
    'Finance & Banking',
    'E-commerce & Retail',
    'Real Estate',
    'Legal Services',
    'Education',
    'Manufacturing',
    'Travel & Hospitality',
    'Professional Services'
  ]

  const writingProcess = [
    'Research & Discovery',
    'Content Strategy',
    'Writing & Creation',
    'Review & Optimization',
    'Publishing & Delivery',
    'Performance Analysis'
  ]

  const faqs = [
    {
      question: 'How much does content writing cost?',
      answer: 'Content writing costs vary based on the type and volume of content needed. Our packages start at $300/month for basic content and can go up to $1,500+/month for enterprise-level content marketing. We also offer per-project pricing for one-time needs.'
    },
    {
      question: 'Do you write content for all industries?',
      answer: 'Yes, our experienced writers cover a wide range of industries including technology, healthcare, finance, e-commerce, and more. We research your industry thoroughly to ensure accurate, relevant, and engaging content that resonates with your target audience.'
    },
    {
      question: 'How do you ensure content is SEO-optimized?',
      answer: 'We conduct keyword research, optimize content structure, use proper headings, include meta descriptions, and follow SEO best practices. Our content is designed to rank well in search engines while remaining engaging and valuable for readers.'
    },
    {
      question: 'What is your content writing process?',
      answer: 'Our process includes research and discovery, content strategy development, writing and creation, review and optimization, publishing assistance, and performance analysis. We work closely with you throughout to ensure the content meets your goals.'
    },
    {
      question: 'Can you help with content strategy and planning?',
      answer: 'Absolutely! We provide comprehensive content strategy services including content calendars, topic planning, audience analysis, and content marketing strategies that align with your business objectives and drive measurable results.'
    }
  ]

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Hero Section */}
      <div className="bg-gradient-to-br from-green-600 to-teal-700 text-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          <div className="flex items-center gap-4 mb-6">
            <Link
              href="/services"
              className="flex items-center gap-2 text-green-100 hover:text-white transition-colors"
            >
              <ArrowLeftIcon className="h-5 w-5" />
              Back to Services
            </Link>
          </div>
          
          <div className="text-center max-w-4xl mx-auto">
            <div className="flex justify-center mb-6">
              <div className="p-4 bg-white/10 rounded-2xl backdrop-blur-sm">
                <PencilIcon className="h-12 w-12 text-white" />
              </div>
            </div>
            
            <h1 className="text-4xl md:text-6xl font-bold mb-6">
              Content Writing
            </h1>
            <p className="text-xl md:text-2xl text-green-100 mb-8 leading-relaxed">
              Engaging content that converts and ranks
            </p>
            <p className="text-lg text-green-100 mb-8">
              From website copy to blog posts, we create compelling content that drives traffic, 
              engages your audience, and converts visitors into customers with strategic messaging and SEO optimization.
            </p>
            
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <button
                onClick={handleAddToCart}
                className="bg-white text-green-600 px-8 py-4 rounded-lg font-semibold hover:bg-green-50 transition-colors"
              >
                Start Project - From $300/month
              </button>
              <Link
                href="/contact"
                className="border-2 border-white text-white px-8 py-4 rounded-lg font-semibold hover:bg-white hover:text-green-600 transition-colors"
              >
                Get Free Content Audit
              </Link>
            </div>
            
            <div className="mt-8 text-green-100">
              <p className="text-lg font-semibold">1000+ Content Pieces Created</p>
            </div>
          </div>
        </div>
      </div>

      {/* Main Services Overview */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-16">
          <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
            Professional Content Writing Services
          </h2>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            Strategic content that drives results across all channels
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
          {mainServices.map((service, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: index * 0.1 }}
              className="bg-white rounded-xl shadow-lg p-6 hover:shadow-xl transition-shadow text-center"
            >
              <div className="p-3 bg-green-100 rounded-lg w-fit mx-auto mb-4">
                <service.icon className="h-8 w-8 text-green-600" />
              </div>
              <h3 className="text-xl font-semibold text-gray-900 mb-3">
                {service.title}
              </h3>
              <p className="text-gray-600">
                {service.description}
              </p>
            </motion.div>
          ))}
        </div>
      </div>

      {/* Content Types */}
      <div className="bg-white py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Types of Content We Create
            </h2>
            <p className="text-xl text-gray-600">
              Comprehensive content solutions for every business need
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {contentTypes.map((type, index) => (
              <div key={index} className="bg-gray-50 rounded-xl p-6 hover:shadow-lg transition-shadow">
                <div className="flex items-center gap-4 mb-3">
                  <div className="p-2 bg-green-100 rounded-lg">
                    <type.icon className="h-6 w-6 text-green-600" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900">{type.name}</h3>
                </div>
                <p className="text-gray-600">{type.description}</p>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Package Pricing */}
      <div className="bg-gray-100 py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Content Writing Packages
            </h2>
            <p className="text-xl text-gray-600">
              Choose the perfect content plan for your business
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            {packages.map((pkg, index) => (
              <motion.div
                key={index}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
                className={`rounded-2xl p-8 ${
                  pkg.popular
                    ? 'bg-green-600 text-white shadow-2xl scale-105'
                    : 'bg-white shadow-lg'
                }`}
              >
                {pkg.popular && (
                  <div className="text-center mb-4">
                    <span className="bg-white text-green-600 px-4 py-1 rounded-full text-sm font-semibold">
                      Most Popular
                    </span>
                  </div>
                )}
                
                <div className="text-center mb-6">
                  <h3 className={`text-2xl font-bold mb-2 ${pkg.popular ? 'text-white' : 'text-gray-900'}`}>
                    {pkg.name}
                  </h3>
                  <p className={`mb-4 ${pkg.popular ? 'text-green-100' : 'text-gray-600'}`}>
                    {pkg.description}
                  </p>
                  <div className="text-4xl font-bold mb-2">
                    ${pkg.price}
                  </div>
                  <div className={`text-sm ${pkg.popular ? 'text-green-100' : 'text-gray-600'}`}>
                    per month
                  </div>
                </div>

                <ul className="space-y-3 mb-8">
                  {pkg.features.map((feature, featureIndex) => (
                    <li key={featureIndex} className="flex items-center gap-3">
                      <CheckCircleIcon className={`h-5 w-5 ${pkg.popular ? 'text-white' : 'text-green-500'}`} />
                      <span className={pkg.popular ? 'text-green-100' : 'text-gray-600'}>
                        {feature}
                      </span>
                    </li>
                  ))}
                </ul>

                <button
                  onClick={handleAddToCart}
                  className={`w-full py-3 px-6 rounded-lg font-semibold transition-colors ${
                    pkg.popular
                      ? 'bg-white text-green-600 hover:bg-green-50'
                      : 'bg-green-600 text-white hover:bg-green-700'
                  }`}
                >
                  Get Started
                </button>
              </motion.div>
            ))}
          </div>
        </div>
      </div>

      {/* Industries We Serve */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-12">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">
            Industries We Serve
          </h2>
          <p className="text-xl text-gray-600">
            Expert content writing across diverse sectors
          </p>
        </div>

        <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-5 gap-4">
          {industries.map((industry, index) => (
            <div key={index} className="bg-white p-4 rounded-lg shadow-sm text-center hover:shadow-md transition-shadow">
              <span className="text-gray-700 font-medium">{industry}</span>
            </div>
          ))}
        </div>
      </div>

      {/* Writing Process */}
      <div className="bg-white py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Our Content Writing Process
            </h2>
            <p className="text-xl text-gray-600">
              Strategic approach to creating compelling content
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-6 gap-6">
            {writingProcess.map((step, index) => (
              <div key={index} className="text-center">
                <div className="w-16 h-16 bg-green-600 text-white rounded-full flex items-center justify-center font-bold text-xl mx-auto mb-4">
                  {index + 1}
                </div>
                <h3 className="font-semibold text-gray-900 mb-2">{step}</h3>
                <div className="text-sm text-gray-600">
                  {index === 0 && "Understanding your audience and goals"}
                  {index === 1 && "Developing content strategy and calendar"}
                  {index === 2 && "Creating engaging, SEO-optimized content"}
                  {index === 3 && "Review, editing, and optimization"}
                  {index === 4 && "Publishing and distribution support"}
                  {index === 5 && "Tracking performance and improvements"}
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Features & Benefits */}
      <div className="bg-gray-100 py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
            <div>
              <h2 className="text-3xl font-bold text-gray-900 mb-6">
                Why Choose Our Content Writing?
              </h2>
              <div className="space-y-4">
                <div className="flex items-start gap-4">
                  <CheckCircleIcon className="h-6 w-6 text-green-600 mt-1" />
                  <div>
                    <h3 className="font-semibold text-gray-900">SEO Optimized</h3>
                    <p className="text-gray-600">Content designed to rank well and drive organic traffic</p>
                  </div>
                </div>
                <div className="flex items-start gap-4">
                  <CheckCircleIcon className="h-6 w-6 text-green-600 mt-1" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Industry Expertise</h3>
                    <p className="text-gray-600">Writers with deep knowledge across various sectors</p>
                  </div>
                </div>
                <div className="flex items-start gap-4">
                  <CheckCircleIcon className="h-6 w-6 text-green-600 mt-1" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Fast Turnaround</h3>
                    <p className="text-gray-600">Quick delivery without compromising quality</p>
                  </div>
                </div>
                <div className="flex items-start gap-4">
                  <CheckCircleIcon className="h-6 w-6 text-green-600 mt-1" />
                  <div>
                    <h3 className="font-semibold text-gray-900">Conversion Focused</h3>
                    <p className="text-gray-600">Content that drives action and business results</p>
                  </div>
                </div>
              </div>
            </div>
            
            <div className="bg-green-50 rounded-2xl p-8">
              <h3 className="text-2xl font-bold text-gray-900 mb-6">Content Performance</h3>
              <div className="space-y-4">
                <div className="bg-white p-4 rounded-lg shadow-sm">
                  <div className="flex justify-between items-center">
                    <span className="text-gray-600">Average Traffic Increase</span>
                    <span className="font-bold text-green-600">150%</span>
                  </div>
                </div>
                <div className="bg-white p-4 rounded-lg shadow-sm">
                  <div className="flex justify-between items-center">
                    <span className="text-gray-600">Engagement Rate</span>
                    <span className="font-bold text-green-600">85%</span>
                  </div>
                </div>
                <div className="bg-white p-4 rounded-lg shadow-sm">
                  <div className="flex justify-between items-center">
                    <span className="text-gray-600">Conversion Improvement</span>
                    <span className="font-bold text-green-600">75%</span>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* FAQ Section */}
      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-12">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">
            Frequently Asked Questions
          </h2>
          <p className="text-xl text-gray-600">
            Everything you need to know about our content writing services
          </p>
        </div>

        <div className="space-y-6">
          {faqs.map((faq, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: index * 0.1 }}
              className="bg-white rounded-xl shadow-lg p-6"
            >
              <h3 className="text-lg font-semibold text-gray-900 mb-3">
                {faq.question}
              </h3>
              <p className="text-gray-600">
                {faq.answer}
              </p>
            </motion.div>
          ))}
        </div>
      </div>

      {/* CTA Section */}
      <div className="bg-gradient-to-r from-green-600 to-teal-600 text-white py-16">
        <div className="max-w-4xl mx-auto text-center px-4 sm:px-6 lg:px-8">
          <h2 className="text-3xl md:text-4xl font-bold mb-4">
            Ready to Elevate Your Content?
          </h2>
          <p className="text-xl text-green-100 mb-8">
            Let our expert writers create content that drives traffic and converts visitors into customers.
          </p>
          
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <button
              onClick={handleAddToCart}
              className="bg-white text-green-600 px-8 py-4 rounded-lg font-semibold hover:bg-green-50 transition-colors"
            >
              Start Writing - From $300/month
            </button>
            <Link
              href="/contact"
              className="border-2 border-white text-white px-8 py-4 rounded-lg font-semibold hover:bg-white hover:text-green-600 transition-colors"
            >
              Get Free Content Audit
            </Link>
          </div>
          
          <div className="mt-8 flex items-center justify-center gap-8 text-green-100">
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>SEO Optimized</span>
            </div>
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>Fast Delivery</span>
            </div>
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>Expert Writers</span>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}