'use client'

import { useState } from 'react'
import Link from 'next/link'
import { motion } from 'framer-motion'
import { 
  CheckCircleIcon, 
  ClockIcon, 
  CurrencyDollarIcon,
  ArrowLeftIcon,
  MegaphoneIcon,
  ChartBarIcon,
  GlobeAltIcon,
  UserGroupIcon,
  EnvelopeIcon,
  DevicePhoneMobileIcon,
  CursorArrowRaysIcon,
  PresentationChartLineIcon,
  TagIcon,
  SpeakerWaveIcon,
  CameraIcon,
  DocumentTextIcon,
  MagnifyingGlassIcon,
  HeartIcon
} from '@heroicons/react/24/outline'
import { useCartStore } from '@/store/cart'

export default function DigitalMarketingPage() {
  const addToCart = useCartStore((state) => state.addToCart)
  const openCart = useCartStore((state) => state.openCart)

  const handleAddToCart = () => {
    addToCart({
      id: 4,
      title: 'Digital Marketing',
      price: 1500,
      quantity: 1
    })
    
    setTimeout(() => {
      openCart()
    }, 100)
  }

  const mainServices = [
    {
      icon: MagnifyingGlassIcon,
      title: 'Search Engine Optimization',
      description: 'Improve your website\'s visibility and ranking on search engines to drive organic traffic and increase online presence.'
    },
    {
      icon: CursorArrowRaysIcon,
      title: 'Pay-Per-Click Advertising',
      description: 'Strategic PPC campaigns on Google Ads and social media platforms to generate immediate traffic and conversions.'
    },
    {
      icon: UserGroupIcon,
      title: 'Social Media Marketing',
      description: 'Engage your audience across all social platforms with compelling content and targeted advertising strategies.'
    },
    {
      icon: EnvelopeIcon,
      title: 'Email Marketing',
      description: 'Build relationships and drive sales through personalized email campaigns and automated marketing sequences.'
    }
  ]

  const digitalChannels = [
    {
      name: 'Google Ads',
      description: 'Search and display advertising',
      icon: MagnifyingGlassIcon,
      color: 'bg-blue-500'
    },
    {
      name: 'Facebook & Instagram',
      description: 'Social media advertising',
      icon: HeartIcon,
      color: 'bg-pink-500'
    },
    {
      name: 'LinkedIn',
      description: 'Professional networking ads',
      icon: UserGroupIcon,
      color: 'bg-indigo-500'
    },
    {
      name: 'YouTube',
      description: 'Video advertising campaigns',
      icon: SpeakerWaveIcon,
      color: 'bg-red-500'
    },
    {
      name: 'Email Platforms',
      description: 'Newsletter and automation',
      icon: EnvelopeIcon,
      color: 'bg-green-500'
    },
    {
      name: 'Content Marketing',
      description: 'Blog and content strategy',
      icon: DocumentTextIcon,
      color: 'bg-purple-500'
    }
  ]

  const packages = [
    {
      name: 'Starter Marketing',
      price: 800,
      description: 'Essential digital marketing for small businesses',
      features: [
        'SEO audit and optimization',
        'Google My Business setup',
        'Social media account setup',
        'Basic content creation',
        'Monthly performance report',
        'Email support'
      ],
      popular: false
    },
    {
      name: 'Growth Marketing',
      price: 1500,
      description: 'Comprehensive marketing for growing businesses',
      features: [
        'Multi-channel campaigns',
        'Google Ads management',
        'Social media advertising',
        'Email marketing automation',
        'Content marketing strategy',
        'Weekly performance reports',
        'Phone & email support'
      ],
      popular: true
    },
    {
      name: 'Enterprise Marketing',
      price: 3000,
      description: 'Full-scale marketing for large organizations',
      features: [
        'Custom marketing strategy',
        'Advanced campaign management',
        'Marketing automation',
        'CRM integration',
        'Advanced analytics',
        'Dedicated account manager',
        '24/7 priority support'
      ],
      popular: false
    }
  ]

  const marketingServices = [
    {
      icon: MagnifyingGlassIcon,
      title: 'Search Engine Optimization (SEO)',
      description: 'Comprehensive SEO strategies to improve your website\'s organic search visibility and drive qualified traffic.',
      details: 'Our SEO services include keyword research, on-page optimization, technical SEO, content strategy, and link building. We focus on sustainable, white-hat techniques that improve your search rankings and drive long-term organic growth.',
      results: ['Higher search rankings', 'Increased organic traffic', 'Better user experience']
    },
    {
      icon: CursorArrowRaysIcon,
      title: 'Pay-Per-Click (PPC) Advertising',
      description: 'Strategic PPC campaigns that maximize your return on investment and drive immediate, measurable results.',
      details: 'We create and manage high-converting PPC campaigns across Google Ads, Facebook, Instagram, and LinkedIn. Our data-driven approach includes keyword research, ad copy optimization, landing page design, and continuous campaign refinement.',
      results: ['Immediate traffic generation', 'Higher conversion rates', 'Measurable ROI']
    },
    {
      icon: UserGroupIcon,
      title: 'Social Media Marketing',
      description: 'Build brand awareness and engage your audience across all major social media platforms.',
      details: 'Our social media strategies include content creation, community management, influencer partnerships, and paid social advertising. We help you build meaningful relationships with your audience and convert followers into customers.',
      results: ['Increased brand awareness', 'Higher engagement rates', 'More qualified leads']
    },
    {
      icon: EnvelopeIcon,
      title: 'Email Marketing & Automation',
      description: 'Nurture leads and retain customers with personalized email campaigns and automated sequences.',
      details: 'We design and implement email marketing strategies that include welcome sequences, newsletter campaigns, product promotions, and automated nurture flows. Our approach focuses on personalization and segmentation for maximum engagement.',
      results: ['Higher email open rates', 'Increased customer retention', 'Better lifetime value']
    }
  ]

  const metrics = [
    { label: 'Average ROI Increase', value: '300%', icon: ChartBarIcon },
    { label: 'Traffic Growth', value: '150%', icon: PresentationChartLineIcon },
    { label: 'Lead Generation', value: '250%', icon: UserGroupIcon },
    { label: 'Conversion Rate', value: '85%', icon: CursorArrowRaysIcon }
  ]

  const faqs = [
    {
      question: 'How much does digital marketing cost?',
      answer: 'Digital marketing costs vary based on your goals and budget. Our packages start at $800/month for basic services and can go up to $3,000+/month for comprehensive enterprise solutions. We work with you to create a strategy that fits your budget and delivers results.'
    },
    {
      question: 'How long does it take to see results from digital marketing?',
      answer: 'Results vary by channel. PPC campaigns can generate immediate traffic, while SEO typically takes 3-6 months to show significant results. Social media and email marketing usually show engagement improvements within the first month, with conversion improvements following soon after.'
    },
    {
      question: 'Which digital marketing channels are best for my business?',
      answer: 'The best channels depend on your target audience, business goals, and budget. We conduct a thorough analysis of your market and competition to recommend the most effective mix of SEO, PPC, social media, and email marketing for your specific situation.'
    },
    {
      question: 'Do you provide detailed reporting and analytics?',
      answer: 'Yes, we provide comprehensive reporting that includes traffic metrics, conversion data, ROI analysis, and actionable insights. Our reports are customized to your business goals and delivered monthly, with weekly updates available for higher-tier packages.'
    },
    {
      question: 'Can you work with our existing marketing team?',
      answer: 'Absolutely! We collaborate seamlessly with internal marketing teams, providing specialized expertise and additional resources. We can serve as an extension of your team or provide consulting and training to enhance your current capabilities.'
    }
  ]

  const process = [
    'Strategy Development',
    'Campaign Setup',
    'Content Creation',
    'Launch & Optimize',
    'Monitor & Analyze',
    'Scale & Grow'
  ]

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Hero Section */}
      <div className="bg-gradient-to-br from-orange-600 to-red-700 text-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          <div className="flex items-center gap-4 mb-6">
            <Link
              href="/services"
              className="flex items-center gap-2 text-orange-100 hover:text-white transition-colors"
            >
              <ArrowLeftIcon className="h-5 w-5" />
              Back to Services
            </Link>
          </div>
          
          <div className="text-center max-w-4xl mx-auto">
            <div className="flex justify-center mb-6">
              <div className="p-4 bg-white/10 rounded-2xl backdrop-blur-sm">
                <MegaphoneIcon className="h-12 w-12 text-white" />
              </div>
            </div>
            
            <h1 className="text-4xl md:text-6xl font-bold mb-6">
              Digital Marketing
            </h1>
            <p className="text-xl md:text-2xl text-orange-100 mb-8 leading-relaxed">
              Grow your business with data-driven digital marketing strategies
            </p>
            <p className="text-lg text-orange-100 mb-8">
              From SEO and PPC to social media and email marketing, we create comprehensive 
              digital strategies that drive traffic, generate leads, and increase revenue for your business.
            </p>
            
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <button
                onClick={handleAddToCart}
                className="bg-white text-orange-600 px-8 py-4 rounded-lg font-semibold hover:bg-orange-50 transition-colors"
              >
                Start Campaign - From $800/month
              </button>
              <Link
                href="/contact"
                className="border-2 border-white text-white px-8 py-4 rounded-lg font-semibold hover:bg-white hover:text-orange-600 transition-colors"
              >
                Get Free Strategy Call
              </Link>
            </div>
            
            <div className="mt-8 text-orange-100">
              <p className="text-lg font-semibold">500+ Successful Campaigns Delivered</p>
            </div>
          </div>
        </div>
      </div>

      {/* Performance Metrics */}
      <div className="bg-white py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Proven Digital Marketing Results
            </h2>
            <p className="text-xl text-gray-600">
              Our data-driven approach delivers measurable business growth
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
            {metrics.map((metric, index) => (
              <motion.div
                key={index}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
                className="text-center bg-orange-50 rounded-xl p-6"
              >
                <div className="p-3 bg-orange-100 rounded-lg w-fit mx-auto mb-4">
                  <metric.icon className="h-8 w-8 text-orange-600" />
                </div>
                <div className="text-3xl font-bold text-gray-900 mb-2">{metric.value}</div>
                <div className="text-gray-600">{metric.label}</div>
              </motion.div>
            ))}
          </div>
        </div>
      </div>

      {/* Main Services Overview */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-16">
          <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
            Complete Digital Marketing Solutions
          </h2>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            Comprehensive marketing services to grow your online presence
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
          {mainServices.map((service, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: index * 0.1 }}
              className="bg-white rounded-xl shadow-lg p-6 hover:shadow-xl transition-shadow text-center"
            >
              <div className="p-3 bg-orange-100 rounded-lg w-fit mx-auto mb-4">
                <service.icon className="h-8 w-8 text-orange-600" />
              </div>
              <h3 className="text-xl font-semibold text-gray-900 mb-3">
                {service.title}
              </h3>
              <p className="text-gray-600">
                {service.description}
              </p>
            </motion.div>
          ))}
        </div>
      </div>

      {/* Digital Channels */}
      <div className="bg-gray-100 py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Digital Marketing Channels
            </h2>
            <p className="text-xl text-gray-600">
              We leverage all major digital platforms to maximize your reach
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {digitalChannels.map((channel, index) => (
              <div key={index} className="bg-white rounded-xl p-6 shadow-lg hover:shadow-xl transition-shadow">
                <div className={`w-12 h-12 ${channel.color} rounded-lg flex items-center justify-center mb-4`}>
                  <channel.icon className="h-6 w-6 text-white" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 mb-2">{channel.name}</h3>
                <p className="text-gray-600">{channel.description}</p>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Package Pricing */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-12">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">
            Digital Marketing Packages
          </h2>
          <p className="text-xl text-gray-600">
            Choose the perfect marketing plan for your business growth
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
          {packages.map((pkg, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: index * 0.1 }}
              className={`rounded-2xl p-8 ${
                pkg.popular
                  ? 'bg-orange-600 text-white shadow-2xl scale-105'
                  : 'bg-white shadow-lg'
              }`}
            >
              {pkg.popular && (
                <div className="text-center mb-4">
                  <span className="bg-white text-orange-600 px-4 py-1 rounded-full text-sm font-semibold">
                    Most Popular
                  </span>
                </div>
              )}
              
              <div className="text-center mb-6">
                <h3 className={`text-2xl font-bold mb-2 ${pkg.popular ? 'text-white' : 'text-gray-900'}`}>
                  {pkg.name}
                </h3>
                <p className={`mb-4 ${pkg.popular ? 'text-orange-100' : 'text-gray-600'}`}>
                  {pkg.description}
                </p>
                <div className="text-4xl font-bold mb-2">
                  ${pkg.price}
                </div>
                <div className={`text-sm ${pkg.popular ? 'text-orange-100' : 'text-gray-600'}`}>
                  per month
                </div>
              </div>

              <ul className="space-y-3 mb-8">
                {pkg.features.map((feature, featureIndex) => (
                  <li key={featureIndex} className="flex items-center gap-3">
                    <CheckCircleIcon className={`h-5 w-5 ${pkg.popular ? 'text-white' : 'text-green-500'}`} />
                    <span className={pkg.popular ? 'text-orange-100' : 'text-gray-600'}>
                      {feature}
                    </span>
                  </li>
                ))}
              </ul>

              <button
                onClick={handleAddToCart}
                className={`w-full py-3 px-6 rounded-lg font-semibold transition-colors ${
                  pkg.popular
                    ? 'bg-white text-orange-600 hover:bg-orange-50'
                    : 'bg-orange-600 text-white hover:bg-orange-700'
                }`}
              >
                Get Started
              </button>
            </motion.div>
          ))}
        </div>
      </div>

      {/* Detailed Marketing Services */}
      <div className="bg-white py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
              Our Digital Marketing Expertise
            </h2>
            <p className="text-xl text-gray-600">
              Comprehensive strategies that drive measurable results
            </p>
          </div>

          <div className="space-y-16">
            {marketingServices.map((service, index) => (
              <motion.div
                key={index}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
                className={`grid grid-cols-1 lg:grid-cols-2 gap-12 items-center ${
                  index % 2 === 1 ? 'lg:grid-flow-col-dense' : ''
                }`}
              >
                <div className={index % 2 === 1 ? 'lg:col-start-2' : ''}>
                  <div className="flex items-center gap-4 mb-4">
                    <div className="p-3 bg-orange-100 rounded-lg">
                      <service.icon className="h-8 w-8 text-orange-600" />
                    </div>
                    <h3 className="text-2xl font-bold text-gray-900">{service.title}</h3>
                  </div>
                  <p className="text-lg text-gray-600 mb-4">{service.description}</p>
                  <p className="text-gray-600 mb-6">{service.details}</p>
                  
                  <div className="space-y-2">
                    <h4 className="font-semibold text-gray-900">Key Results:</h4>
                    {service.results.map((result, resultIndex) => (
                      <div key={resultIndex} className="flex items-center gap-2">
                        <CheckCircleIcon className="h-5 w-5 text-green-500" />
                        <span className="text-gray-600">{result}</span>
                      </div>
                    ))}
                  </div>
                </div>
                
                <div className={`bg-gradient-to-br ${
                  index % 2 === 0 
                    ? 'from-orange-50 to-red-50' 
                    : 'from-red-50 to-orange-50'
                } rounded-2xl p-8 ${index % 2 === 1 ? 'lg:col-start-1' : ''}`}>
                  <div className="text-center">
                    <service.icon className={`h-16 w-16 mx-auto mb-6 ${
                      index % 2 === 0 ? 'text-orange-600' : 'text-red-600'
                    }`} />
                    <div className="space-y-4">
                      <div className="bg-white p-4 rounded-lg shadow-sm">
                        <div className="font-semibold text-gray-900">Data-Driven Approach</div>
                        <div className="text-sm text-gray-600">Analytics and optimization focused</div>
                      </div>
                      <div className="bg-white p-4 rounded-lg shadow-sm">
                        <div className="font-semibold text-gray-900">Measurable Results</div>
                        <div className="text-sm text-gray-600">Clear ROI and performance metrics</div>
                      </div>
                    </div>
                  </div>
                </div>
              </motion.div>
            ))}
          </div>
        </div>
      </div>

      {/* Process */}
      <div className="bg-gray-100 py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Our Marketing Process
            </h2>
            <p className="text-xl text-gray-600">
              Strategic approach to digital marketing success
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-6 gap-6">
            {process.map((step, index) => (
              <div key={index} className="text-center">
                <div className="w-16 h-16 bg-orange-600 text-white rounded-full flex items-center justify-center font-bold text-xl mx-auto mb-4">
                  {index + 1}
                </div>
                <h3 className="font-semibold text-gray-900 mb-2">{step}</h3>
                <div className="text-sm text-gray-600">
                  {index === 0 && "Market research and strategy planning"}
                  {index === 1 && "Campaign setup and configuration"}
                  {index === 2 && "Content development and creative assets"}
                  {index === 3 && "Campaign launch and optimization"}
                  {index === 4 && "Performance tracking and reporting"}
                  {index === 5 && "Scaling successful campaigns"}
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* FAQ Section */}
      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-12">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">
            Frequently Asked Questions
          </h2>
          <p className="text-xl text-gray-600">
            Everything you need to know about digital marketing
          </p>
        </div>

        <div className="space-y-6">
          {faqs.map((faq, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: index * 0.1 }}
              className="bg-white rounded-xl shadow-lg p-6"
            >
              <h3 className="text-lg font-semibold text-gray-900 mb-3">
                {faq.question}
              </h3>
              <p className="text-gray-600">
                {faq.answer}
              </p>
            </motion.div>
          ))}
        </div>
      </div>

      {/* CTA Section */}
      <div className="bg-gradient-to-r from-orange-600 to-red-600 text-white py-16">
        <div className="max-w-4xl mx-auto text-center px-4 sm:px-6 lg:px-8">
          <h2 className="text-3xl md:text-4xl font-bold mb-4">
            Ready to Accelerate Your Growth?
          </h2>
          <p className="text-xl text-orange-100 mb-8">
            Let our digital marketing experts create a strategy that drives real business results.
          </p>
          
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <button
              onClick={handleAddToCart}
              className="bg-white text-orange-600 px-8 py-4 rounded-lg font-semibold hover:bg-orange-50 transition-colors"
            >
              Start Marketing - From $800/month
            </button>
            <Link
              href="/contact"
              className="border-2 border-white text-white px-8 py-4 rounded-lg font-semibold hover:bg-white hover:text-orange-600 transition-colors"
            >
              Get Free Strategy Session
            </Link>
          </div>
          
          <div className="mt-8 flex items-center justify-center gap-8 text-orange-100">
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>Proven Results</span>
            </div>
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>Data-Driven</span>
            </div>
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>Full Transparency</span>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}