'use client'

import { useState } from 'react'
import Link from 'next/link'
import { motion } from 'framer-motion'
import { 
  CheckCircleIcon, 
  ClockIcon, 
  CurrencyDollarIcon,
  ArrowLeftIcon,
  ShoppingBagIcon,
  CreditCardIcon,
  ChartBarIcon,
  ShieldCheckIcon,
  GlobeAltIcon,
  DevicePhoneMobileIcon,
  TruckIcon,
  UserGroupIcon,
  BuildingStorefrontIcon,
  CogIcon,
  CloudIcon,
  LockClosedIcon,
  BanknotesIcon,
  TagIcon
} from '@heroicons/react/24/outline'
import { useCartStore } from '@/store/cart'

export default function EcommerceDevelopmentPage() {
  const addToCart = useCartStore((state) => state.addToCart)
  const openCart = useCartStore((state) => state.openCart)

  const handleAddToCart = () => {
    addToCart({
      id: 2,
      title: 'E-commerce Development',
      price: 3500,
      quantity: 1
    })
    
    setTimeout(() => {
      openCart()
    }, 100)
  }

  const mainServices = [
    {
      icon: BuildingStorefrontIcon,
      title: 'Custom Online Stores',
      description: 'Build fully customized e-commerce platforms tailored to your business needs with advanced features and seamless user experience.'
    },
    {
      icon: CreditCardIcon,
      title: 'Payment Integration',
      description: 'Secure payment gateway integration supporting multiple payment methods including credit cards, PayPal, Stripe, and digital wallets.'
    },
    {
      icon: TruckIcon,
      title: 'Inventory Management',
      description: 'Comprehensive inventory tracking, order management, and automated shipping solutions to streamline your operations.'
    },
    {
      icon: DevicePhoneMobileIcon,
      title: 'Mobile Commerce',
      description: 'Mobile-optimized shopping experiences with responsive design and native mobile app development for iOS and Android.'
    }
  ]

  const platforms = [
    { name: 'Shopify', description: 'Complete e-commerce solution', popular: true },
    { name: 'WooCommerce', description: 'WordPress-based flexibility', popular: true },
    { name: 'Magento', description: 'Enterprise-grade platform', popular: false },
    { name: 'Custom Development', description: 'Tailored solutions', popular: true },
    { name: 'BigCommerce', description: 'Scalable SaaS platform', popular: false },
    { name: 'PrestaShop', description: 'Open-source solution', popular: false }
  ]

  const features = [
    'Custom Store Design & Development',
    'Product Catalog Management',
    'Shopping Cart & Checkout',
    'Payment Gateway Integration',
    'Order Management System',
    'Inventory Tracking',
    'Customer Account Management',
    'Email Marketing Integration',
    'SEO Optimization',
    'Analytics & Reporting',
    'Mobile Responsive Design',
    'Security & SSL Certificates',
    'Multi-language Support',
    'Currency Conversion',
    'Shipping Calculator',
    'Tax Management',
    'Coupon & Discount System',
    'Review & Rating System'
  ]

  const detailedServices = [
    {
      icon: ShoppingBagIcon,
      title: 'Complete E-commerce Store Development',
      description: 'From concept to launch, we create fully functional online stores that drive sales and provide exceptional user experiences.',
      details: 'Our comprehensive approach includes market analysis, user experience design, custom development, payment integration, and performance optimization. We ensure your store is ready to compete in today\'s digital marketplace.',
      highlight: 'Full-service solution from design to deployment'
    },
    {
      icon: CreditCardIcon,
      title: 'Payment Gateway & Security',
      description: 'Secure payment processing with multiple gateway options, fraud protection, and PCI compliance for customer trust and safety.',
      details: 'We integrate leading payment providers like Stripe, PayPal, Square, and regional payment methods. Our security implementations include SSL certificates, fraud detection, and secure customer data handling.',
      highlight: 'PCI DSS compliant with advanced fraud protection'
    },
    {
      icon: ChartBarIcon,
      title: 'Analytics & Performance Optimization',
      description: 'Advanced analytics, conversion tracking, and performance optimization to maximize your store\'s revenue potential.',
      details: 'We implement comprehensive tracking systems, A/B testing capabilities, and performance monitoring. Our optimization strategies focus on improving conversion rates, reducing cart abandonment, and enhancing user engagement.',
      highlight: 'Data-driven optimization for maximum conversions'
    },
    {
      icon: CloudIcon,
      title: 'Cloud Hosting & Scalability',
      description: 'Reliable cloud hosting solutions that scale with your business growth, ensuring fast loading times and 99.9% uptime.',
      details: 'Our hosting solutions include CDN integration, automated backups, security monitoring, and scalable infrastructure that grows with your business. We ensure your store can handle traffic spikes during peak sales periods.',
      highlight: '99.9% uptime guarantee with global CDN'
    }
  ]

  const packages = [
    {
      name: 'Starter Store',
      price: 2500,
      description: 'Perfect for small businesses starting online',
      features: [
        'Up to 50 products',
        'Basic payment integration',
        'Mobile responsive design',
        'SEO optimization',
        'SSL certificate',
        '1 month support'
      ],
      popular: false
    },
    {
      name: 'Professional Store',
      price: 5000,
      description: 'Complete solution for growing businesses',
      features: [
        'Unlimited products',
        'Multiple payment gateways',
        'Advanced inventory management',
        'Customer accounts',
        'Email marketing integration',
        'Analytics dashboard',
        '3 months support'
      ],
      popular: true
    },
    {
      name: 'Enterprise Store',
      price: 10000,
      description: 'Custom solution for large-scale operations',
      features: [
        'Custom development',
        'ERP integration',
        'Multi-store management',
        'Advanced reporting',
        'API development',
        'Dedicated support',
        '6 months support'
      ],
      popular: false
    }
  ]

  const technologies = [
    { name: 'React/Next.js', category: 'Frontend' },
    { name: 'Node.js', category: 'Backend' },
    { name: 'Shopify', category: 'Platform' },
    { name: 'WooCommerce', category: 'Platform' },
    { name: 'Stripe', category: 'Payment' },
    { name: 'PayPal', category: 'Payment' },
    { name: 'MongoDB', category: 'Database' },
    { name: 'PostgreSQL', category: 'Database' }
  ]

  const faqs = [
    {
      question: 'How much does e-commerce development cost?',
      answer: 'E-commerce development costs vary based on complexity and features. Our packages start at $2,500 for basic stores and can go up to $10,000+ for enterprise solutions with custom features and integrations.'
    },
    {
      question: 'Which e-commerce platform is best for my business?',
      answer: 'The best platform depends on your specific needs. Shopify is great for quick setup, WooCommerce offers flexibility with WordPress, and custom development provides unlimited possibilities. We help you choose based on your requirements and budget.'
    },
    {
      question: 'How long does it take to develop an e-commerce store?',
      answer: 'Development time ranges from 4-6 weeks for basic stores to 12-16 weeks for complex custom solutions. We provide detailed timelines during the planning phase and keep you updated throughout the process.'
    },
    {
      question: 'Do you provide ongoing maintenance and support?',
      answer: 'Yes, we offer comprehensive maintenance packages including security updates, performance monitoring, backup management, and technical support. Our support plans range from basic monthly maintenance to dedicated enterprise support.'
    },
    {
      question: 'Can you integrate with existing business systems?',
      answer: 'Absolutely! We specialize in integrating e-commerce stores with existing ERP systems, inventory management, accounting software, and other business tools to streamline your operations and reduce manual work.'
    }
  ]

  const process = [
    'Requirements Analysis',
    'Platform Selection',
    'Design & Prototyping',
    'Development & Integration',
    'Testing & Optimization',
    'Launch & Support'
  ]

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Hero Section */}
      <div className="bg-gradient-to-br from-purple-600 to-indigo-800 text-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          <div className="flex items-center gap-4 mb-6">
            <Link
              href="/services"
              className="flex items-center gap-2 text-purple-100 hover:text-white transition-colors"
            >
              <ArrowLeftIcon className="h-5 w-5" />
              Back to Services
            </Link>
          </div>
          
          <div className="text-center max-w-4xl mx-auto">
            <div className="flex justify-center mb-6">
              <div className="p-4 bg-white/10 rounded-2xl backdrop-blur-sm">
                <ShoppingBagIcon className="h-12 w-12 text-white" />
              </div>
            </div>
            
            <h1 className="text-4xl md:text-6xl font-bold mb-6">
              E-commerce Development
            </h1>
            <p className="text-xl md:text-2xl text-purple-100 mb-8 leading-relaxed">
              Build powerful online stores that drive sales and grow your business
            </p>
            <p className="text-lg text-purple-100 mb-8">
              From custom e-commerce platforms to marketplace integrations, we create comprehensive 
              online shopping experiences that convert visitors into customers and maximize your revenue potential.
            </p>
            
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <button
                onClick={handleAddToCart}
                className="bg-white text-purple-600 px-8 py-4 rounded-lg font-semibold hover:bg-purple-50 transition-colors"
              >
                Start Project - From $2,500
              </button>
              <Link
                href="/contact"
                className="border-2 border-white text-white px-8 py-4 rounded-lg font-semibold hover:bg-white hover:text-purple-600 transition-colors"
              >
                Get Free Consultation
              </Link>
            </div>
            
            <div className="mt-8 text-purple-100">
              <p className="text-lg font-semibold">Trusted by 200+ Online Businesses</p>
            </div>
          </div>
        </div>
      </div>

      {/* Main Services Overview */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-16">
          <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
            Complete E-commerce Solutions
          </h2>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            Everything you need to sell online successfully
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
          {mainServices.map((service, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: index * 0.1 }}
              className="bg-white rounded-xl shadow-lg p-6 hover:shadow-xl transition-shadow text-center"
            >
              <div className="p-3 bg-purple-100 rounded-lg w-fit mx-auto mb-4">
                <service.icon className="h-8 w-8 text-purple-600" />
              </div>
              <h3 className="text-xl font-semibold text-gray-900 mb-3">
                {service.title}
              </h3>
              <p className="text-gray-600">
                {service.description}
              </p>
            </motion.div>
          ))}
        </div>
      </div>

      {/* E-commerce Platforms */}
      <div className="bg-white py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              E-commerce Platforms We Work With
            </h2>
            <p className="text-xl text-gray-600">
              Choose the right platform for your business needs
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {platforms.map((platform, index) => (
              <div
                key={index}
                className={`p-6 rounded-xl border-2 transition-all ${
                  platform.popular
                    ? 'border-purple-200 bg-purple-50 shadow-lg'
                    : 'border-gray-200 bg-white hover:shadow-md'
                }`}
              >
                <div className="flex items-center justify-between mb-3">
                  <h3 className="text-lg font-semibold text-gray-900">{platform.name}</h3>
                  {platform.popular && (
                    <span className="bg-purple-600 text-white text-xs px-2 py-1 rounded-full">
                      Popular
                    </span>
                  )}
                </div>
                <p className="text-gray-600">{platform.description}</p>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Package Pricing */}
      <div className="bg-gray-100 py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              E-commerce Development Packages
            </h2>
            <p className="text-xl text-gray-600">
              Choose the perfect package for your business size and needs
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            {packages.map((pkg, index) => (
              <motion.div
                key={index}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
                className={`rounded-2xl p-8 ${
                  pkg.popular
                    ? 'bg-purple-600 text-white shadow-2xl scale-105'
                    : 'bg-white shadow-lg'
                }`}
              >
                {pkg.popular && (
                  <div className="text-center mb-4">
                    <span className="bg-white text-purple-600 px-4 py-1 rounded-full text-sm font-semibold">
                      Most Popular
                    </span>
                  </div>
                )}
                
                <div className="text-center mb-6">
                  <h3 className={`text-2xl font-bold mb-2 ${pkg.popular ? 'text-white' : 'text-gray-900'}`}>
                    {pkg.name}
                  </h3>
                  <p className={`mb-4 ${pkg.popular ? 'text-purple-100' : 'text-gray-600'}`}>
                    {pkg.description}
                  </p>
                  <div className="text-4xl font-bold mb-2">
                    ${pkg.price.toLocaleString()}
                  </div>
                  <div className={`text-sm ${pkg.popular ? 'text-purple-100' : 'text-gray-600'}`}>
                    One-time payment
                  </div>
                </div>

                <ul className="space-y-3 mb-8">
                  {pkg.features.map((feature, featureIndex) => (
                    <li key={featureIndex} className="flex items-center gap-3">
                      <CheckCircleIcon className={`h-5 w-5 ${pkg.popular ? 'text-white' : 'text-green-500'}`} />
                      <span className={pkg.popular ? 'text-purple-100' : 'text-gray-600'}>
                        {feature}
                      </span>
                    </li>
                  ))}
                </ul>

                <button
                  onClick={handleAddToCart}
                  className={`w-full py-3 px-6 rounded-lg font-semibold transition-colors ${
                    pkg.popular
                      ? 'bg-white text-purple-600 hover:bg-purple-50'
                      : 'bg-purple-600 text-white hover:bg-purple-700'
                  }`}
                >
                  Get Started
                </button>
              </motion.div>
            ))}
          </div>
        </div>
      </div>

      {/* Detailed Services */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-16">
          <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
            Our E-commerce Expertise
          </h2>
          <p className="text-xl text-gray-600">
            Comprehensive services for online retail success
          </p>
        </div>

        <div className="space-y-16">
          {detailedServices.map((service, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: index * 0.1 }}
              className={`grid grid-cols-1 lg:grid-cols-2 gap-12 items-center ${
                index % 2 === 1 ? 'lg:grid-flow-col-dense' : ''
              }`}
            >
              <div className={index % 2 === 1 ? 'lg:col-start-2' : ''}>
                <div className="flex items-center gap-4 mb-4">
                  <div className="p-3 bg-purple-100 rounded-lg">
                    <service.icon className="h-8 w-8 text-purple-600" />
                  </div>
                  <h3 className="text-2xl font-bold text-gray-900">{service.title}</h3>
                </div>
                <p className="text-lg text-gray-600 mb-4">{service.description}</p>
                <p className="text-gray-600 mb-4">{service.details}</p>
                <div className="bg-purple-50 p-4 rounded-lg">
                  <div className="flex items-center gap-2">
                    <ShieldCheckIcon className="h-5 w-5 text-purple-600" />
                    <span className="font-semibold text-purple-800">{service.highlight}</span>
                  </div>
                </div>
              </div>
              
              <div className={`bg-gradient-to-br ${
                index % 2 === 0 
                  ? 'from-purple-50 to-indigo-50' 
                  : 'from-indigo-50 to-purple-50'
              } rounded-2xl p-8 ${index % 2 === 1 ? 'lg:col-start-1' : ''}`}>
                <div className="text-center">
                  <service.icon className={`h-16 w-16 mx-auto mb-4 ${
                    index % 2 === 0 ? 'text-purple-600' : 'text-indigo-600'
                  }`} />
                  <div className="grid grid-cols-2 gap-3">
                    {technologies.slice(index * 2, index * 2 + 4).map((tech, techIndex) => (
                      <div key={techIndex} className="bg-white p-3 rounded-lg shadow-sm text-center">
                        <div className="font-medium text-gray-900 text-sm">{tech.name}</div>
                        <div className="text-xs text-gray-600">{tech.category}</div>
                      </div>
                    ))}
                  </div>
                </div>
              </div>
            </motion.div>
          ))}
        </div>
      </div>

      {/* Features Grid */}
      <div className="bg-white py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Complete E-commerce Features
            </h2>
            <p className="text-xl text-gray-600">
              Everything your online store needs to succeed
            </p>
          </div>

          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
            {features.map((feature, index) => (
              <div key={index} className="flex items-center gap-3 p-3 bg-gray-50 rounded-lg">
                <div className="w-2 h-2 bg-purple-500 rounded-full flex-shrink-0"></div>
                <span className="text-gray-700 text-sm">{feature}</span>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Process */}
      <div className="bg-gray-100 py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Our Development Process
            </h2>
            <p className="text-xl text-gray-600">
              From concept to successful online store
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-6 gap-6">
            {process.map((step, index) => (
              <div key={index} className="text-center">
                <div className="w-16 h-16 bg-purple-600 text-white rounded-full flex items-center justify-center font-bold text-xl mx-auto mb-4">
                  {index + 1}
                </div>
                <h3 className="font-semibold text-gray-900 mb-2">{step}</h3>
                <div className="text-sm text-gray-600">
                  {index === 0 && "Understanding your products and market"}
                  {index === 1 && "Choosing the best e-commerce platform"}
                  {index === 2 && "Creating user-friendly store design"}
                  {index === 3 && "Building features and integrations"}
                  {index === 4 && "Quality assurance and optimization"}
                  {index === 5 && "Launch and ongoing support"}
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* FAQ Section */}
      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-12">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">
            Frequently Asked Questions
          </h2>
          <p className="text-xl text-gray-600">
            Everything you need to know about e-commerce development
          </p>
        </div>

        <div className="space-y-6">
          {faqs.map((faq, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: index * 0.1 }}
              className="bg-white rounded-xl shadow-lg p-6"
            >
              <h3 className="text-lg font-semibold text-gray-900 mb-3">
                {faq.question}
              </h3>
              <p className="text-gray-600">
                {faq.answer}
              </p>
            </motion.div>
          ))}
        </div>
      </div>

      {/* CTA Section */}
      <div className="bg-gradient-to-r from-purple-600 to-indigo-600 text-white py-16">
        <div className="max-w-4xl mx-auto text-center px-4 sm:px-6 lg:px-8">
          <h2 className="text-3xl md:text-4xl font-bold mb-4">
            Ready to Launch Your Online Store?
          </h2>
          <p className="text-xl text-purple-100 mb-8">
            Start selling online with a professional e-commerce store that converts visitors into customers.
          </p>
          
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <button
              onClick={handleAddToCart}
              className="bg-white text-purple-600 px-8 py-4 rounded-lg font-semibold hover:bg-purple-50 transition-colors"
            >
              Start Your Store - From $2,500
            </button>
            <Link
              href="/contact"
              className="border-2 border-white text-white px-8 py-4 rounded-lg font-semibold hover:bg-white hover:text-purple-600 transition-colors"
            >
              Get Free Quote
            </Link>
          </div>
          
          <div className="mt-8 flex items-center justify-center gap-8 text-purple-100">
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>Secure Payments</span>
            </div>
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>Mobile Optimized</span>
            </div>
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>SEO Ready</span>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}