'use client'

import { useState } from 'react'
import Link from 'next/link'
import { motion } from 'framer-motion'
import { 
  CheckCircleIcon, 
  ClockIcon, 
  CurrencyDollarIcon,
  ArrowLeftIcon,
  PaintBrushIcon,
  PhotoIcon,
  RectangleGroupIcon,
  UserGroupIcon,
  DocumentTextIcon,
  PrinterIcon,
  GlobeAltIcon,
  DevicePhoneMobileIcon,
  TagIcon,
  PresentationChartLineIcon,
  SwatchIcon,
  CubeIcon
} from '@heroicons/react/24/outline'
import { useCartStore } from '@/store/cart'

export default function GraphicDesignPage() {
  const addToCart = useCartStore((state) => state.addToCart)
  const openCart = useCartStore((state) => state.openCart)

  const handleAddToCart = () => {
    addToCart({
      id: 6,
      title: 'Graphic Design',
      price: 400,
      quantity: 1
    })
    
    setTimeout(() => {
      openCart()
    }, 100)
  }

  const mainServices = [
    {
      icon: SwatchIcon,
      title: 'Brand Identity Design',
      description: 'Complete brand identity packages including logo design, color palettes, typography, and brand guidelines.'
    },
    {
      icon: GlobeAltIcon,
      title: 'Web Design',
      description: 'Modern, responsive website designs that look great and provide excellent user experience across all devices.'
    },
    {
      icon: PrinterIcon,
      title: 'Print Design',
      description: 'Professional print materials including brochures, flyers, business cards, and marketing collateral.'
    },
    {
      icon: DevicePhoneMobileIcon,
      title: 'Digital Graphics',
      description: 'Social media graphics, banner ads, infographics, and digital marketing materials for online campaigns.'
    }
  ]

  const designTypes = [
    { name: 'Logo Design', icon: SwatchIcon, description: 'Professional logo creation' },
    { name: 'Business Cards', icon: DocumentTextIcon, description: 'Professional business cards' },
    { name: 'Brochures', icon: PrinterIcon, description: 'Marketing brochures and flyers' },
    { name: 'Website Design', icon: GlobeAltIcon, description: 'Modern web design layouts' },
    { name: 'Social Media', icon: UserGroupIcon, description: 'Social media graphics' },
    { name: 'Packaging', icon: CubeIcon, description: 'Product packaging design' },
    { name: 'Infographics', icon: PresentationChartLineIcon, description: 'Data visualization design' },
    { name: 'Brand Guidelines', icon: RectangleGroupIcon, description: 'Complete brand standards' }
  ]

  const packages = [
    {
      name: 'Starter Design',
      price: 250,
      description: 'Essential design for new businesses',
      features: [
        'Logo design (3 concepts)',
        'Business card design',
        'Basic brand guidelines',
        'Social media templates (5)',
        '2 revisions included',
        '7-day delivery'
      ],
      popular: false
    },
    {
      name: 'Professional Design',
      price: 600,
      description: 'Complete design package for growing businesses',
      features: [
        'Complete brand identity',
        'Logo + variations',
        'Stationery package',
        'Website design mockups',
        'Social media kit (20 templates)',
        'Marketing materials',
        'Unlimited revisions',
        '5-day delivery'
      ],
      popular: true
    },
    {
      name: 'Enterprise Design',
      price: 1200,
      description: 'Comprehensive design for large organizations',
      features: [
        'Full brand identity system',
        'Advanced logo package',
        'Complete stationery suite',
        'Website + mobile design',
        'Marketing campaign materials',
        'Packaging design',
        'Brand guidelines document',
        'Dedicated designer',
        '3-day delivery'
      ],
      popular: false
    }
  ]

  const designProcess = [
    'Brief & Discovery',
    'Research & Concepts',
    'Design Creation',
    'Client Review',
    'Revisions & Refinement',
    'Final Delivery'
  ]

  const industries = [
    'Technology',
    'Healthcare',
    'Finance',
    'Retail',
    'Real Estate',
    'Food & Beverage',
    'Education',
    'Non-Profit',
    'Professional Services',
    'Entertainment'
  ]

  const faqs = [
    {
      question: 'How much does graphic design cost?',
      answer: 'Graphic design costs vary based on the scope and complexity of the project. Our packages start at $250 for basic design needs and can go up to $1,200+ for comprehensive brand identity systems. We also offer hourly rates for specific design tasks.'
    },
    {
      question: 'How long does a design project take?',
      answer: 'Project timelines depend on complexity and scope. Simple logos can be completed in 3-5 days, while complete brand identity packages may take 1-2 weeks. We provide detailed timelines for each project during the initial consultation.'
    },
    {
      question: 'Do you provide source files and different formats?',
      answer: 'Yes, we provide all source files in various formats including AI, PSD, PDF, PNG, JPG, and SVG. You\'ll receive high-resolution files suitable for both print and digital use, ensuring you have everything needed for any application.'
    },
    {
      question: 'How many revisions are included?',
      answer: 'Our packages include different revision levels - Starter includes 2 revisions, Professional includes unlimited revisions, and Enterprise includes unlimited revisions with dedicated support. Additional revisions beyond the package limit are available at an hourly rate.'
    },
    {
      question: 'Can you work with our existing brand guidelines?',
      answer: 'Absolutely! We can work within your existing brand guidelines to create consistent designs, or help update and modernize your current brand identity while maintaining brand recognition and equity.'
    }
  ]

  const tools = [
    'Adobe Illustrator',
    'Adobe Photoshop',
    'Adobe InDesign',
    'Figma',
    'Sketch',
    'Adobe XD'
  ]

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Hero Section */}
      <div className="bg-gradient-to-br from-pink-600 to-purple-700 text-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          <div className="flex items-center gap-4 mb-6">
            <Link
              href="/services"
              className="flex items-center gap-2 text-pink-100 hover:text-white transition-colors"
            >
              <ArrowLeftIcon className="h-5 w-5" />
              Back to Services
            </Link>
          </div>
          
          <div className="text-center max-w-4xl mx-auto">
            <div className="flex justify-center mb-6">
              <div className="p-4 bg-white/10 rounded-2xl backdrop-blur-sm">
                <PaintBrushIcon className="h-12 w-12 text-white" />
              </div>
            </div>
            
            <h1 className="text-4xl md:text-6xl font-bold mb-6">
              Graphic Design
            </h1>
            <p className="text-xl md:text-2xl text-pink-100 mb-8 leading-relaxed">
              Creative designs that make your brand stand out
            </p>
            <p className="text-lg text-pink-100 mb-8">
              From logos and branding to web design and marketing materials, we create visually stunning 
              designs that capture your brand essence and engage your audience across all touchpoints.
            </p>
            
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <button
                onClick={handleAddToCart}
                className="bg-white text-pink-600 px-8 py-4 rounded-lg font-semibold hover:bg-pink-50 transition-colors"
              >
                Start Project - From $250
              </button>
              <Link
                href="/contact"
                className="border-2 border-white text-white px-8 py-4 rounded-lg font-semibold hover:bg-white hover:text-pink-600 transition-colors"
              >
                Get Free Design Consultation
              </Link>
            </div>
            
            <div className="mt-8 text-pink-100">
              <p className="text-lg font-semibold">2000+ Designs Created for Happy Clients</p>
            </div>
          </div>
        </div>
      </div>

      {/* Main Services Overview */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-16">
          <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
            Complete Graphic Design Services
          </h2>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            Professional design solutions for all your visual communication needs
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
          {mainServices.map((service, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: index * 0.1 }}
              className="bg-white rounded-xl shadow-lg p-6 hover:shadow-xl transition-shadow text-center"
            >
              <div className="p-3 bg-pink-100 rounded-lg w-fit mx-auto mb-4">
                <service.icon className="h-8 w-8 text-pink-600" />
              </div>
              <h3 className="text-xl font-semibold text-gray-900 mb-3">
                {service.title}
              </h3>
              <p className="text-gray-600">
                {service.description}
              </p>
            </motion.div>
          ))}
        </div>
      </div>

      {/* Design Types */}
      <div className="bg-white py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Types of Design We Create
            </h2>
            <p className="text-xl text-gray-600">
              Comprehensive design solutions across all mediums
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            {designTypes.map((type, index) => (
              <div key={index} className="bg-gray-50 rounded-xl p-6 hover:shadow-lg transition-shadow text-center">
                <div className="p-3 bg-pink-100 rounded-lg w-fit mx-auto mb-4">
                  <type.icon className="h-8 w-8 text-pink-600" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 mb-2">{type.name}</h3>
                <p className="text-gray-600 text-sm">{type.description}</p>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Package Pricing */}
      <div className="bg-gray-100 py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Graphic Design Packages
            </h2>
            <p className="text-xl text-gray-600">
              Choose the perfect design package for your needs
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
            {packages.map((pkg, index) => (
              <motion.div
                key={index}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
                className={`rounded-2xl p-8 ${
                  pkg.popular
                    ? 'bg-pink-600 text-white shadow-2xl scale-105'
                    : 'bg-white shadow-lg'
                }`}
              >
                {pkg.popular && (
                  <div className="text-center mb-4">
                    <span className="bg-white text-pink-600 px-4 py-1 rounded-full text-sm font-semibold">
                      Most Popular
                    </span>
                  </div>
                )}
                
                <div className="text-center mb-6">
                  <h3 className={`text-2xl font-bold mb-2 ${pkg.popular ? 'text-white' : 'text-gray-900'}`}>
                    {pkg.name}
                  </h3>
                  <p className={`mb-4 ${pkg.popular ? 'text-pink-100' : 'text-gray-600'}`}>
                    {pkg.description}
                  </p>
                  <div className="text-4xl font-bold mb-2">
                    ${pkg.price}
                  </div>
                  <div className={`text-sm ${pkg.popular ? 'text-pink-100' : 'text-gray-600'}`}>
                    one-time payment
                  </div>
                </div>

                <ul className="space-y-3 mb-8">
                  {pkg.features.map((feature, featureIndex) => (
                    <li key={featureIndex} className="flex items-center gap-3">
                      <CheckCircleIcon className={`h-5 w-5 ${pkg.popular ? 'text-white' : 'text-green-500'}`} />
                      <span className={pkg.popular ? 'text-pink-100' : 'text-gray-600'}>
                        {feature}
                      </span>
                    </li>
                  ))}
                </ul>

                <button
                  onClick={handleAddToCart}
                  className={`w-full py-3 px-6 rounded-lg font-semibold transition-colors ${
                    pkg.popular
                      ? 'bg-white text-pink-600 hover:bg-pink-50'
                      : 'bg-pink-600 text-white hover:bg-pink-700'
                  }`}
                >
                  Get Started
                </button>
              </motion.div>
            ))}
          </div>
        </div>
      </div>

      {/* Design Process */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-12">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">
            Our Design Process
          </h2>
          <p className="text-xl text-gray-600">
            Strategic approach to creating impactful designs
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-6 gap-6">
          {designProcess.map((step, index) => (
            <div key={index} className="text-center">
              <div className="w-16 h-16 bg-pink-600 text-white rounded-full flex items-center justify-center font-bold text-xl mx-auto mb-4">
                {index + 1}
              </div>
              <h3 className="font-semibold text-gray-900 mb-2">{step}</h3>
              <div className="text-sm text-gray-600">
                {index === 0 && "Understanding your needs and goals"}
                {index === 1 && "Market research and creative concepts"}
                {index === 2 && "Designing multiple creative options"}
                {index === 3 && "Presenting designs for feedback"}
                {index === 4 && "Refining based on your input"}
                {index === 5 && "Delivering final files and assets"}
              </div>
            </div>
          ))}
        </div>
      </div>

      {/* Tools & Software */}
      <div className="bg-white py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Professional Design Tools
            </h2>
            <p className="text-xl text-gray-600">
              We use industry-standard software for professional results
            </p>
          </div>

          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-6">
            {tools.map((tool, index) => (
              <div key={index} className="bg-gray-50 p-6 rounded-xl text-center hover:shadow-lg transition-shadow">
                <div className="p-3 bg-pink-100 rounded-lg w-fit mx-auto mb-3">
                  <PhotoIcon className="h-8 w-8 text-pink-600" />
                </div>
                <h3 className="font-semibold text-gray-900">{tool}</h3>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Industries */}
      <div className="bg-gray-100 py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Industries We Design For
            </h2>
            <p className="text-xl text-gray-600">
              Creative solutions across diverse business sectors
            </p>
          </div>

          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-5 gap-4">
            {industries.map((industry, index) => (
              <div key={index} className="bg-white p-4 rounded-lg shadow-sm text-center hover:shadow-md transition-shadow">
                <span className="text-gray-700 font-medium">{industry}</span>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Features & Benefits */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
          <div>
            <h2 className="text-3xl font-bold text-gray-900 mb-6">
              Why Choose Our Design Services?
            </h2>
            <div className="space-y-4">
              <div className="flex items-start gap-4">
                <CheckCircleIcon className="h-6 w-6 text-pink-600 mt-1" />
                <div>
                  <h3 className="font-semibold text-gray-900">Creative Excellence</h3>
                  <p className="text-gray-600">Award-winning designs that stand out in the market</p>
                </div>
              </div>
              <div className="flex items-start gap-4">
                <CheckCircleIcon className="h-6 w-6 text-pink-600 mt-1" />
                <div>
                  <h3 className="font-semibold text-gray-900">Brand Consistency</h3>
                  <p className="text-gray-600">Cohesive design across all brand touchpoints</p>
                </div>
              </div>
              <div className="flex items-start gap-4">
                <CheckCircleIcon className="h-6 w-6 text-pink-600 mt-1" />
                <div>
                  <h3 className="font-semibold text-gray-900">Fast Turnaround</h3>
                  <p className="text-gray-600">Quick delivery without compromising quality</p>
                </div>
              </div>
              <div className="flex items-start gap-4">
                <CheckCircleIcon className="h-6 w-6 text-pink-600 mt-1" />
                <div>
                  <h3 className="font-semibold text-gray-900">Multiple Formats</h3>
                  <p className="text-gray-600">All file formats for print and digital use</p>
                </div>
              </div>
            </div>
          </div>
          
          <div className="bg-pink-50 rounded-2xl p-8">
            <h3 className="text-2xl font-bold text-gray-900 mb-6">Design Impact</h3>
            <div className="space-y-4">
              <div className="bg-white p-4 rounded-lg shadow-sm">
                <div className="flex justify-between items-center">
                  <span className="text-gray-600">Brand Recognition Increase</span>
                  <span className="font-bold text-pink-600">200%</span>
                </div>
              </div>
              <div className="bg-white p-4 rounded-lg shadow-sm">
                <div className="flex justify-between items-center">
                  <span className="text-gray-600">Client Satisfaction Rate</span>
                  <span className="font-bold text-pink-600">98%</span>
                </div>
              </div>
              <div className="bg-white p-4 rounded-lg shadow-sm">
                <div className="flex justify-between items-center">
                  <span className="text-gray-600">Project Success Rate</span>
                  <span className="font-bold text-pink-600">95%</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* FAQ Section */}
      <div className="bg-white py-16">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Frequently Asked Questions
            </h2>
            <p className="text-xl text-gray-600">
              Everything you need to know about our graphic design services
            </p>
          </div>

          <div className="space-y-6">
            {faqs.map((faq, index) => (
              <motion.div
                key={index}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
                className="bg-gray-50 rounded-xl shadow-lg p-6"
              >
                <h3 className="text-lg font-semibold text-gray-900 mb-3">
                  {faq.question}
                </h3>
                <p className="text-gray-600">
                  {faq.answer}
                </p>
              </motion.div>
            ))}
          </div>
        </div>
      </div>

      {/* CTA Section */}
      <div className="bg-gradient-to-r from-pink-600 to-purple-600 text-white py-16">
        <div className="max-w-4xl mx-auto text-center px-4 sm:px-6 lg:px-8">
          <h2 className="text-3xl md:text-4xl font-bold mb-4">
            Ready to Create Amazing Designs?
          </h2>
          <p className="text-xl text-pink-100 mb-8">
            Let our creative team bring your vision to life with stunning graphic design.
          </p>
          
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <button
              onClick={handleAddToCart}
              className="bg-white text-pink-600 px-8 py-4 rounded-lg font-semibold hover:bg-pink-50 transition-colors"
            >
              Start Design Project - From $250
            </button>
            <Link
              href="/contact"
              className="border-2 border-white text-white px-8 py-4 rounded-lg font-semibold hover:bg-white hover:text-pink-600 transition-colors"
            >
              Get Free Design Consultation
            </Link>
          </div>
          
          <div className="mt-8 flex items-center justify-center gap-8 text-pink-100">
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>Creative Excellence</span>
            </div>
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>Fast Delivery</span>
            </div>
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>All File Formats</span>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}