'use client'

import { useState } from 'react'
import Link from 'next/link'
import { motion } from 'framer-motion'
import { 
  CheckCircleIcon, 
  ClockIcon, 
  CurrencyDollarIcon,
  ArrowLeftIcon,
  DevicePhoneMobileIcon,
  CloudIcon,
  CogIcon,
  ShieldCheckIcon,
  RocketLaunchIcon,
  UserGroupIcon,
  ChartBarIcon,
  LockClosedIcon,
  BellIcon,
  MapPinIcon,
  CameraIcon,
  SpeakerWaveIcon,
  GlobeAltIcon,
  WifiIcon,
  ShoppingCartIcon,
  WrenchScrewdriverIcon
} from '@heroicons/react/24/outline'
import { useCartStore } from '@/store/cart'

export default function MobileAppDevelopmentPage() {
  const addToCart = useCartStore((state) => state.addToCart)
  const openCart = useCartStore((state) => state.openCart)

  const handleAddToCart = () => {
    addToCart({
      id: 3,
      title: 'Mobile App Development',
      price: 4500,
      quantity: 1
    })
    
    setTimeout(() => {
      openCart()
    }, 100)
  }

  const mainServices = [
    {
      icon: DevicePhoneMobileIcon,
      title: 'Native App Development',
      description: 'Build high-performance native applications for iOS and Android using platform-specific technologies for optimal user experience.'
    },
    {
      icon: GlobeAltIcon,
      title: 'Cross-Platform Apps',
      description: 'Develop apps that work seamlessly across multiple platforms using React Native and Flutter for cost-effective solutions.'
    },
    {
      icon: CloudIcon,
      title: 'Backend & API Integration',
      description: 'Robust backend development with secure APIs, cloud integration, and real-time data synchronization.'
    },
    {
      icon: RocketLaunchIcon,
      title: 'App Store Deployment',
      description: 'Complete app store submission process including optimization, compliance, and ongoing app store management.'
    }
  ]

  const appTypes = [
    {
      title: 'Business Apps',
      description: 'Productivity and business process applications',
      icon: CogIcon,
      examples: ['CRM Systems', 'Project Management', 'Team Collaboration']
    },
    {
      title: 'E-commerce Apps',
      description: 'Mobile shopping and marketplace applications',
      icon: ShoppingCartIcon,
      examples: ['Online Stores', 'Marketplace Apps', 'Payment Solutions']
    },
    {
      title: 'Social Apps',
      description: 'Community and social networking applications',
      icon: UserGroupIcon,
      examples: ['Social Networks', 'Chat Apps', 'Community Platforms']
    },
    {
      title: 'Utility Apps',
      description: 'Practical tools and utility applications',
      icon: WrenchScrewdriverIcon,
      examples: ['Calculators', 'Converters', 'Productivity Tools']
    }
  ]

  const platforms = [
    { name: 'iOS (Swift)', description: 'Native iPhone & iPad apps', color: 'bg-blue-500' },
    { name: 'Android (Kotlin)', description: 'Native Android applications', color: 'bg-green-500' },
    { name: 'React Native', description: 'Cross-platform development', color: 'bg-purple-500' },
    { name: 'Flutter', description: 'Google\'s UI toolkit', color: 'bg-cyan-500' }
  ]

  const features = [
    'Custom UI/UX Design',
    'Push Notifications',
    'Offline Functionality',
    'GPS & Location Services',
    'Camera Integration',
    'Social Media Integration',
    'Payment Gateway Integration',
    'Real-time Chat',
    'Cloud Storage Integration',
    'Biometric Authentication',
    'Analytics & Reporting',
    'Multi-language Support',
    'In-app Purchases',
    'AR/VR Capabilities',
    'IoT Device Integration',
    'API Development'
  ]

  const detailedServices = [
    {
      icon: DevicePhoneMobileIcon,
      title: 'Native Mobile App Development',
      description: 'Platform-specific development for iOS and Android ensuring optimal performance and user experience.',
      details: 'Our native development approach leverages platform-specific features and capabilities, resulting in faster, more responsive apps with better integration with device hardware and operating system features.',
      technologies: ['Swift', 'Kotlin', 'Xcode', 'Android Studio']
    },
    {
      icon: GlobeAltIcon,
      title: 'Cross-Platform Development',
      description: 'Cost-effective solutions using React Native and Flutter to reach both iOS and Android users.',
      details: 'Cross-platform development allows you to maintain a single codebase while targeting multiple platforms, reducing development time and costs while ensuring consistent user experience across devices.',
      technologies: ['React Native', 'Flutter', 'Dart', 'JavaScript']
    },
    {
      icon: CloudIcon,
      title: 'Backend Development & APIs',
      description: 'Scalable backend infrastructure with secure APIs and cloud integration for your mobile applications.',
      details: 'We build robust backend systems that support your mobile app with features like user authentication, data storage, push notifications, and real-time synchronization across devices.',
      technologies: ['Node.js', 'Firebase', 'AWS', 'MongoDB']
    },
    {
      icon: ShieldCheckIcon,
      title: 'App Security & Performance',
      description: 'Implementation of advanced security measures and performance optimization for enterprise-grade applications.',
      details: 'Our security-first approach includes data encryption, secure authentication, API security, and compliance with industry standards to protect user data and ensure app reliability.',
      technologies: ['OAuth', 'SSL/TLS', 'Encryption', 'Security Audits']
    }
  ]

  const packages = [
    {
      name: 'Basic App',
      price: 3500,
      timeline: '6-8 weeks',
      description: 'Simple app with core functionality',
      features: [
        'Single platform (iOS or Android)',
        'Up to 5 screens',
        'Basic UI/UX design',
        'API integration',
        'App store submission',
        '1 month support'
      ],
      popular: false
    },
    {
      name: 'Standard App',
      price: 6500,
      timeline: '10-12 weeks',
      description: 'Feature-rich app for growing businesses',
      features: [
        'Cross-platform (iOS & Android)',
        'Up to 15 screens',
        'Custom UI/UX design',
        'Backend development',
        'Push notifications',
        'Analytics integration',
        '3 months support'
      ],
      popular: true
    },
    {
      name: 'Premium App',
      price: 12000,
      timeline: '16-20 weeks',
      description: 'Enterprise-grade application',
      features: [
        'Native development',
        'Unlimited screens',
        'Advanced features',
        'Third-party integrations',
        'Real-time functionality',
        'Admin dashboard',
        '6 months support'
      ],
      popular: false
    }
  ]

  const faqs = [
    {
      question: 'How much does mobile app development cost?',
      answer: 'Mobile app development costs vary based on complexity, features, and platforms. Our packages start at $3,500 for basic apps and can go up to $12,000+ for enterprise applications with advanced features.'
    },
    {
      question: 'Should I choose native or cross-platform development?',
      answer: 'Native development offers the best performance and platform-specific features, while cross-platform development is more cost-effective and faster to market. We help you choose based on your requirements, budget, and timeline.'
    },
    {
      question: 'How long does it take to develop a mobile app?',
      answer: 'Development time depends on app complexity. Simple apps take 6-8 weeks, while complex enterprise applications can take 16-20 weeks. We provide detailed timelines during the planning phase.'
    },
    {
      question: 'Do you help with app store submission?',
      answer: 'Yes, we handle the complete app store submission process including App Store and Google Play Store requirements, optimization, and approval process. We ensure your app meets all guidelines for successful publication.'
    },
    {
      question: 'Can you maintain and update my app after launch?',
      answer: 'Absolutely! We offer comprehensive maintenance packages including bug fixes, security updates, feature enhancements, and OS compatibility updates to keep your app running smoothly.'
    }
  ]

  const process = [
    'Strategy & Planning',
    'UI/UX Design',
    'Development',
    'Testing & QA',
    'Deployment',
    'Support & Updates'
  ]

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Hero Section */}
      <div className="bg-gradient-to-br from-indigo-600 to-purple-800 text-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
          <div className="flex items-center gap-4 mb-6">
            <Link
              href="/services"
              className="flex items-center gap-2 text-indigo-100 hover:text-white transition-colors"
            >
              <ArrowLeftIcon className="h-5 w-5" />
              Back to Services
            </Link>
          </div>
          
          <div className="text-center max-w-4xl mx-auto">
            <div className="flex justify-center mb-6">
              <div className="p-4 bg-white/10 rounded-2xl backdrop-blur-sm">
                <DevicePhoneMobileIcon className="h-12 w-12 text-white" />
              </div>
            </div>
            
            <h1 className="text-4xl md:text-6xl font-bold mb-6">
              Mobile App Development
            </h1>
            <p className="text-xl md:text-2xl text-indigo-100 mb-8 leading-relaxed">
              Transform your ideas into powerful mobile applications
            </p>
            <p className="text-lg text-indigo-100 mb-8">
              From concept to app store, we create stunning mobile applications for iOS and Android 
              that engage users and drive business growth with cutting-edge technology and exceptional user experience.
            </p>
            
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <button
                onClick={handleAddToCart}
                className="bg-white text-indigo-600 px-8 py-4 rounded-lg font-semibold hover:bg-indigo-50 transition-colors"
              >
                Start Project - From $3,500
              </button>
              <Link
                href="/contact"
                className="border-2 border-white text-white px-8 py-4 rounded-lg font-semibold hover:bg-white hover:text-indigo-600 transition-colors"
              >
                Get Free Consultation
              </Link>
            </div>
            
            <div className="mt-8 text-indigo-100">
              <p className="text-lg font-semibold">100+ Apps Delivered Successfully</p>
            </div>
          </div>
        </div>
      </div>

      {/* Main Services Overview */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-16">
          <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
            Complete Mobile App Solutions
          </h2>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            End-to-end mobile app development services for all platforms
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
          {mainServices.map((service, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: index * 0.1 }}
              className="bg-white rounded-xl shadow-lg p-6 hover:shadow-xl transition-shadow text-center"
            >
              <div className="p-3 bg-indigo-100 rounded-lg w-fit mx-auto mb-4">
                <service.icon className="h-8 w-8 text-indigo-600" />
              </div>
              <h3 className="text-xl font-semibold text-gray-900 mb-3">
                {service.title}
              </h3>
              <p className="text-gray-600">
                {service.description}
              </p>
            </motion.div>
          ))}
        </div>
      </div>

      {/* App Types */}
      <div className="bg-white py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Types of Apps We Develop
            </h2>
            <p className="text-xl text-gray-600">
              Specialized solutions for every industry and use case
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            {appTypes.map((type, index) => (
              <div key={index} className="bg-gray-50 rounded-xl p-6 hover:shadow-lg transition-shadow">
                <div className="p-3 bg-indigo-100 rounded-lg w-fit mb-4">
                  <type.icon className="h-8 w-8 text-indigo-600" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 mb-2">{type.title}</h3>
                <p className="text-gray-600 mb-4">{type.description}</p>
                <div className="space-y-1">
                  {type.examples.map((example, exampleIndex) => (
                    <div key={exampleIndex} className="text-sm text-gray-500">
                      • {example}
                    </div>
                  ))}
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Platforms */}
      <div className="bg-gray-100 py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Development Platforms
            </h2>
            <p className="text-xl text-gray-600">
              Expert development across all major mobile platforms
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            {platforms.map((platform, index) => (
              <div key={index} className="bg-white rounded-xl p-6 text-center shadow-lg hover:shadow-xl transition-shadow">
                <div className={`w-16 h-16 ${platform.color} rounded-full flex items-center justify-center mx-auto mb-4`}>
                  <DevicePhoneMobileIcon className="h-8 w-8 text-white" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 mb-2">{platform.name}</h3>
                <p className="text-gray-600">{platform.description}</p>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Package Pricing */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-12">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">
            Mobile App Development Packages
          </h2>
          <p className="text-xl text-gray-600">
            Choose the perfect package for your app requirements
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
          {packages.map((pkg, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: index * 0.1 }}
              className={`rounded-2xl p-8 ${
                pkg.popular
                  ? 'bg-indigo-600 text-white shadow-2xl scale-105'
                  : 'bg-white shadow-lg'
              }`}
            >
              {pkg.popular && (
                <div className="text-center mb-4">
                  <span className="bg-white text-indigo-600 px-4 py-1 rounded-full text-sm font-semibold">
                    Most Popular
                  </span>
                </div>
              )}
              
              <div className="text-center mb-6">
                <h3 className={`text-2xl font-bold mb-2 ${pkg.popular ? 'text-white' : 'text-gray-900'}`}>
                  {pkg.name}
                </h3>
                <p className={`mb-4 ${pkg.popular ? 'text-indigo-100' : 'text-gray-600'}`}>
                  {pkg.description}
                </p>
                <div className="text-4xl font-bold mb-2">
                  ${pkg.price.toLocaleString()}
                </div>
                <div className={`text-sm ${pkg.popular ? 'text-indigo-100' : 'text-gray-600'}`}>
                  Timeline: {pkg.timeline}
                </div>
              </div>

              <ul className="space-y-3 mb-8">
                {pkg.features.map((feature, featureIndex) => (
                  <li key={featureIndex} className="flex items-center gap-3">
                    <CheckCircleIcon className={`h-5 w-5 ${pkg.popular ? 'text-white' : 'text-green-500'}`} />
                    <span className={pkg.popular ? 'text-indigo-100' : 'text-gray-600'}>
                      {feature}
                    </span>
                  </li>
                ))}
              </ul>

              <button
                onClick={handleAddToCart}
                className={`w-full py-3 px-6 rounded-lg font-semibold transition-colors ${
                  pkg.popular
                    ? 'bg-white text-indigo-600 hover:bg-indigo-50'
                    : 'bg-indigo-600 text-white hover:bg-indigo-700'
                }`}
              >
                Get Started
              </button>
            </motion.div>
          ))}
        </div>
      </div>

      {/* Detailed Services */}
      <div className="bg-white py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
              Our Mobile Development Expertise
            </h2>
            <p className="text-xl text-gray-600">
              Comprehensive mobile app development services
            </p>
          </div>

          <div className="space-y-16">
            {detailedServices.map((service, index) => (
              <motion.div
                key={index}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
                className={`grid grid-cols-1 lg:grid-cols-2 gap-12 items-center ${
                  index % 2 === 1 ? 'lg:grid-flow-col-dense' : ''
                }`}
              >
                <div className={index % 2 === 1 ? 'lg:col-start-2' : ''}>
                  <div className="flex items-center gap-4 mb-4">
                    <div className="p-3 bg-indigo-100 rounded-lg">
                      <service.icon className="h-8 w-8 text-indigo-600" />
                    </div>
                    <h3 className="text-2xl font-bold text-gray-900">{service.title}</h3>
                  </div>
                  <p className="text-lg text-gray-600 mb-4">{service.description}</p>
                  <p className="text-gray-600 mb-4">{service.details}</p>
                  <div className="flex flex-wrap gap-2">
                    {service.technologies.map((tech, techIndex) => (
                      <span key={techIndex} className="bg-indigo-100 text-indigo-800 px-3 py-1 rounded-full text-sm">
                        {tech}
                      </span>
                    ))}
                  </div>
                </div>
                
                <div className={`bg-gradient-to-br ${
                  index % 2 === 0 
                    ? 'from-indigo-50 to-purple-50' 
                    : 'from-purple-50 to-indigo-50'
                } rounded-2xl p-8 ${index % 2 === 1 ? 'lg:col-start-1' : ''}`}>
                  <div className="text-center">
                    <service.icon className={`h-16 w-16 mx-auto mb-6 ${
                      index % 2 === 0 ? 'text-indigo-600' : 'text-purple-600'
                    }`} />
                    <div className="space-y-4">
                      <div className="bg-white p-4 rounded-lg shadow-sm">
                        <div className="font-semibold text-gray-900">Expert Development</div>
                        <div className="text-sm text-gray-600">Professional mobile app solutions</div>
                      </div>
                      <div className="bg-white p-4 rounded-lg shadow-sm">
                        <div className="font-semibold text-gray-900">Latest Technologies</div>
                        <div className="text-sm text-gray-600">Cutting-edge development tools</div>
                      </div>
                    </div>
                  </div>
                </div>
              </motion.div>
            ))}
          </div>
        </div>
      </div>

      {/* Features Grid */}
      <div className="bg-gray-100 py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Mobile App Features
            </h2>
            <p className="text-xl text-gray-600">
              Comprehensive features for modern mobile applications
            </p>
          </div>

          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
            {features.map((feature, index) => (
              <div key={index} className="flex items-center gap-3 p-3 bg-white rounded-lg shadow-sm">
                <div className="w-2 h-2 bg-indigo-500 rounded-full flex-shrink-0"></div>
                <span className="text-gray-700 text-sm">{feature}</span>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Process */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-16">
        <div className="text-center mb-12">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">
            Our Development Process
          </h2>
          <p className="text-xl text-gray-600">
            From idea to app store success
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-6 gap-6">
          {process.map((step, index) => (
            <div key={index} className="text-center">
              <div className="w-16 h-16 bg-indigo-600 text-white rounded-full flex items-center justify-center font-bold text-xl mx-auto mb-4">
                {index + 1}
              </div>
              <h3 className="font-semibold text-gray-900 mb-2">{step}</h3>
              <div className="text-sm text-gray-600">
                {index === 0 && "Defining app requirements and goals"}
                {index === 1 && "Creating intuitive user interfaces"}
                {index === 2 && "Building with modern technologies"}
                {index === 3 && "Comprehensive quality assurance"}
                {index === 4 && "App store submission and launch"}
                {index === 5 && "Ongoing maintenance and updates"}
              </div>
            </div>
          ))}
        </div>
      </div>

      {/* FAQ Section */}
      <div className="bg-white py-16">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Frequently Asked Questions
            </h2>
            <p className="text-xl text-gray-600">
              Everything you need to know about mobile app development
            </p>
          </div>

          <div className="space-y-6">
            {faqs.map((faq, index) => (
              <motion.div
                key={index}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
                className="bg-gray-50 rounded-xl shadow-lg p-6"
              >
                <h3 className="text-lg font-semibold text-gray-900 mb-3">
                  {faq.question}
                </h3>
                <p className="text-gray-600">
                  {faq.answer}
                </p>
              </motion.div>
            ))}
          </div>
        </div>
      </div>

      {/* CTA Section */}
      <div className="bg-gradient-to-r from-indigo-600 to-purple-600 text-white py-16">
        <div className="max-w-4xl mx-auto text-center px-4 sm:px-6 lg:px-8">
          <h2 className="text-3xl md:text-4xl font-bold mb-4">
            Ready to Build Your Mobile App?
          </h2>
          <p className="text-xl text-indigo-100 mb-8">
            Transform your idea into a successful mobile application that users love.
          </p>
          
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <button
              onClick={handleAddToCart}
              className="bg-white text-indigo-600 px-8 py-4 rounded-lg font-semibold hover:bg-indigo-50 transition-colors"
            >
              Start Your App - From $3,500
            </button>
            <Link
              href="/contact"
              className="border-2 border-white text-white px-8 py-4 rounded-lg font-semibold hover:bg-white hover:text-indigo-600 transition-colors"
            >
              Get Free Consultation
            </Link>
          </div>
          
          <div className="mt-8 flex items-center justify-center gap-8 text-indigo-100">
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>App Store Ready</span>
            </div>
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>Cross-Platform</span>
            </div>
            <div className="flex items-center gap-2">
              <CheckCircleIcon className="h-5 w-5" />
              <span>Native Performance</span>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}