'use client'

import { useState } from 'react'
import Link from 'next/link'
import { useCartStore } from '@/store/cart'
import AdvancedServiceSearch from '@/components/AdvancedServiceSearch'
import ServiceComparison from '@/components/ServiceComparison'
import { 
  CodeBracketIcon,
  DevicePhoneMobileIcon,
  GlobeAltIcon,
  ChartBarIcon,
  PaintBrushIcon,
  ShieldCheckIcon,
  CloudIcon,
  MagnifyingGlassIcon,
  PencilIcon,
  MegaphoneIcon,
  ShoppingBagIcon,
  AdjustmentsHorizontalIcon,
  Squares2X2Icon,
  CheckIcon
} from '@heroicons/react/24/outline'

const services = [
  {
    id: 1,
    title: 'Website Development',
    description: 'Custom, responsive websites built with modern technologies like React, Next.js, and Node.js.',
    price: 2500,
    duration: '3-6 weeks',
    icon: CodeBracketIcon,
    slug: 'website-development',
    features: [
      'Responsive Design',
      'SEO Optimized',
      'Fast Loading',
      'Security Features',
      'Content Management',
      '1 Year Support'
    ],
    category: 'Web Development',
    popular: true
  },
  {
    id: 2,
    title: 'Mobile App Development',
    description: 'Native and cross-platform mobile applications for iOS and Android platforms.',
    price: 4500,
    duration: '10-16 weeks',
    icon: DevicePhoneMobileIcon,
    slug: 'mobile-app-development',
    features: [
      'Cross-platform Support',
      'Native Performance',
      'App Store Deployment',
      'Push Notifications',
      'Offline Functionality',
      '6 Months Support'
    ],
    category: 'Mobile Development'
  },
  {
    id: 3,
    title: 'E-commerce Development',
    description: 'Complete online store solutions with payment integration and inventory management.',
    price: 3500,
    duration: '6-12 weeks',
    icon: ShoppingBagIcon,
    slug: 'ecommerce-development',
    features: [
      'Payment Gateway Integration',
      'Inventory Management',
      'Order Tracking',
      'Multi-currency Support',
      'Analytics Dashboard',
      '1 Year Support'
    ],
    category: 'E-commerce',
    popular: true
  },
  {
    id: 4,
    title: 'Social Media Marketing',
    description: 'Comprehensive social media marketing strategies to boost your online presence and drive growth.',
    price: 500,
    duration: '2-3 months',
    icon: ChartBarIcon,
    slug: 'social-media-marketing',
    features: [
      'Content Development & Strategy',
      'Campaign Management & Optimization',
      'Community Management',
      'Social Media Listening',
      'Influencer Marketing',
      'Monthly Analytics & Reporting'
    ],
    category: 'Digital Marketing'
  },
  {
    id: 5,
    title: 'SEO Optimization',
    description: 'Search engine optimization to improve your website\'s visibility and ranking.',
    price: 1200,
    duration: '3-6 months',
    icon: MagnifyingGlassIcon,
    slug: 'seo-optimization',
    features: [
      'Keyword Research & Analysis',
      'On-page & Off-page Optimization', 
      'Technical SEO Audits',
      'Content Strategy & Creation',
      'Link Building & Outreach',
      'Monthly Performance Tracking'
    ],
    category: 'Digital Marketing',
    popular: true
  },
  {
    id: 6,
    title: 'Digital Marketing',
    description: 'Complete digital marketing campaigns including PPC, social media, and email marketing.',
    price: 1500,
    duration: '3-6 months',
    icon: MegaphoneIcon,
    slug: 'digital-marketing',
    features: [
      'PPC Campaign Management',
      'Social Media Advertising',
      'Email Marketing Automation',
      'Content Marketing',
      'Analytics & Reporting',
      'Conversion Optimization'
    ],
    category: 'Digital Marketing'
  },
  {
    id: 7,
    title: 'Content Writing',
    description: 'Professional content writing services for websites, blogs, and marketing materials.',
    price: 500,
    duration: '2-4 weeks',
    icon: PencilIcon,
    slug: 'content-writing',
    features: [
      'Website Copy',
      'Blog Articles',
      'SEO Content',
      'Email Marketing Copy',
      'Social Media Content',
      'Technical Writing'
    ],
    category: 'Content Services'
  },
  {
    id: 8,
    title: 'Graphic Design',
    description: 'Creative graphic design services for branding, marketing materials, and digital assets.',
    price: 400,
    duration: '1-3 weeks',
    icon: PaintBrushIcon,
    slug: 'graphic-design',
    features: [
      'Logo Design',
      'Brand Identity',
      'Marketing Materials',
      'Web Graphics',
      'Print Design',
      'Social Media Graphics'
    ],
    category: 'Design Services'
  }
]

const categories = ['All', 'Web Development', 'Mobile Development', 'E-commerce', 'Digital Marketing', 'Content Services', 'Design Services']

export default function ServicesPage() {
  const [selectedCategory, setSelectedCategory] = useState('All')
  const [activeView, setActiveView] = useState<'grid' | 'search' | 'compare'>('grid')
  const addToCart = useCartStore((state) => state.addToCart)
  const openCart = useCartStore((state) => state.openCart)

  const filteredServices = selectedCategory === 'All' 
    ? services 
    : services.filter(service => service.category === selectedCategory)

  const handleAddToCart = (service: typeof services[0]) => {
    addToCart({
      id: service.id,
      title: service.title,
      price: service.price,
      quantity: 1
    })
    
    // Open the cart to show the item was added
    setTimeout(() => {
      openCart()
    }, 100)
  }

  return (
    <div className="min-h-screen bg-gray-50 py-12">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <div className="text-center mb-12">
          <h1 className="text-4xl font-bold text-gray-900 mb-4">
            Our Services
          </h1>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            We provide comprehensive digital solutions to help your business thrive in the modern world.
            From web development to digital marketing, we&apos;ve got you covered.
          </p>
        </div>

        {/* View Toggle */}
        <div className="flex justify-center mb-8">
          <div className="flex bg-white rounded-lg p-1 shadow-sm border">
            <button
              onClick={() => setActiveView('grid')}
              className={`flex items-center px-4 py-2 rounded-md font-medium transition-all ${
                activeView === 'grid'
                  ? 'bg-blue-600 text-white shadow-sm'
                  : 'text-gray-600 hover:bg-gray-100'
              }`}
            >
              <Squares2X2Icon className="w-4 h-4 mr-2" />
              Grid View
            </button>
            <button
              onClick={() => setActiveView('search')}
              className={`flex items-center px-4 py-2 rounded-md font-medium transition-all ${
                activeView === 'search'
                  ? 'bg-blue-600 text-white shadow-sm'
                  : 'text-gray-600 hover:bg-gray-100'
              }`}
            >
              <AdjustmentsHorizontalIcon className="w-4 h-4 mr-2" />
              Advanced Search
            </button>
            <button
              onClick={() => setActiveView('compare')}
              className={`flex items-center px-4 py-2 rounded-md font-medium transition-all ${
                activeView === 'compare'
                  ? 'bg-blue-600 text-white shadow-sm'
                  : 'text-gray-600 hover:bg-gray-100'
              }`}
            >
              <CheckIcon className="w-4 h-4 mr-2" />
              Compare Services
            </button>
          </div>
        </div>

        {/* Advanced Search View */}
        {activeView === 'search' && (
          <div className="mb-12">
            <AdvancedServiceSearch 
              services={services}
              onFilteredResults={(filteredResults) => {
                // You can add logic here to update the grid view with filtered results
                console.log('Filtered results:', filteredResults)
              }}
            />
          </div>
        )}

        {/* Service Comparison View */}
        {activeView === 'compare' && (
          <div className="mb-12">
            <ServiceComparison 
              availableServices={services}
            />
          </div>
        )}

        {/* Grid View */}
        {activeView === 'grid' && (
          <>
            {/* Category Filter */}
            <div className="flex flex-wrap justify-center gap-4 mb-12">
              {categories.map((category) => (
                <button
                  key={category}
                  onClick={() => setSelectedCategory(category)}
                  className={`px-6 py-3 rounded-full font-medium transition-colors ${
                    selectedCategory === category
                      ? 'bg-blue-600 text-white'
                      : 'bg-white text-gray-600 hover:bg-gray-100'
                  }`}
                >
                  {category}
                </button>
              ))}
            </div>

            {/* Anchor points for navigation */}
            <div id="website-development" className="scroll-mt-20"></div>
            <div id="mobile-apps" className="scroll-mt-20"></div>
            <div id="digital-marketing" className="scroll-mt-20"></div>
            <div id="design" className="scroll-mt-20"></div>
            <div id="cloud" className="scroll-mt-20"></div>
            <div id="seo" className="scroll-mt-20"></div>
            
            {/* Services Grid */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
              {filteredServices.map((service) => (
                <div
                  key={service.id}
                  id={service.slug}
                  className="bg-white rounded-xl shadow-lg overflow-hidden hover:shadow-xl transition-shadow duration-300 relative"
                >
                  {/* Popular Badge */}
                  {service.popular && (
                    <div className="absolute top-4 right-4 bg-blue-600 text-white px-3 py-1 rounded-full text-xs font-medium">
                      Popular
                    </div>
                  )}

                  {/* Service Header */}
                  <div className="p-6 border-b border-gray-100">
                    <div className="flex items-center mb-4">
                      <div className="p-3 bg-blue-100 rounded-lg">
                        <service.icon className="h-6 w-6 text-blue-600" />
                      </div>
                      <div className="ml-4">
                        <h3 className="text-xl font-semibold text-gray-900">{service.title}</h3>
                        <span className="text-sm text-gray-500">{service.category}</span>
                      </div>
                    </div>
                    <p className="text-gray-600">{service.description}</p>
                  </div>

                  {/* Pricing */}
                  <div className="px-6 py-4 bg-gray-50">
                    <div className="flex justify-between items-center">
                      <div>
                        <span className="text-3xl font-bold text-gray-900">
                          ${service.price.toLocaleString()}
                        </span>
                        <span className="text-gray-500 ml-2">starting from</span>
                      </div>
                      <div className="text-right">
                        <div className="text-sm text-gray-500">Duration</div>
                        <div className="font-medium text-gray-900">{service.duration}</div>
                      </div>
                    </div>
                  </div>

                  {/* Features */}
                  <div className="p-6">
                    <h4 className="font-semibold text-gray-900 mb-3">What&apos;s Included:</h4>
                    <ul className="space-y-2">
                      {service.features.map((feature, index) => (
                        <li key={index} className="flex items-center text-sm text-gray-600">
                          <div className="w-1.5 h-1.5 bg-blue-600 rounded-full mr-3"></div>
                          {feature}
                        </li>
                      ))}
                    </ul>
                  </div>

                  {/* Action Buttons */}
                  <div className="p-6 pt-0 space-y-3">
                    <Link
                      href={`/services/${service.slug}`}
                      className="block w-full text-center bg-gray-100 text-gray-700 py-3 px-6 rounded-lg font-medium hover:bg-gray-200 transition-colors"
                    >
                      Learn More
                    </Link>
                    <button
                      onClick={() => handleAddToCart(service)}
                      className="w-full bg-blue-600 text-white py-3 px-6 rounded-lg font-medium hover:bg-blue-700 transition-colors"
                    >
                      Add to Cart
                    </button>
                  </div>
                </div>
              ))}
            </div>
          </>
        )}

        {/* Call to Action */}
        <div className="mt-16 text-center">
          <div className="bg-white rounded-2xl shadow-lg p-8 max-w-4xl mx-auto">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">
              Need Something Custom?
            </h2>
            <p className="text-xl text-gray-600 mb-6">
              Don&apos;t see exactly what you&apos;re looking for? We specialize in custom solutions 
              tailored to your specific business needs.
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <a
                href="#contact"
                className="bg-blue-600 text-white px-8 py-3 rounded-lg font-medium hover:bg-blue-700 transition-colors"
              >
                Get Custom Quote
              </a>
              <a
                href="/contact"
                className="border border-gray-300 text-gray-700 px-8 py-3 rounded-lg font-medium hover:bg-gray-50 transition-colors"
              >
                Contact Us
              </a>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}