'use client'

import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { 
  CheckCircleIcon, 
  ExclamationTriangleIcon,
  XCircleIcon,
  ClockIcon,
  ServerIcon,
  GlobeAltIcon,
  ShieldCheckIcon,
  ChartBarIcon
} from '@heroicons/react/24/outline'

interface ServiceStatus {
  name: string
  status: 'operational' | 'degraded' | 'down' | 'maintenance'
  uptime: string
  responseTime: string
  icon: any
}

export default function SystemStatus() {
  const [lastUpdate, setLastUpdate] = useState(new Date())

  useEffect(() => {
    const interval = setInterval(() => {
      setLastUpdate(new Date())
    }, 30000) // Update every 30 seconds

    return () => clearInterval(interval)
  }, [])

  const services: ServiceStatus[] = [
    {
      name: 'Shared Hosting',
      status: 'operational',
      uptime: '99.98%',
      responseTime: '245ms',
      icon: ServerIcon
    },
    {
      name: 'VPS Hosting',
      status: 'operational',
      uptime: '99.99%',
      responseTime: '178ms',
      icon: ServerIcon
    },
    {
      name: 'Dedicated Servers',
      status: 'operational',
      uptime: '100%',
      responseTime: '156ms',
      icon: ServerIcon
    },
    {
      name: 'Domain Services',
      status: 'operational',
      uptime: '99.97%',
      responseTime: '312ms',
      icon: GlobeAltIcon
    },
    {
      name: 'Email Services',
      status: 'operational',
      uptime: '99.96%',
      responseTime: '423ms',
      icon: ShieldCheckIcon
    },
    {
      name: 'Control Panel',
      status: 'operational',
      uptime: '99.95%',
      responseTime: '534ms',
      icon: ChartBarIcon
    }
  ]

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'operational':
        return 'text-green-600 bg-green-50 border-green-200'
      case 'degraded':
        return 'text-yellow-600 bg-yellow-50 border-yellow-200'
      case 'down':
        return 'text-red-600 bg-red-50 border-red-200'
      case 'maintenance':
        return 'text-blue-600 bg-blue-50 border-blue-200'
      default:
        return 'text-gray-600 bg-gray-50 border-gray-200'
    }
  }

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'operational':
        return <CheckCircleIcon className="w-6 h-6 text-green-600" />
      case 'degraded':
        return <ExclamationTriangleIcon className="w-6 h-6 text-yellow-600" />
      case 'down':
        return <XCircleIcon className="w-6 h-6 text-red-600" />
      case 'maintenance':
        return <ClockIcon className="w-6 h-6 text-blue-600" />
      default:
        return <CheckCircleIcon className="w-6 h-6 text-gray-600" />
    }
  }

  const getStatusText = (status: string) => {
    switch (status) {
      case 'operational':
        return 'All Systems Operational'
      case 'degraded':
        return 'Degraded Performance'
      case 'down':
        return 'Service Disruption'
      case 'maintenance':
        return 'Scheduled Maintenance'
      default:
        return 'Unknown Status'
    }
  }

  const overallStatus = services.every(s => s.status === 'operational') ? 'operational' : 'degraded'

  return (
    <div className="min-h-screen bg-gradient-to-b from-slate-50 to-white">
      {/* Hero Section */}
      <section className="bg-slate-900 text-white py-20">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="text-center"
          >
            <div className="flex justify-center mb-6">
              <motion.div
                animate={{ scale: [1, 1.05, 1] }}
                transition={{ repeat: Infinity, duration: 2 }}
                className="rounded-full bg-green-600/10 p-4"
              >
                <CheckCircleIcon className="w-12 h-12 text-green-400" />
              </motion.div>
            </div>
            <h1 className="text-4xl md:text-5xl font-bold mb-6">
              System Status
            </h1>
            <p className="text-xl text-slate-300 mb-4">
              Real-time monitoring of all Nuwair Systems services
            </p>
            <div className="inline-flex items-center px-6 py-3 bg-green-600/20 border border-green-500 rounded-lg">
              <CheckCircleIcon className="w-5 h-5 text-green-400 mr-2" />
              <span className="text-green-300 font-semibold">{getStatusText(overallStatus)}</span>
            </div>
          </motion.div>
        </div>
      </section>

      {/* Last Updated */}
      <section className="py-4 bg-white border-b border-slate-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex items-center justify-center text-sm text-slate-600">
            <ClockIcon className="w-4 h-4 mr-2" />
            Last updated: {lastUpdate.toLocaleString('en-US', { 
              month: 'short', 
              day: 'numeric', 
              year: 'numeric',
              hour: '2-digit', 
              minute: '2-digit',
              second: '2-digit'
            })}
            <span className="ml-4 text-slate-400">•</span>
            <span className="ml-4">Auto-refresh every 30 seconds</span>
          </div>
        </div>
      </section>

      {/* Overall Stats */}
      <section className="py-12 bg-slate-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid md:grid-cols-3 gap-6">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.1 }}
              className="bg-white rounded-xl shadow-sm border border-slate-200 p-6 text-center"
            >
              <div className="text-3xl font-bold text-green-600 mb-2">99.98%</div>
              <div className="text-slate-600">Average Uptime (30 days)</div>
            </motion.div>
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.2 }}
              className="bg-white rounded-xl shadow-sm border border-slate-200 p-6 text-center"
            >
              <div className="text-3xl font-bold text-blue-600 mb-2">308ms</div>
              <div className="text-slate-600">Average Response Time</div>
            </motion.div>
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.3 }}
              className="bg-white rounded-xl shadow-sm border border-slate-200 p-6 text-center"
            >
              <div className="text-3xl font-bold text-slate-900 mb-2">0</div>
              <div className="text-slate-600">Active Incidents</div>
            </motion.div>
          </div>
        </div>
      </section>

      {/* Service Status */}
      <section className="py-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <h2 className="text-3xl font-bold text-slate-900 mb-8 text-center">
            Service Status
          </h2>
          <div className="space-y-4">
            {services.map((service, index) => (
              <motion.div
                key={index}
                initial={{ opacity: 0, x: -20 }}
                animate={{ opacity: 1, x: 0 }}
                transition={{ delay: index * 0.1 }}
                className="bg-white rounded-xl shadow-sm border border-slate-200 p-6 hover:shadow-md transition-shadow"
              >
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-4 flex-1">
                    <service.icon className="w-8 h-8 text-slate-600" />
                    <div className="flex-1">
                      <h3 className="text-lg font-semibold text-slate-900">
                        {service.name}
                      </h3>
                      <div className="flex items-center space-x-4 mt-1 text-sm text-slate-600">
                        <span>Uptime: {service.uptime}</span>
                        <span>•</span>
                        <span>Response: {service.responseTime}</span>
                      </div>
                    </div>
                  </div>
                  <div className={`flex items-center px-4 py-2 rounded-lg border ${getStatusColor(service.status)}`}>
                    {getStatusIcon(service.status)}
                    <span className="ml-2 font-semibold capitalize">{service.status}</span>
                  </div>
                </div>
              </motion.div>
            ))}
          </div>
        </div>
      </section>

      {/* Incident History */}
      <section className="py-16 bg-slate-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <h2 className="text-3xl font-bold text-slate-900 mb-8 text-center">
            Recent Incidents
          </h2>
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="bg-white rounded-xl shadow-sm border border-slate-200 p-12 text-center"
          >
            <CheckCircleIcon className="w-16 h-16 text-green-600 mx-auto mb-4" />
            <h3 className="text-xl font-semibold text-slate-900 mb-2">
              No Recent Incidents
            </h3>
            <p className="text-slate-600">
              All services have been running smoothly. No incidents reported in the last 30 days.
            </p>
          </motion.div>
        </div>
      </section>

      {/* Subscribe to Updates */}
      <section className="py-16">
        <div className="max-w-3xl mx-auto px-4 sm:px-6 lg:px-8">
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="bg-gradient-to-r from-blue-600 to-indigo-600 rounded-2xl shadow-xl p-8 md:p-12 text-center text-white"
          >
            <h2 className="text-3xl font-bold mb-4">
              Stay Informed
            </h2>
            <p className="text-lg text-blue-100 mb-8">
              Subscribe to receive real-time updates about service status and planned maintenance
            </p>
            <div className="flex flex-col sm:flex-row gap-4 max-w-md mx-auto">
              <input
                type="email"
                placeholder="Enter your email"
                className="flex-1 px-4 py-3 rounded-lg text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-300"
              />
              <button className="px-6 py-3 bg-white text-blue-600 rounded-lg font-semibold hover:bg-blue-50 transition-colors">
                Subscribe
              </button>
            </div>
            <p className="text-sm text-blue-200 mt-4">
              You can unsubscribe at any time. View our Privacy Policy.
            </p>
          </motion.div>
        </div>
      </section>
    </div>
  )
}
