'use client'

import { useState, useEffect } from 'react'
import { useSession } from 'next-auth/react'
import { motion, AnimatePresence } from 'framer-motion'
import {
  SparklesIcon,
  BuildingOfficeIcon,
  ShoppingBagIcon,
  AcademicCapIcon,
  HeartIcon,
  CpuChipIcon,
  LightBulbIcon,
  ChartBarIcon,
  CurrencyDollarIcon,
  ClockIcon,
  CheckCircleIcon,
  ArrowRightIcon,
  DocumentTextIcon,
  StarIcon
} from '@heroicons/react/24/outline'

interface BusinessProfile {
  id: string
  type: string
  industry: string
  size: 'startup' | 'small' | 'medium' | 'enterprise'
  budget: 'basic' | 'standard' | 'premium' | 'enterprise'
  goals: string[]
  currentWebsite: boolean
  targetAudience: string
  timeframe: 'urgent' | 'standard' | 'flexible'
  location: 'local' | 'national' | 'international'
}

interface ServiceRecommendation {
  serviceId: string
  serviceName: string
  priority: 'high' | 'medium' | 'low'
  confidence: number
  estimatedPrice: {
    min: number
    max: number
    currency: string
  }
  timeline: {
    min: number
    max: number
    unit: 'days' | 'weeks' | 'months'
  }
  benefits: string[]
  reasoning: string
  package: 'basic' | 'standard' | 'premium' | 'enterprise'
  roi: number
  features: string[]
}

interface BusinessType {
  id: string
  name: string
  icon: any
  description: string
  industries: string[]
  commonGoals: string[]
  recommendedServices: string[]
}

const BUSINESS_TYPES: BusinessType[] = [
  {
    id: 'ecommerce',
    name: 'E-commerce Business',
    icon: ShoppingBagIcon,
    description: 'Online retail and product sales',
    industries: ['Retail', 'Fashion', 'Electronics', 'Food & Beverage', 'Handmade'],
    commonGoals: ['Increase Sales', 'Expand Market Reach', 'Improve User Experience', 'Mobile Commerce'],
    recommendedServices: ['website-development', 'mobile-app-development', 'seo-optimization', 'social-media-marketing']
  },
  {
    id: 'service-business',
    name: 'Service Provider',
    icon: BuildingOfficeIcon,
    description: 'Professional services and consultancy',
    industries: ['Consulting', 'Legal', 'Healthcare', 'Finance', 'Real Estate'],
    commonGoals: ['Lead Generation', 'Brand Authority', 'Client Management', 'Online Presence'],
    recommendedServices: ['website-development', 'seo-optimization', 'cloud-solutions', 'ui-ux-design']
  },
  {
    id: 'tech-startup',
    name: 'Tech Startup',
    icon: CpuChipIcon,
    description: 'Technology and innovation companies',
    industries: ['Software', 'AI/ML', 'Fintech', 'SaaS', 'Blockchain'],
    commonGoals: ['Product Development', 'Scalability', 'User Acquisition', 'Investment Ready'],
    recommendedServices: ['mobile-app-development', 'cloud-solutions', 'ui-ux-design', 'website-development']
  },
  {
    id: 'education',
    name: 'Educational Institution',
    icon: AcademicCapIcon,
    description: 'Schools, universities, and training centers',
    industries: ['Schools', 'Universities', 'Online Learning', 'Training Centers'],
    commonGoals: ['Student Engagement', 'Online Learning', 'Administrative Efficiency', 'Parent Communication'],
    recommendedServices: ['website-development', 'mobile-app-development', 'cloud-solutions', 'ui-ux-design']
  },
  {
    id: 'healthcare',
    name: 'Healthcare Provider',
    icon: HeartIcon,
    description: 'Medical practices and health services',
    industries: ['Hospitals', 'Clinics', 'Dental', 'Veterinary', 'Mental Health'],
    commonGoals: ['Patient Management', 'Appointment Booking', 'Telemedicine', 'Compliance'],
    recommendedServices: ['website-development', 'mobile-app-development', 'cloud-solutions', 'seo-optimization']
  },
  {
    id: 'creative',
    name: 'Creative Agency',
    icon: LightBulbIcon,
    description: 'Design, marketing, and creative services',
    industries: ['Advertising', 'Design', 'Photography', 'Video Production', 'Marketing'],
    commonGoals: ['Portfolio Showcase', 'Client Acquisition', 'Brand Building', 'Creative Tools'],
    recommendedServices: ['ui-ux-design', 'website-development', 'social-media-marketing', 'graphic-design']
  }
]

const AI_RECOMMENDATIONS_ENGINE = {
  analyzeBusinessProfile: (profile: Partial<BusinessProfile>): ServiceRecommendation[] => {
    const recommendations: ServiceRecommendation[] = []
    
    // Core website development (always high priority)
    if (!profile.currentWebsite) {
      recommendations.push({
        serviceId: 'website-development',
        serviceName: 'Website Development',
        priority: 'high',
        confidence: 95,
        estimatedPrice: {
          min: profile.budget === 'basic' ? 25000 : profile.budget === 'standard' ? 50000 : profile.budget === 'premium' ? 100000 : 200000,
          max: profile.budget === 'basic' ? 50000 : profile.budget === 'standard' ? 100000 : profile.budget === 'premium' ? 200000 : 500000,
          currency: 'PKR'
        },
        timeline: {
          min: profile.timeframe === 'urgent' ? 2 : 3,
          max: profile.timeframe === 'urgent' ? 4 : profile.timeframe === 'standard' ? 6 : 8,
          unit: 'weeks'
        },
        benefits: ['Professional Online Presence', '24/7 Customer Access', 'Increased Credibility', 'Lead Generation'],
        reasoning: 'Every modern business needs a professional website as the foundation of their digital presence.',
        package: profile.budget || 'standard',
        roi: 300,
        features: ['Responsive Design', 'SEO Optimized', 'Contact Forms', 'Content Management']
      })
    }

    // Mobile app for specific business types
    if (['ecommerce', 'tech-startup', 'education', 'healthcare'].includes(profile.type || '')) {
      recommendations.push({
        serviceId: 'mobile-app-development',
        serviceName: 'Mobile App Development',
        priority: profile.type === 'ecommerce' || profile.type === 'tech-startup' ? 'high' : 'medium',
        confidence: profile.type === 'ecommerce' ? 90 : 75,
        estimatedPrice: {
          min: profile.budget === 'basic' ? 100000 : 200000,
          max: profile.budget === 'enterprise' ? 1000000 : 500000,
          currency: 'PKR'
        },
        timeline: {
          min: 8,
          max: 16,
          unit: 'weeks'
        },
        benefits: ['Mobile-First Experience', 'Push Notifications', 'Offline Functionality', 'App Store Presence'],
        reasoning: `${profile.type === 'ecommerce' ? 'E-commerce businesses see 2.5x higher conversion rates' : 'Mobile apps increase customer engagement by 3x'} compared to mobile websites.`,
        package: profile.budget || 'standard',
        roi: profile.type === 'ecommerce' ? 250 : 200,
        features: ['Native iOS/Android', 'Push Notifications', 'Offline Mode', 'Analytics Integration']
      })
    }

    // SEO for all business types
    recommendations.push({
      serviceId: 'seo-optimization',
      serviceName: 'SEO Optimization',
      priority: 'high',
      confidence: 85,
      estimatedPrice: {
        min: 15000,
        max: 50000,
        currency: 'PKR'
      },
      timeline: {
        min: 3,
        max: 6,
        unit: 'months'
      },
      benefits: ['Higher Search Rankings', 'Organic Traffic Growth', 'Long-term Results', 'Cost-Effective Marketing'],
      reasoning: 'SEO provides the highest ROI among digital marketing strategies, with 70% of users clicking organic results.',
      package: profile.budget || 'standard',
      roi: 400,
      features: ['Keyword Research', 'Content Optimization', 'Technical SEO', 'Monthly Reports']
    })

    // Cloud solutions for scalable businesses
    if (['tech-startup', 'education', 'healthcare', 'service-business'].includes(profile.type || '') && 
        ['medium', 'enterprise'].includes(profile.size || '')) {
      recommendations.push({
        serviceId: 'cloud-solutions',
        serviceName: 'Cloud Solutions',
        priority: 'medium',
        confidence: 80,
        estimatedPrice: {
          min: 50000,
          max: 200000,
          currency: 'PKR'
        },
        timeline: {
          min: 4,
          max: 8,
          unit: 'weeks'
        },
        benefits: ['Scalability', 'Cost Reduction', 'Data Security', 'Remote Access'],
        reasoning: 'Cloud infrastructure reduces IT costs by 20-30% while improving scalability and security.',
        package: profile.budget || 'standard',
        roi: 180,
        features: ['AWS/Azure Setup', 'Auto Scaling', 'Backup Solutions', 'Security Configuration']
      })
    }

    // Social media marketing for customer-facing businesses
    if (['ecommerce', 'creative', 'service-business'].includes(profile.type || '')) {
      recommendations.push({
        serviceId: 'social-media-marketing',
        serviceName: 'Social Media Marketing',
        priority: profile.type === 'ecommerce' ? 'high' : 'medium',
        confidence: 75,
        estimatedPrice: {
          min: 25000,
          max: 75000,
          currency: 'PKR'
        },
        timeline: {
          min: 1,
          max: 3,
          unit: 'months'
        },
        benefits: ['Brand Awareness', 'Customer Engagement', 'Lead Generation', 'Community Building'],
        reasoning: 'Social media marketing generates 3x more leads than traditional marketing and costs 62% less.',
        package: profile.budget || 'standard',
        roi: 300,
        features: ['Content Strategy', 'Campaign Management', 'Analytics', 'Community Management']
      })
    }

    return recommendations.sort((a, b) => {
      const priorityOrder = { high: 3, medium: 2, low: 1 }
      return priorityOrder[b.priority] - priorityOrder[a.priority] || b.confidence - a.confidence
    })
  },

  generateProjectProposal: (profile: BusinessProfile, selectedServices: string[]): string => {
    const totalBudget = selectedServices.length * (profile.budget === 'basic' ? 50000 : 
                                                   profile.budget === 'standard' ? 100000 : 
                                                   profile.budget === 'premium' ? 200000 : 400000)
    
    return `
# Digital Transformation Proposal for ${profile.industry} Business

## Executive Summary
Based on our AI analysis of your business profile, we've identified key opportunities to enhance your digital presence and achieve your goals of ${profile.goals.join(', ')}.

## Recommended Solution Package
- **Total Investment**: Rs. ${totalBudget.toLocaleString()}
- **Timeline**: 3-6 months phased implementation
- **Expected ROI**: 250-400% within first year

## Implementation Strategy
1. **Phase 1**: Foundation (Website/App Development)
2. **Phase 2**: Optimization (SEO/Performance)
3. **Phase 3**: Growth (Marketing/Analytics)

## Success Metrics
- 300% increase in online leads
- 50% improvement in customer engagement
- 40% reduction in operational costs

*This proposal is generated by our AI recommendation engine based on industry best practices and your specific business requirements.*
    `
  }
}

export default function AIServiceRecommendations() {
  const { data: session } = useSession()
  const [currentStep, setCurrentStep] = useState<'profile' | 'analysis' | 'recommendations' | 'proposal'>('profile')
  const [businessProfile, setBusinessProfile] = useState<Partial<BusinessProfile>>({})
  const [recommendations, setRecommendations] = useState<ServiceRecommendation[]>([])
  const [selectedServices, setSelectedServices] = useState<string[]>([])
  const [loading, setLoading] = useState(false)

  const handleProfileUpdate = (field: keyof BusinessProfile, value: any) => {
    setBusinessProfile(prev => ({ ...prev, [field]: value }))
  }

  const generateRecommendations = async () => {
    setLoading(true)
    setCurrentStep('analysis')
    
    // Simulate AI processing
    await new Promise(resolve => setTimeout(resolve, 2000))
    
    const aiRecommendations = AI_RECOMMENDATIONS_ENGINE.analyzeBusinessProfile(businessProfile)
    setRecommendations(aiRecommendations)
    setCurrentStep('recommendations')
    setLoading(false)
  }

  const handleServiceToggle = (serviceId: string) => {
    setSelectedServices(prev => 
      prev.includes(serviceId) 
        ? prev.filter(id => id !== serviceId)
        : [...prev, serviceId]
    )
  }

  const generateProposal = () => {
    setCurrentStep('proposal')
  }

  const formatCurrency = (amount: number): string => {
    return `Rs. ${amount.toLocaleString('en-PK')}`
  }

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'high': return 'bg-red-100 text-red-800 border-red-200'
      case 'medium': return 'bg-yellow-100 text-yellow-800 border-yellow-200'
      case 'low': return 'bg-green-100 text-green-800 border-green-200'
      default: return 'bg-gray-100 text-gray-800 border-gray-200'
    }
  }

  const renderBusinessProfile = () => (
    <div className="space-y-8">
      <div className="text-center">
        <SparklesIcon className="w-16 h-16 text-blue-600 mx-auto mb-4" />
        <h2 className="text-2xl font-bold text-gray-900 mb-2">AI Service Recommendations</h2>
        <p className="text-gray-600">Tell us about your business and get personalized service recommendations</p>
      </div>

      <div className="space-y-6">
        {/* Business Type Selection */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-3">What type of business do you have?</label>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {BUSINESS_TYPES.map((type) => {
              const IconComponent = type.icon
              return (
                <motion.button
                  key={type.id}
                  onClick={() => handleProfileUpdate('type', type.id)}
                  className={`p-4 border-2 rounded-lg text-left transition-all ${
                    businessProfile.type === type.id
                      ? 'border-blue-500 bg-blue-50'
                      : 'border-gray-200 hover:border-gray-300'
                  }`}
                  whileHover={{ scale: 1.02 }}
                  whileTap={{ scale: 0.98 }}
                >
                  <IconComponent className="w-8 h-8 text-blue-600 mb-3" />
                  <h3 className="font-medium text-gray-900 mb-1">{type.name}</h3>
                  <p className="text-sm text-gray-600">{type.description}</p>
                </motion.button>
              )
            })}
          </div>
        </div>

        {/* Industry */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">Industry</label>
          <select
            value={businessProfile.industry || ''}
            onChange={(e) => handleProfileUpdate('industry', e.target.value)}
            className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
          >
            <option value="">Select your industry</option>
            <option value="Technology">Technology</option>
            <option value="Healthcare">Healthcare</option>
            <option value="Education">Education</option>
            <option value="Retail">Retail</option>
            <option value="Finance">Finance</option>
            <option value="Real Estate">Real Estate</option>
            <option value="Food & Beverage">Food & Beverage</option>
            <option value="Manufacturing">Manufacturing</option>
            <option value="Consulting">Consulting</option>
            <option value="Other">Other</option>
          </select>
        </div>

        {/* Business Size */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-3">Business Size</label>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
            {[
              { id: 'startup', label: 'Startup', desc: '1-10 employees' },
              { id: 'small', label: 'Small', desc: '11-50 employees' },
              { id: 'medium', label: 'Medium', desc: '51-200 employees' },
              { id: 'enterprise', label: 'Enterprise', desc: '200+ employees' }
            ].map((size) => (
              <button
                key={size.id}
                onClick={() => handleProfileUpdate('size', size.id)}
                className={`p-3 border-2 rounded-lg text-center transition-all ${
                  businessProfile.size === size.id
                    ? 'border-blue-500 bg-blue-50'
                    : 'border-gray-200 hover:border-gray-300'
                }`}
              >
                <div className="font-medium text-gray-900">{size.label}</div>
                <div className="text-xs text-gray-600">{size.desc}</div>
              </button>
            ))}
          </div>
        </div>

        {/* Budget Range */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-3">Budget Range</label>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
            {[
              { id: 'basic', label: 'Basic', range: 'Rs. 25K - 100K' },
              { id: 'standard', label: 'Standard', range: 'Rs. 100K - 300K' },
              { id: 'premium', label: 'Premium', range: 'Rs. 300K - 800K' },
              { id: 'enterprise', label: 'Enterprise', range: 'Rs. 800K+' }
            ].map((budget) => (
              <button
                key={budget.id}
                onClick={() => handleProfileUpdate('budget', budget.id)}
                className={`p-3 border-2 rounded-lg text-center transition-all ${
                  businessProfile.budget === budget.id
                    ? 'border-blue-500 bg-blue-50'
                    : 'border-gray-200 hover:border-gray-300'
                }`}
              >
                <div className="font-medium text-gray-900">{budget.label}</div>
                <div className="text-xs text-gray-600">{budget.range}</div>
              </button>
            ))}
          </div>
        </div>

        {/* Business Goals */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-3">Primary Goals (Select all that apply)</label>
          <div className="grid grid-cols-2 md:grid-cols-3 gap-3">
            {[
              'Increase Sales', 'Generate Leads', 'Brand Awareness', 'Customer Engagement',
              'Operational Efficiency', 'Market Expansion', 'Cost Reduction', 'Innovation'
            ].map((goal) => (
              <label key={goal} className="flex items-center space-x-2">
                <input
                  type="checkbox"
                  checked={businessProfile.goals?.includes(goal) || false}
                  onChange={(e) => {
                    const currentGoals = businessProfile.goals || []
                    const newGoals = e.target.checked
                      ? [...currentGoals, goal]
                      : currentGoals.filter(g => g !== goal)
                    handleProfileUpdate('goals', newGoals)
                  }}
                  className="h-4 w-4 text-blue-600 rounded border-gray-300 focus:ring-blue-500"
                />
                <span className="text-sm text-gray-700">{goal}</span>
              </label>
            ))}
          </div>
        </div>

        {/* Current Website */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-3">Do you currently have a website?</label>
          <div className="flex space-x-4">
            <label className="flex items-center">
              <input
                type="radio"
                checked={businessProfile.currentWebsite === true}
                onChange={() => handleProfileUpdate('currentWebsite', true)}
                className="h-4 w-4 text-blue-600 border-gray-300 focus:ring-blue-500"
              />
              <span className="ml-2 text-sm text-gray-700">Yes</span>
            </label>
            <label className="flex items-center">
              <input
                type="radio"
                checked={businessProfile.currentWebsite === false}
                onChange={() => handleProfileUpdate('currentWebsite', false)}
                className="h-4 w-4 text-blue-600 border-gray-300 focus:ring-blue-500"
              />
              <span className="ml-2 text-sm text-gray-700">No</span>
            </label>
          </div>
        </div>

        {/* Timeline */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-3">Project Timeline</label>
          <div className="grid grid-cols-3 gap-3">
            {[
              { id: 'urgent', label: 'Urgent', desc: 'ASAP' },
              { id: 'standard', label: 'Standard', desc: '2-3 months' },
              { id: 'flexible', label: 'Flexible', desc: '3+ months' }
            ].map((timeline) => (
              <button
                key={timeline.id}
                onClick={() => handleProfileUpdate('timeframe', timeline.id)}
                className={`p-3 border-2 rounded-lg text-center transition-all ${
                  businessProfile.timeframe === timeline.id
                    ? 'border-blue-500 bg-blue-50'
                    : 'border-gray-200 hover:border-gray-300'
                }`}
              >
                <div className="font-medium text-gray-900">{timeline.label}</div>
                <div className="text-xs text-gray-600">{timeline.desc}</div>
              </button>
            ))}
          </div>
        </div>
      </div>

      <div className="flex justify-center">
        <button
          onClick={generateRecommendations}
          disabled={!businessProfile.type || !businessProfile.industry || !businessProfile.size}
          className="px-8 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors flex items-center space-x-2"
        >
          <SparklesIcon className="w-5 h-5" />
          <span>Generate AI Recommendations</span>
        </button>
      </div>
    </div>
  )

  const renderAnalysis = () => (
    <div className="text-center space-y-8">
      <div className="mx-auto w-24 h-24 bg-blue-100 rounded-full flex items-center justify-center">
        <motion.div
          animate={{ rotate: 360 }}
          transition={{ duration: 2, repeat: Infinity, ease: "linear" }}
        >
          <CpuChipIcon className="w-12 h-12 text-blue-600" />
        </motion.div>
      </div>
      
      <div>
        <h2 className="text-2xl font-bold text-gray-900 mb-2">AI Analysis in Progress</h2>
        <p className="text-gray-600">Analyzing your business profile and generating personalized recommendations...</p>
      </div>

      <div className="space-y-3">
        {[
          'Analyzing business type and industry trends',
          'Calculating budget optimization strategies',
          'Matching services to your specific goals',
          'Estimating project timelines and ROI',
          'Generating custom recommendations'
        ].map((step, index) => (
          <motion.div
            key={index}
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ delay: index * 0.3 }}
            className="flex items-center justify-center space-x-2 text-sm text-gray-600"
          >
            <CheckCircleIcon className="w-4 h-4 text-green-500" />
            <span>{step}</span>
          </motion.div>
        ))}
      </div>
    </div>
  )

  const renderRecommendations = () => (
    <div className="space-y-8">
      <div className="text-center">
        <h2 className="text-2xl font-bold text-gray-900 mb-2">AI-Powered Recommendations</h2>
        <p className="text-gray-600">Based on your profile, here are our intelligent service recommendations</p>
      </div>

      <div className="space-y-6">
        {recommendations.map((rec, index) => (
          <motion.div
            key={rec.serviceId}
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: index * 0.1 }}
            className={`border-2 rounded-lg p-6 transition-all ${
              selectedServices.includes(rec.serviceId)
                ? 'border-blue-500 bg-blue-50'
                : 'border-gray-200 hover:border-gray-300'
            }`}
          >
            <div className="flex items-start justify-between mb-4">
              <div className="flex-1">
                <div className="flex items-center space-x-3 mb-2">
                  <h3 className="text-lg font-semibold text-gray-900">{rec.serviceName}</h3>
                  <span className={`px-2 py-1 text-xs font-medium rounded-full border ${getPriorityColor(rec.priority)}`}>
                    {rec.priority} priority
                  </span>
                  <div className="flex items-center space-x-1">
                    <StarIcon className="w-4 h-4 text-yellow-500 fill-current" />
                    <span className="text-sm text-gray-600">{rec.confidence}% match</span>
                  </div>
                </div>
                <p className="text-gray-600 mb-3">{rec.reasoning}</p>
              </div>
              
              <label className="flex items-center ml-4">
                <input
                  type="checkbox"
                  checked={selectedServices.includes(rec.serviceId)}
                  onChange={() => handleServiceToggle(rec.serviceId)}
                  className="h-5 w-5 text-blue-600 rounded border-gray-300 focus:ring-blue-500"
                />
              </label>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
              <div className="flex items-center space-x-2">
                <CurrencyDollarIcon className="w-5 h-5 text-green-600" />
                <div>
                  <div className="text-sm font-medium text-gray-900">
                    {formatCurrency(rec.estimatedPrice.min)} - {formatCurrency(rec.estimatedPrice.max)}
                  </div>
                  <div className="text-xs text-gray-600">Estimated Cost</div>
                </div>
              </div>
              
              <div className="flex items-center space-x-2">
                <ClockIcon className="w-5 h-5 text-blue-600" />
                <div>
                  <div className="text-sm font-medium text-gray-900">
                    {rec.timeline.min}-{rec.timeline.max} {rec.timeline.unit}
                  </div>
                  <div className="text-xs text-gray-600">Timeline</div>
                </div>
              </div>
              
              <div className="flex items-center space-x-2">
                <ChartBarIcon className="w-5 h-5 text-purple-600" />
                <div>
                  <div className="text-sm font-medium text-gray-900">{rec.roi}% ROI</div>
                  <div className="text-xs text-gray-600">Expected Return</div>
                </div>
              </div>
            </div>

            <div className="space-y-3">
              <div>
                <h4 className="text-sm font-medium text-gray-900 mb-2">Key Benefits:</h4>
                <div className="grid grid-cols-2 gap-2">
                  {rec.benefits.map((benefit, idx) => (
                    <div key={idx} className="flex items-center space-x-2">
                      <CheckCircleIcon className="w-4 h-4 text-green-500" />
                      <span className="text-sm text-gray-700">{benefit}</span>
                    </div>
                  ))}
                </div>
              </div>
              
              <div>
                <h4 className="text-sm font-medium text-gray-900 mb-2">Included Features:</h4>
                <div className="flex flex-wrap gap-2">
                  {rec.features.map((feature, idx) => (
                    <span key={idx} className="px-2 py-1 bg-gray-100 text-gray-700 text-xs rounded-full">
                      {feature}
                    </span>
                  ))}
                </div>
              </div>
            </div>
          </motion.div>
        ))}
      </div>

      {selectedServices.length > 0 && (
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-6">
          <h3 className="font-medium text-blue-900 mb-3">Selected Services Summary</h3>
          <div className="space-y-2 mb-4">
            {selectedServices.map(serviceId => {
              const rec = recommendations.find(r => r.serviceId === serviceId)
              return rec ? (
                <div key={serviceId} className="flex justify-between text-sm text-blue-800">
                  <span>{rec.serviceName}</span>
                  <span>{formatCurrency(rec.estimatedPrice.min)} - {formatCurrency(rec.estimatedPrice.max)}</span>
                </div>
              ) : null
            })}
          </div>
          <div className="flex justify-between text-sm font-medium text-blue-900 border-t border-blue-200 pt-2">
            <span>Total Estimated Investment:</span>
            <span>
              {formatCurrency(selectedServices.reduce((sum, serviceId) => {
                const rec = recommendations.find(r => r.serviceId === serviceId)
                return sum + (rec?.estimatedPrice.min || 0)
              }, 0))} - {formatCurrency(selectedServices.reduce((sum, serviceId) => {
                const rec = recommendations.find(r => r.serviceId === serviceId)
                return sum + (rec?.estimatedPrice.max || 0)
              }, 0))}
            </span>
          </div>
        </div>
      )}

      <div className="flex justify-between">
        <button
          onClick={() => setCurrentStep('profile')}
          className="px-6 py-2 text-gray-600 hover:text-gray-900 transition-colors"
        >
          Back to Profile
        </button>
        <button
          onClick={generateProposal}
          disabled={selectedServices.length === 0}
          className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors flex items-center space-x-2"
        >
          <DocumentTextIcon className="w-5 h-5" />
          <span>Generate Proposal</span>
          <ArrowRightIcon className="w-4 h-4" />
        </button>
      </div>
    </div>
  )

  const renderProposal = () => {
    const proposal = AI_RECOMMENDATIONS_ENGINE.generateProjectProposal(
      businessProfile as BusinessProfile, 
      selectedServices
    )
    
    return (
      <div className="space-y-8">
        <div className="text-center">
          <h2 className="text-2xl font-bold text-gray-900 mb-2">AI-Generated Proposal</h2>
          <p className="text-gray-600">Your personalized project proposal based on AI analysis</p>
        </div>

        <div className="bg-white border border-gray-200 rounded-lg p-8">
          <pre className="whitespace-pre-wrap text-sm text-gray-800 font-mono leading-relaxed">
            {proposal}
          </pre>
        </div>

        <div className="flex justify-between">
          <button
            onClick={() => setCurrentStep('recommendations')}
            className="px-6 py-2 text-gray-600 hover:text-gray-900 transition-colors"
          >
            Back to Recommendations
          </button>
          <div className="space-x-3">
            <button className="px-6 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors">
              Download PDF
            </button>
            <button className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
              Contact Our Team
            </button>
          </div>
        </div>
      </div>
    )
  }

  if (!session) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-50">
        <div className="text-center">
          <SparklesIcon className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <h2 className="text-2xl font-bold text-gray-900 mb-2">AI Service Recommendations</h2>
          <p className="text-gray-600 mb-6">Please sign in to get personalized service recommendations.</p>
          <a
            href="/auth/signin"
            className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            Sign In
          </a>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Progress Indicator */}
        <div className="mb-8">
          <div className="flex items-center justify-between">
            {[
              { step: 'profile', label: 'Business Profile' },
              { step: 'analysis', label: 'AI Analysis' },
              { step: 'recommendations', label: 'Recommendations' },
              { step: 'proposal', label: 'Proposal' }
            ].map((item, index) => (
              <div key={item.step} className="flex items-center">
                <div className={`w-8 h-8 rounded-full flex items-center justify-center text-sm font-medium ${
                  currentStep === item.step
                    ? 'bg-blue-600 text-white'
                    : index < ['profile', 'analysis', 'recommendations', 'proposal'].indexOf(currentStep)
                    ? 'bg-green-600 text-white'
                    : 'bg-gray-200 text-gray-600'
                }`}>
                  {index < ['profile', 'analysis', 'recommendations', 'proposal'].indexOf(currentStep) ? (
                    <CheckCircleIcon className="w-5 h-5" />
                  ) : (
                    index + 1
                  )}
                </div>
                <span className={`ml-2 text-sm font-medium ${
                  currentStep === item.step ? 'text-blue-600' : 'text-gray-500'
                }`}>
                  {item.label}
                </span>
                {index < 3 && (
                  <div className={`w-16 h-0.5 mx-4 ${
                    index < ['profile', 'analysis', 'recommendations', 'proposal'].indexOf(currentStep)
                      ? 'bg-green-600'
                      : 'bg-gray-200'
                  }`} />
                )}
              </div>
            ))}
          </div>
        </div>

        {/* Content */}
        <AnimatePresence mode="wait">
          <motion.div
            key={currentStep}
            initial={{ opacity: 0, x: 20 }}
            animate={{ opacity: 1, x: 0 }}
            exit={{ opacity: 0, x: -20 }}
            transition={{ duration: 0.3 }}
          >
            {currentStep === 'profile' && renderBusinessProfile()}
            {currentStep === 'analysis' && renderAnalysis()}
            {currentStep === 'recommendations' && renderRecommendations()}
            {currentStep === 'proposal' && renderProposal()}
          </motion.div>
        </AnimatePresence>
      </div>
    </div>
  )
}