'use client'

import { useState, useEffect } from 'react'
import { useSession } from 'next-auth/react'
import { motion, AnimatePresence } from 'framer-motion'
import {
  CreditCardIcon,
  BanknotesIcon,
  DevicePhoneMobileIcon,
  GlobeAltIcon,
  CheckCircleIcon,
  ExclamationTriangleIcon,
  InformationCircleIcon,
  XMarkIcon,
  CalendarIcon,
  ClockIcon,
  DocumentTextIcon
} from '@heroicons/react/24/outline'

interface PaymentMethod {
  id: string
  name: string
  type: 'card' | 'bank' | 'mobile' | 'crypto' | 'installment'
  provider: string
  icon: string
  fee: number
  processingTime: string
  supported: boolean
  popular?: boolean
}

interface PaymentOption {
  id: string
  name: string
  description: string
  installments?: number
  monthlyAmount?: number
  totalAmount: number
  interestRate?: number
  popular?: boolean
}

interface Invoice {
  id: string
  amount: number
  currency: string
  description: string
  items: InvoiceItem[]
  tax: number
  discount?: number
  dueDate: Date
  status: 'pending' | 'paid' | 'overdue' | 'cancelled'
}

interface InvoiceItem {
  id: string
  name: string
  quantity: number
  price: number
  total: number
}

const PAYMENT_METHODS: PaymentMethod[] = [
  {
    id: 'stripe',
    name: 'Credit/Debit Card',
    type: 'card',
    provider: 'Stripe',
    icon: '💳',
    fee: 2.9,
    processingTime: 'Instant',
    supported: true,
    popular: true
  },
  {
    id: 'paypal',
    name: 'PayPal',
    type: 'card',
    provider: 'PayPal',
    icon: '🅿️',
    fee: 3.4,
    processingTime: 'Instant',
    supported: true
  },
  {
    id: 'easypaisa',
    name: 'Easypaisa',
    type: 'mobile',
    provider: 'Easypaisa',
    icon: '📱',
    fee: 1.5,
    processingTime: '2-5 minutes',
    supported: true,
    popular: true
  },
  {
    id: 'jazzcash',
    name: 'JazzCash',
    type: 'mobile',
    provider: 'JazzCash',
    icon: '💰',
    fee: 1.5,
    processingTime: '2-5 minutes',
    supported: true
  },
  {
    id: 'hbl',
    name: 'HBL Bank Transfer',
    type: 'bank',
    provider: 'HBL',
    icon: '🏦',
    fee: 0.5,
    processingTime: '1-2 business days',
    supported: true
  },
  {
    id: 'ubl',
    name: 'UBL Bank Transfer',
    type: 'bank',
    provider: 'UBL',
    icon: '🏦',
    fee: 0.5,
    processingTime: '1-2 business days',
    supported: true
  },
  {
    id: 'meezan',
    name: 'Meezan Bank (Islamic)',
    type: 'bank',
    provider: 'Meezan',
    icon: '🕌',
    fee: 0.0,
    processingTime: '1-2 business days',
    supported: true
  },
  {
    id: 'sadapay',
    name: 'SadaPay',
    type: 'mobile',
    provider: 'SadaPay',
    icon: '💳',
    fee: 1.0,
    processingTime: 'Instant',
    supported: true
  }
]

const INSTALLMENT_OPTIONS: PaymentOption[] = [
  {
    id: 'full',
    name: 'Full Payment',
    description: 'Pay the complete amount upfront',
    totalAmount: 0, // Will be calculated
    popular: true
  },
  {
    id: 'monthly3',
    name: '3 Monthly Installments',
    description: 'Split payment over 3 months',
    installments: 3,
    totalAmount: 0,
    interestRate: 2.5
  },
  {
    id: 'monthly6',
    name: '6 Monthly Installments',
    description: 'Split payment over 6 months',
    installments: 6,
    totalAmount: 0,
    interestRate: 5.0
  },
  {
    id: 'milestone',
    name: 'Milestone Based',
    description: 'Pay based on project milestones',
    installments: 4,
    totalAmount: 0,
    interestRate: 0.0,
    popular: true
  }
]

interface AdvancedPaymentGatewayProps {
  invoice?: Invoice
  amount?: number
  description?: string
  onSuccess?: (paymentData: any) => void
  onError?: (error: string) => void
  className?: string
}

export default function AdvancedPaymentGateway({
  invoice,
  amount = 0,
  description = '',
  onSuccess,
  onError,
  className = ''
}: AdvancedPaymentGatewayProps) {
  const { data: session } = useSession()
  const [selectedMethod, setSelectedMethod] = useState<PaymentMethod | null>(null)
  const [selectedOption, setSelectedOption] = useState<PaymentOption | null>(null)
  const [processing, setProcessing] = useState(false)
  const [currentStep, setCurrentStep] = useState<'method' | 'details' | 'confirmation' | 'success'>('method')
  const [paymentData, setPaymentData] = useState<any>({})
  const [errors, setErrors] = useState<string[]>([])

  const totalAmount = invoice?.amount || amount
  const currency = invoice?.currency || 'PKR'

  useEffect(() => {
    if (INSTALLMENT_OPTIONS.length > 0) {
      const updatedOptions = INSTALLMENT_OPTIONS.map(option => ({
        ...option,
        totalAmount: calculateTotalAmount(totalAmount, option),
        monthlyAmount: option.installments ? calculateMonthlyAmount(totalAmount, option) : undefined
      }))
      
      // Set default selection
      setSelectedOption(updatedOptions[0])
    }
  }, [totalAmount])

  const calculateTotalAmount = (baseAmount: number, option: PaymentOption): number => {
    if (!option.interestRate) return baseAmount
    return baseAmount + (baseAmount * option.interestRate / 100)
  }

  const calculateMonthlyAmount = (baseAmount: number, option: PaymentOption): number => {
    if (!option.installments) return baseAmount
    const total = calculateTotalAmount(baseAmount, option)
    return total / option.installments
  }

  const calculateFees = (amount: number, method: PaymentMethod): number => {
    return amount * (method.fee / 100)
  }

  const formatCurrency = (amount: number, currency: string = 'PKR'): string => {
    if (currency === 'PKR') {
      return `Rs. ${amount.toLocaleString('en-PK')}`
    }
    return `$${amount.toLocaleString('en-US')}`
  }

  const handlePaymentSubmit = async () => {
    if (!selectedMethod || !selectedOption) {
      setErrors(['Please select a payment method and option'])
      return
    }

    setProcessing(true)
    setErrors([])

    try {
      // Different processing logic based on payment method type
      let paymentStatus = 'completed'
      let processingTime = 2000

      if (selectedMethod.type === 'bank') {
        // Bank transfers require manual verification
        paymentStatus = 'pending'
        processingTime = 1000
      } else if (selectedMethod.type === 'mobile') {
        // Mobile payments (Easypaisa, JazzCash) 
        processingTime = 3000
      } else if (selectedMethod.type === 'card') {
        // Card payments (Stripe, PayPal)
        processingTime = 2000
      }

      // Simulate payment processing
      await new Promise(resolve => setTimeout(resolve, processingTime))

      const paymentResult = {
        transactionId: `TXN-${Date.now()}`,
        amount: selectedOption.totalAmount,
        method: selectedMethod.name,
        status: paymentStatus,
        timestamp: new Date(),
        paymentType: selectedMethod.type,
        requiresVerification: selectedMethod.type === 'bank'
      }

      setPaymentData(paymentResult)
      setCurrentStep('success')
      onSuccess?.(paymentResult)
    } catch (error) {
      console.error('Payment processing error:', error)
      const errorMessage = `Payment processing failed for ${selectedMethod.name}. Please try again.`
      setErrors([errorMessage])
      onError?.(errorMessage)
    } finally {
      setProcessing(false)
    }
  }

  const renderMethodSelection = () => (
    <div className="space-y-6">
      <div>
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Select Payment Method</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          {PAYMENT_METHODS.filter(method => method.supported).map((method) => (
            <motion.div
              key={method.id}
              onClick={() => setSelectedMethod(method)}
              className={`relative p-4 border-2 rounded-lg cursor-pointer transition-all ${
                selectedMethod?.id === method.id
                  ? 'border-blue-500 bg-blue-50'
                  : 'border-gray-200 hover:border-gray-300'
              }`}
              whileHover={{ scale: 1.02 }}
              whileTap={{ scale: 0.98 }}
            >
              {method.popular && (
                <span className="absolute -top-2 -right-2 px-2 py-1 bg-green-500 text-white text-xs rounded-full">
                  Popular
                </span>
              )}
              
              <div className="flex items-center space-x-3">
                <span className="text-2xl">{method.icon}</span>
                <div className="flex-1">
                  <h4 className="font-medium text-gray-900">{method.name}</h4>
                  <p className="text-sm text-gray-600">{method.provider}</p>
                  <div className="flex items-center justify-between mt-2">
                    <span className="text-xs text-gray-500">Fee: {method.fee}%</span>
                    <span className="text-xs text-gray-500">{method.processingTime}</span>
                  </div>
                </div>
              </div>
              
              {selectedMethod?.id === method.id && (
                <CheckCircleIcon className="absolute top-2 right-2 w-5 h-5 text-blue-500" />
              )}
            </motion.div>
          ))}
        </div>
      </div>

      <div>
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Payment Options</h3>
        <div className="space-y-3">
          {INSTALLMENT_OPTIONS.map((option) => {
            const optionTotalAmount = calculateTotalAmount(totalAmount, option)
            const monthlyAmount = option.installments ? calculateMonthlyAmount(totalAmount, option) : null
            
            return (
              <motion.div
                key={option.id}
                onClick={() => setSelectedOption(option)}
                className={`p-4 border-2 rounded-lg cursor-pointer transition-all ${
                  selectedOption?.id === option.id
                    ? 'border-blue-500 bg-blue-50'
                    : 'border-gray-200 hover:border-gray-300'
                }`}
                whileHover={{ scale: 1.01 }}
              >
                <div className="flex items-center justify-between">
                  <div className="flex-1">
                    <div className="flex items-center space-x-2">
                      <h4 className="font-medium text-gray-900">{option.name}</h4>
                      {option.popular && (
                        <span className="px-2 py-1 bg-green-100 text-green-800 text-xs rounded-full">
                          Recommended
                        </span>
                      )}
                    </div>
                    <p className="text-sm text-gray-600 mt-1">{option.description}</p>
                    {monthlyAmount && (
                      <p className="text-sm text-blue-600 mt-1">
                        {formatCurrency(monthlyAmount)} per month × {option.installments} months
                      </p>
                    )}
                    {option.interestRate && option.interestRate > 0 && (
                      <p className="text-xs text-orange-600 mt-1">
                        Interest: {option.interestRate}%
                      </p>
                    )}
                  </div>
                  <div className="text-right">
                    <p className="font-semibold text-gray-900">
                      {formatCurrency(optionTotalAmount)}
                    </p>
                    {option.interestRate && option.interestRate > 0 && (
                      <p className="text-xs text-gray-500 line-through">
                        {formatCurrency(totalAmount)}
                      </p>
                    )}
                  </div>
                </div>
              </motion.div>
            )
          })}
        </div>
      </div>

      {selectedMethod && selectedOption && (
        <div className="bg-gray-50 p-4 rounded-lg">
          <h4 className="font-medium text-gray-900 mb-3">Payment Summary</h4>
          <div className="space-y-2 text-sm">
            <div className="flex justify-between">
              <span>Amount:</span>
              <span>{formatCurrency(totalAmount)}</span>
            </div>
            {selectedOption.interestRate && selectedOption.interestRate > 0 && (
              <div className="flex justify-between text-orange-600">
                <span>Interest ({selectedOption.interestRate}%):</span>
                <span>{formatCurrency(selectedOption.totalAmount - totalAmount)}</span>
              </div>
            )}
            <div className="flex justify-between text-red-600">
              <span>Processing Fee ({selectedMethod.fee}%):</span>
              <span>{formatCurrency(calculateFees(selectedOption.totalAmount, selectedMethod))}</span>
            </div>
            <div className="border-t pt-2 flex justify-between font-semibold">
              <span>Total:</span>
              <span>{formatCurrency(selectedOption.totalAmount + calculateFees(selectedOption.totalAmount, selectedMethod))}</span>
            </div>
          </div>
        </div>
      )}

      <div className="flex justify-between">
        <button
          onClick={() => setCurrentStep('method')}
          className="px-6 py-2 text-gray-600 hover:text-gray-900 transition-colors"
          disabled={processing}
        >
          Cancel
        </button>
        <button
          onClick={() => setCurrentStep('details')}
          disabled={!selectedMethod || !selectedOption || processing}
          className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
        >
          Continue to Payment
        </button>
      </div>
    </div>
  )

  const renderPaymentDetails = () => (
    <div className="space-y-6">
      <div>
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Payment Details</h3>
        <div className="bg-blue-50 p-4 rounded-lg mb-6">
          <div className="flex items-center space-x-3">
            <span className="text-2xl">{selectedMethod?.icon}</span>
            <div>
              <h4 className="font-medium text-gray-900">{selectedMethod?.name}</h4>
              <p className="text-sm text-gray-600">{selectedOption?.name}</p>
            </div>
          </div>
        </div>
      </div>

      {selectedMethod?.type === 'card' && (
        <div className="space-y-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Card Number
            </label>
            <input
              type="text"
              placeholder="1234 5678 9012 3456"
              className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              maxLength={19}
            />
          </div>
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Expiry Date
              </label>
              <input
                type="text"
                placeholder="MM/YY"
                className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                maxLength={5}
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                CVV
              </label>
              <input
                type="text"
                placeholder="123"
                className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                maxLength={4}
              />
            </div>
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Cardholder Name
            </label>
            <input
              type="text"
              placeholder="John Doe"
              className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            />
          </div>
        </div>
      )}

      {selectedMethod?.type === 'mobile' && (
        <div className="space-y-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Mobile Number
            </label>
            <input
              type="tel"
              placeholder="+92 300 1234567"
              className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            />
          </div>
          <div className="bg-yellow-50 border border-yellow-200 p-4 rounded-lg">
            <div className="flex items-start space-x-2">
              <InformationCircleIcon className="w-5 h-5 text-yellow-600 mt-0.5" />
              <div>
                <p className="text-sm text-yellow-800">
                  You will receive an SMS with payment instructions on your registered mobile number.
                </p>
              </div>
            </div>
          </div>
        </div>
      )}

      {selectedMethod?.type === 'bank' && (
        <div className="space-y-4">
          <div className="bg-blue-50 border border-blue-200 p-4 rounded-lg">
            <h4 className="font-medium text-blue-900 mb-3">Bank Transfer Details</h4>
            <div className="space-y-2 text-sm text-blue-800">
              <div className="flex justify-between">
                <span>Account Title:</span>
                <span className="font-medium">NuwairSystems (Pvt) Ltd</span>
              </div>
              <div className="flex justify-between">
                <span>Account Number:</span>
                <span className="font-medium">1234567890123456</span>
              </div>
              <div className="flex justify-between">
                <span>Bank:</span>
                <span className="font-medium">{selectedMethod.provider}</span>
              </div>
              <div className="flex justify-between">
                <span>Reference:</span>
                <span className="font-medium">PAY-{Date.now()}</span>
              </div>
            </div>
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Transaction Reference (optional)
            </label>
            <input
              type="text"
              placeholder="Enter transfer reference number"
              className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            />
          </div>
        </div>
      )}

      <div className="flex justify-between">
        <button
          onClick={() => setCurrentStep('method')}
          className="px-6 py-2 text-gray-600 hover:text-gray-900 transition-colors"
          disabled={processing}
        >
          Back
        </button>
        <button
          onClick={handlePaymentSubmit}
          disabled={processing}
          className="px-6 py-3 bg-green-600 text-white rounded-lg hover:bg-green-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors flex items-center space-x-2"
        >
          {processing ? (
            <>
              <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
              <span>Processing...</span>
            </>
          ) : (
            <span>Complete Payment</span>
          )}
        </button>
      </div>
    </div>
  )

  const renderSuccess = () => {
    const isBankTransfer = paymentData?.paymentType === 'bank'
    const isPending = paymentData?.status === 'pending'
    
    return (
      <div className="text-center space-y-6">
        <div className={`mx-auto w-16 h-16 rounded-full flex items-center justify-center ${
          isPending ? 'bg-yellow-100' : 'bg-green-100'
        }`}>
          {isPending ? (
            <ClockIcon className="w-10 h-10 text-yellow-600" />
          ) : (
            <CheckCircleIcon className="w-10 h-10 text-green-600" />
          )}
        </div>
        <div>
          <h3 className="text-xl font-semibold text-gray-900 mb-2">
            {isPending ? 'Payment Submitted!' : 'Payment Successful!'}
          </h3>
          <p className="text-gray-600">
            {isBankTransfer 
              ? 'Your payment request has been submitted. Please complete the bank transfer using the details below.'
              : 'Your payment has been processed successfully.'
            }
          </p>
        </div>
        
        {isBankTransfer && (
          <div className="bg-blue-50 border border-blue-200 p-6 rounded-lg">
            <h4 className="font-medium text-blue-900 mb-3">Complete Your Bank Transfer</h4>
            <div className="text-left space-y-3 text-sm">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <span className="text-blue-700">Account Title:</span>
                  <div className="font-medium text-blue-900">NuwairSystems (Pvt) Ltd</div>
                </div>
                <div>
                  <span className="text-blue-700">Account Number:</span>
                  <div className="font-medium text-blue-900">03451234567890</div>
                </div>
                <div>
                  <span className="text-blue-700">Bank:</span>
                  <div className="font-medium text-blue-900">{paymentData.method}</div>
                </div>
                <div>
                  <span className="text-blue-700">Amount:</span>
                  <div className="font-medium text-blue-900">{formatCurrency(paymentData.amount)}</div>
                </div>
              </div>
              <div className="mt-4 p-3 bg-yellow-50 border border-yellow-200 rounded">
                <p className="text-yellow-800 text-sm">
                  <strong>Important:</strong> Please mention transaction ID "{paymentData.transactionId}" in your transfer reference and send the receipt to payments@nuwairsystems.com
                </p>
              </div>
            </div>
          </div>
        )}
        
        <div className="bg-gray-50 p-6 rounded-lg">
          <h4 className="font-medium text-gray-900 mb-4">Transaction Details</h4>
          <div className="space-y-2 text-sm">
            <div className="flex justify-between">
              <span>Transaction ID:</span>
              <span className="font-medium">{paymentData.transactionId}</span>
            </div>
            <div className="flex justify-between">
              <span>Amount:</span>
              <span className="font-medium">{formatCurrency(paymentData.amount)}</span>
            </div>
            <div className="flex justify-between">
              <span>Method:</span>
              <span className="font-medium">{paymentData.method}</span>
            </div>
            <div className="flex justify-between">
              <span>Status:</span>
              <span className={`font-medium ${
                isPending ? 'text-yellow-600' : 'text-green-600'
              }`}>
                {isPending ? 'Pending Verification' : 'Completed'}
              </span>
            </div>
            <div className="flex justify-between">
              <span>Date:</span>
              <span className="font-medium">{paymentData.timestamp?.toLocaleDateString()}</span>
            </div>
          </div>
        </div>

        <div className="flex justify-center space-x-4">
          <button
            onClick={() => window.print()}
            className="px-6 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
          >
            Print Receipt
          </button>
          <button
            onClick={() => setCurrentStep('method')}
            className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            New Payment
          </button>
        </div>
      </div>
    )
  }

  if (!session) {
    return (
      <div className="text-center py-12">
        <CreditCardIcon className="w-16 h-16 text-gray-400 mx-auto mb-4" />
        <h3 className="text-xl font-semibold text-gray-900 mb-2">Sign In Required</h3>
        <p className="text-gray-600 mb-6">Please sign in to access payment options.</p>
        <a
          href="/auth/signin"
          className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
        >
          Sign In
        </a>
      </div>
    )
  }

  return (
    <div className={`max-w-2xl mx-auto ${className}`}>
      {/* Progress Indicator */}
      <div className="mb-8">
        <div className="flex items-center justify-between">
          {['Method', 'Details', 'Success'].map((step, index) => (
            <div key={step} className="flex items-center">
              <div className={`w-8 h-8 rounded-full flex items-center justify-center text-sm font-medium ${
                (currentStep === 'method' && index === 0) ||
                (currentStep === 'details' && index <= 1) ||
                (currentStep === 'success' && index <= 2)
                  ? 'bg-blue-600 text-white'
                  : 'bg-gray-200 text-gray-600'
              }`}>
                {index + 1}
              </div>
              <span className={`ml-2 text-sm font-medium ${
                (currentStep === 'method' && index === 0) ||
                (currentStep === 'details' && index <= 1) ||
                (currentStep === 'success' && index <= 2)
                  ? 'text-blue-600'
                  : 'text-gray-500'
              }`}>
                {step}
              </span>
              {index < 2 && (
                <div className={`w-16 h-0.5 mx-4 ${
                  (currentStep === 'details' && index === 0) ||
                  (currentStep === 'success' && index <= 1)
                    ? 'bg-blue-600'
                    : 'bg-gray-200'
                }`} />
              )}
            </div>
          ))}
        </div>
      </div>

      {/* Error Messages */}
      {errors.length > 0 && (
        <div className="mb-6 p-4 bg-red-50 border border-red-200 rounded-lg">
          <div className="flex items-start space-x-2">
            <ExclamationTriangleIcon className="w-5 h-5 text-red-600 mt-0.5" />
            <div>
              <h4 className="text-sm font-medium text-red-800">Payment Error</h4>
              <ul className="text-sm text-red-700 mt-1">
                {errors.map((error, index) => (
                  <li key={index}>• {error}</li>
                ))}
              </ul>
            </div>
          </div>
        </div>
      )}

      {/* Content */}
      <AnimatePresence mode="wait">
        <motion.div
          key={currentStep}
          initial={{ opacity: 0, x: 20 }}
          animate={{ opacity: 1, x: 0 }}
          exit={{ opacity: 0, x: -20 }}
          transition={{ duration: 0.3 }}
        >
          {currentStep === 'method' && renderMethodSelection()}
          {currentStep === 'details' && renderPaymentDetails()}
          {currentStep === 'success' && renderSuccess()}
        </motion.div>
      </AnimatePresence>
    </div>
  )
}