'use client'

import { useState, useMemo, useEffect } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { 
  MagnifyingGlassIcon,
  AdjustmentsHorizontalIcon,
  XMarkIcon,
  FunnelIcon,
  CheckIcon
} from '@heroicons/react/24/outline'

interface Service {
  id: number
  title: string
  description: string
  price: number
  duration: string
  category: string
  slug: string
  features: string[]
  popular?: boolean
}

interface SearchFiltersProps {
  services: Service[]
  onFilteredResults: (results: Service[]) => void
}

export default function AdvancedServiceSearch({ services, onFilteredResults }: SearchFiltersProps) {
  const [searchQuery, setSearchQuery] = useState('')
  const [showFilters, setShowFilters] = useState(false)
  const [filters, setFilters] = useState({
    priceRange: [0, 15000],
    categories: [] as string[],
    duration: [] as string[],
    features: [] as string[]
  })

  // Extract unique values for filter options
  const filterOptions = useMemo(() => {
    const categories = [...new Set(services.map(s => s.category))]
    const durations = [...new Set(services.map(s => s.duration))]
    const allFeatures = [...new Set(services.flatMap(s => s.features))]
    
    return {
      categories,
      durations,
      features: allFeatures
    }
  }, [services])

  // Filter services based on search query and filters
  const filteredServices = useMemo(() => {
    let results = services.filter(service => {
      // Text search
      const searchMatch = searchQuery === '' || 
        service.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
        service.description.toLowerCase().includes(searchQuery.toLowerCase()) ||
        service.features.some(feature => feature.toLowerCase().includes(searchQuery.toLowerCase()))

      // Price range filter
      const priceMatch = service.price >= filters.priceRange[0] && service.price <= filters.priceRange[1]

      // Category filter
      const categoryMatch = filters.categories.length === 0 || filters.categories.includes(service.category)

      // Duration filter
      const durationMatch = filters.duration.length === 0 || filters.duration.includes(service.duration)

      // Features filter
      const featuresMatch = filters.features.length === 0 || 
        filters.features.every(feature => service.features.includes(feature))

      return searchMatch && priceMatch && categoryMatch && durationMatch && featuresMatch
    })

    // Sort by relevance (popular services first, then by price)
    results.sort((a, b) => {
      if (a.popular && !b.popular) return -1
      if (!a.popular && b.popular) return 1
      return a.price - b.price
    })

    return results
  }, [services, searchQuery, filters])

  // Update parent component with filtered results
  useEffect(() => {
    onFilteredResults(filteredServices)
  }, [filteredServices, onFilteredResults])

  const handleCategoryFilter = (category: string) => {
    setFilters(prev => ({
      ...prev,
      categories: prev.categories.includes(category)
        ? prev.categories.filter(c => c !== category)
        : [...prev.categories, category]
    }))
  }

  const handleDurationFilter = (duration: string) => {
    setFilters(prev => ({
      ...prev,
      duration: prev.duration.includes(duration)
        ? prev.duration.filter(d => d !== duration)
        : [...prev.duration, duration]
    }))
  }

  const handleFeatureFilter = (feature: string) => {
    setFilters(prev => ({
      ...prev,
      features: prev.features.includes(feature)
        ? prev.features.filter(f => f !== feature)
        : [...prev.features, feature]
    }))
  }

  const handlePriceRangeChange = (min: number, max: number) => {
    setFilters(prev => ({
      ...prev,
      priceRange: [min, max]
    }))
  }

  const clearAllFilters = () => {
    setFilters({
      priceRange: [0, 15000],
      categories: [],
      duration: [],
      features: []
    })
    setSearchQuery('')
  }

  const activeFiltersCount = filters.categories.length + filters.duration.length + filters.features.length + 
    (filters.priceRange[0] > 0 || filters.priceRange[1] < 15000 ? 1 : 0)

  return (
    <div className="bg-white rounded-2xl shadow-lg p-6 mb-8">
      {/* Search Bar */}
      <div className="relative mb-6">
        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
          <MagnifyingGlassIcon className="h-5 w-5 text-gray-400" />
        </div>
        <input
          type="text"
          placeholder="Search services, features, or descriptions..."
          value={searchQuery}
          onChange={(e) => setSearchQuery(e.target.value)}
          className="block w-full pl-10 pr-12 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent text-gray-900 placeholder-gray-500"
        />
        <button
          onClick={() => setShowFilters(!showFilters)}
          className={`absolute inset-y-0 right-0 pr-3 flex items-center ${
            activeFiltersCount > 0 ? 'text-blue-600' : 'text-gray-400'
          }`}
        >
          <div className="flex items-center gap-2">
            {activeFiltersCount > 0 && (
              <span className="bg-blue-600 text-white text-xs rounded-full w-5 h-5 flex items-center justify-center">
                {activeFiltersCount}
              </span>
            )}
            <FunnelIcon className="h-5 w-5" />
          </div>
        </button>
      </div>

      {/* Quick Filter Chips */}
      <div className="flex flex-wrap gap-2 mb-4">
        {filterOptions.categories.slice(0, 4).map(category => (
          <button
            key={category}
            onClick={() => handleCategoryFilter(category)}
            className={`px-3 py-1 rounded-full text-sm font-medium transition-colors ${
              filters.categories.includes(category)
                ? 'bg-blue-600 text-white'
                : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
            }`}
          >
            {category}
          </button>
        ))}
      </div>

      {/* Advanced Filters Panel */}
      <AnimatePresence>
        {showFilters && (
          <motion.div
            initial={{ opacity: 0, height: 0 }}
            animate={{ opacity: 1, height: 'auto' }}
            exit={{ opacity: 0, height: 0 }}
            className="border-t border-gray-200 pt-6"
          >
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-lg font-semibold text-gray-900">Advanced Filters</h3>
              <div className="flex items-center gap-2">
                {activeFiltersCount > 0 && (
                  <button
                    onClick={clearAllFilters}
                    className="text-sm text-gray-500 hover:text-gray-700"
                  >
                    Clear All
                  </button>
                )}
                <button
                  onClick={() => setShowFilters(false)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  <XMarkIcon className="h-5 w-5" />
                </button>
              </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              {/* Price Range */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-3">
                  Price Range
                </label>
                <div className="space-y-3">
                  <input
                    type="range"
                    min="0"
                    max="15000"
                    step="100"
                    value={filters.priceRange[1]}
                    onChange={(e) => handlePriceRangeChange(filters.priceRange[0], parseInt(e.target.value))}
                    className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                  />
                  <div className="flex justify-between text-sm text-gray-600">
                    <span>${filters.priceRange[0]}</span>
                    <span>${filters.priceRange[1]}</span>
                  </div>
                  <div className="grid grid-cols-2 gap-2">
                    <button
                      onClick={() => handlePriceRangeChange(0, 1000)}
                      className={`px-2 py-1 rounded text-xs ${
                        filters.priceRange[1] <= 1000 ? 'bg-blue-100 text-blue-700' : 'bg-gray-100 text-gray-600'
                      }`}
                    >
                      Under $1K
                    </button>
                    <button
                      onClick={() => handlePriceRangeChange(1000, 5000)}
                      className={`px-2 py-1 rounded text-xs ${
                        filters.priceRange[0] >= 1000 && filters.priceRange[1] <= 5000 
                          ? 'bg-blue-100 text-blue-700' : 'bg-gray-100 text-gray-600'
                      }`}
                    >
                      $1K-$5K
                    </button>
                  </div>
                </div>
              </div>

              {/* Categories */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-3">
                  Service Categories
                </label>
                <div className="space-y-2 max-h-32 overflow-y-auto">
                  {filterOptions.categories.map(category => (
                    <label key={category} className="flex items-center">
                      <input
                        type="checkbox"
                        checked={filters.categories.includes(category)}
                        onChange={() => handleCategoryFilter(category)}
                        className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                      />
                      <span className="ml-2 text-sm text-gray-700">{category}</span>
                    </label>
                  ))}
                </div>
              </div>

              {/* Duration */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-3">
                  Project Duration
                </label>
                <div className="space-y-2">
                  {filterOptions.durations.map(duration => (
                    <label key={duration} className="flex items-center">
                      <input
                        type="checkbox"
                        checked={filters.duration.includes(duration)}
                        onChange={() => handleDurationFilter(duration)}
                        className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                      />
                      <span className="ml-2 text-sm text-gray-700">{duration}</span>
                    </label>
                  ))}
                </div>
              </div>

              {/* Popular Features */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-3">
                  Popular Features
                </label>
                <div className="space-y-2 max-h-32 overflow-y-auto">
                  {filterOptions.features.slice(0, 8).map(feature => (
                    <label key={feature} className="flex items-center">
                      <input
                        type="checkbox"
                        checked={filters.features.includes(feature)}
                        onChange={() => handleFeatureFilter(feature)}
                        className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                      />
                      <span className="ml-2 text-sm text-gray-700">{feature}</span>
                    </label>
                  ))}
                </div>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>

      {/* Results Summary */}
      <div className="flex items-center justify-between mt-4 pt-4 border-t border-gray-200">
        <div className="text-sm text-gray-600">
          {filteredServices.length === services.length 
            ? `Showing all ${services.length} services`
            : `Showing ${filteredServices.length} of ${services.length} services`
          }
        </div>
        {activeFiltersCount > 0 && (
          <div className="flex items-center gap-2">
            <span className="text-sm text-gray-600">Active filters:</span>
            <span className="bg-blue-100 text-blue-800 text-xs px-2 py-1 rounded-full">
              {activeFiltersCount}
            </span>
          </div>
        )}
      </div>
    </div>
  )
}