'use client'

import { useState, useEffect } from 'react'
import { useSession } from 'next-auth/react'
import { motion, AnimatePresence } from 'framer-motion'
import {
  DocumentTextIcon,
  PlusIcon,
  EyeIcon,
  ArrowDownTrayIcon,
  PaperAirplaneIcon,
  ClockIcon,
  CheckCircleIcon,
  ExclamationTriangleIcon,
  CreditCardIcon,
  CalendarIcon,
  UserIcon,
  BuildingOfficeIcon
} from '@heroicons/react/24/outline'

interface Invoice {
  id: string
  invoiceNumber: string
  clientId: string
  clientName: string
  clientEmail: string
  clientAddress: string
  issueDate: Date
  dueDate: Date
  status: 'draft' | 'sent' | 'paid' | 'overdue' | 'cancelled'
  items: InvoiceItem[]
  subtotal: number
  tax: number
  taxRate: number
  discount: number
  discountType: 'percentage' | 'fixed'
  total: number
  currency: string
  notes?: string
  terms?: string
  paymentMethod?: string
  paidDate?: Date
  recurringConfig?: RecurringConfig
}

interface InvoiceItem {
  id: string
  description: string
  quantity: number
  rate: number
  amount: number
  category?: string
}

interface RecurringConfig {
  enabled: boolean
  frequency: 'weekly' | 'monthly' | 'quarterly' | 'yearly'
  interval: number
  endDate?: Date
  nextInvoiceDate?: Date
}

interface InvoiceTemplate {
  id: string
  name: string
  description: string
  items: Omit<InvoiceItem, 'id'>[]
  notes?: string
  terms?: string
}

const INVOICE_TEMPLATES: InvoiceTemplate[] = [
  {
    id: 'website-dev',
    name: 'Website Development',
    description: 'Standard website development package',
    items: [
      { description: 'UI/UX Design', quantity: 1, rate: 50000, amount: 50000, category: 'Design' },
      { description: 'Frontend Development', quantity: 1, rate: 80000, amount: 80000, category: 'Development' },
      { description: 'Backend Development', quantity: 1, rate: 70000, amount: 70000, category: 'Development' },
      { description: 'Testing & QA', quantity: 1, rate: 20000, amount: 20000, category: 'Testing' }
    ],
    terms: 'Payment due within 30 days. 50% advance payment required.',
    notes: 'Includes 3 rounds of revisions and 1 month free maintenance.'
  },
  {
    id: 'mobile-app',
    name: 'Mobile App Development',
    description: 'Mobile application development package',
    items: [
      { description: 'App Design (iOS/Android)', quantity: 1, rate: 80000, amount: 80000, category: 'Design' },
      { description: 'Native Development', quantity: 1, rate: 150000, amount: 150000, category: 'Development' },
      { description: 'API Integration', quantity: 1, rate: 40000, amount: 40000, category: 'Development' },
      { description: 'Testing & Deployment', quantity: 1, rate: 30000, amount: 30000, category: 'Testing' }
    ],
    terms: 'Payment in 3 milestones: 40% advance, 40% on completion, 20% after testing.',
    notes: 'Includes app store submission and 2 months support.'
  },
  {
    id: 'hosting-maintenance',
    name: 'Hosting & Maintenance',
    description: 'Monthly hosting and maintenance services',
    items: [
      { description: 'Web Hosting', quantity: 1, rate: 2000, amount: 2000, category: 'Hosting' },
      { description: 'SSL Certificate', quantity: 1, rate: 500, amount: 500, category: 'Security' },
      { description: 'Daily Backups', quantity: 1, rate: 1000, amount: 1000, category: 'Maintenance' },
      { description: 'Security Monitoring', quantity: 1, rate: 1500, amount: 1500, category: 'Security' }
    ],
    terms: 'Monthly recurring payment. Auto-renewal unless cancelled.',
    notes: 'Includes 24/7 monitoring and priority support.'
  }
]

const MOCK_INVOICES: Invoice[] = [
  {
    id: 'inv-1',
    invoiceNumber: 'INV-2025-001',
    clientId: 'client-1',
    clientName: 'Tech Solutions Ltd',
    clientEmail: 'billing@techsolutions.com',
    clientAddress: 'Block B, Johar Town, Lahore, Pakistan',
    issueDate: new Date('2025-10-25'),
    dueDate: new Date('2025-11-25'),
    status: 'sent',
    items: INVOICE_TEMPLATES[0].items.map((item, index) => ({ id: `item-${index}`, ...item })),
    subtotal: 220000,
    tax: 38600,
    taxRate: 17.5,
    discount: 0,
    discountType: 'fixed',
    total: 258600,
    currency: 'PKR',
    notes: INVOICE_TEMPLATES[0].notes,
    terms: INVOICE_TEMPLATES[0].terms
  },
  {
    id: 'inv-2',
    invoiceNumber: 'INV-2025-002',
    clientId: 'client-2',
    clientName: 'StartupCo',
    clientEmail: 'admin@startupco.pk',
    clientAddress: 'DHA Phase 5, Karachi, Pakistan',
    issueDate: new Date('2025-10-20'),
    dueDate: new Date('2025-11-20'),
    status: 'paid',
    items: INVOICE_TEMPLATES[2].items.map((item, index) => ({ id: `item-${index}`, ...item })),
    subtotal: 5000,
    tax: 875,
    taxRate: 17.5,
    discount: 500,
    discountType: 'fixed',
    total: 5375,
    currency: 'PKR',
    paidDate: new Date('2025-10-22'),
    paymentMethod: 'Bank Transfer',
    recurringConfig: {
      enabled: true,
      frequency: 'monthly',
      interval: 1,
      nextInvoiceDate: new Date('2025-11-20')
    }
  }
]

export default function AutomatedInvoicing() {
  const { data: session } = useSession()
  const [invoices, setInvoices] = useState<Invoice[]>([])
  const [selectedTab, setSelectedTab] = useState<'invoices' | 'create' | 'templates' | 'settings'>('invoices')
  const [selectedInvoice, setSelectedInvoice] = useState<Invoice | null>(null)
  const [showCreateModal, setShowCreateModal] = useState(false)
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    if (session?.user) {
      // Simulate API call
      setTimeout(() => {
        setInvoices(MOCK_INVOICES)
        setLoading(false)
      }, 1000)
    }
  }, [session])

  const getStatusColor = (status: Invoice['status']) => {
    switch (status) {
      case 'draft': return 'bg-gray-100 text-gray-800'
      case 'sent': return 'bg-blue-100 text-blue-800'
      case 'paid': return 'bg-green-100 text-green-800'
      case 'overdue': return 'bg-red-100 text-red-800'
      case 'cancelled': return 'bg-orange-100 text-orange-800'
      default: return 'bg-gray-100 text-gray-800'
    }
  }

  const getStatusIcon = (status: Invoice['status']) => {
    switch (status) {
      case 'draft': return <DocumentTextIcon className="w-4 h-4" />
      case 'sent': return <PaperAirplaneIcon className="w-4 h-4" />
      case 'paid': return <CheckCircleIcon className="w-4 h-4" />
      case 'overdue': return <ExclamationTriangleIcon className="w-4 h-4" />
      case 'cancelled': return <ClockIcon className="w-4 h-4" />
      default: return <DocumentTextIcon className="w-4 h-4" />
    }
  }

  const formatCurrency = (amount: number, currency: string = 'PKR'): string => {
    if (currency === 'PKR') {
      return `Rs. ${amount.toLocaleString('en-PK')}`
    }
    return `$${amount.toLocaleString('en-US')}`
  }

  const calculateOverdueDays = (dueDate: Date): number => {
    const today = new Date()
    const diffTime = today.getTime() - dueDate.getTime()
    return Math.ceil(diffTime / (1000 * 60 * 60 * 24))
  }

  const handleSendInvoice = async (invoiceId: string) => {
    setInvoices(prev => prev.map(inv => 
      inv.id === invoiceId 
        ? { ...inv, status: 'sent' as const }
        : inv
    ))
  }

  const handleMarkAsPaid = async (invoiceId: string) => {
    setInvoices(prev => prev.map(inv => 
      inv.id === invoiceId 
        ? { 
            ...inv, 
            status: 'paid' as const, 
            paidDate: new Date(),
            paymentMethod: 'Manual Entry' 
          }
        : inv
    ))
  }

  const renderInvoicesList = () => (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h3 className="text-lg font-semibold text-gray-900">All Invoices</h3>
        <button
          onClick={() => setShowCreateModal(true)}
          className="flex items-center space-x-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
        >
          <PlusIcon className="w-4 h-4" />
          <span>Create Invoice</span>
        </button>
      </div>

      {/* Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        {[
          { label: 'Total Invoices', value: invoices.length, color: 'blue' },
          { label: 'Pending Payment', value: invoices.filter(inv => inv.status === 'sent').length, color: 'yellow' },
          { label: 'Paid', value: invoices.filter(inv => inv.status === 'paid').length, color: 'green' },
          { label: 'Overdue', value: invoices.filter(inv => inv.status === 'overdue').length, color: 'red' }
        ].map((stat, index) => (
          <div key={index} className="bg-white p-4 rounded-lg border border-gray-200">
            <p className="text-sm text-gray-600">{stat.label}</p>
            <p className={`text-2xl font-semibold ${
              stat.color === 'blue' ? 'text-blue-600' :
              stat.color === 'yellow' ? 'text-yellow-600' :
              stat.color === 'green' ? 'text-green-600' :
              'text-red-600'
            }`}>
              {stat.value}
            </p>
          </div>
        ))}
      </div>

      {/* Invoices Table */}
      <div className="bg-white rounded-lg border border-gray-200 overflow-hidden">
        <div className="overflow-x-auto">
          <table className="min-w-full divide-y divide-gray-200">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Invoice
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Client
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Amount
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Status
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Due Date
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Actions
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {invoices.map((invoice) => (
                <motion.tr
                  key={invoice.id}
                  initial={{ opacity: 0 }}
                  animate={{ opacity: 1 }}
                  className="hover:bg-gray-50"
                >
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div>
                      <div className="text-sm font-medium text-gray-900">{invoice.invoiceNumber}</div>
                      <div className="text-sm text-gray-500">
                        {invoice.issueDate.toLocaleDateString()}
                        {invoice.recurringConfig?.enabled && (
                          <span className="ml-2 px-2 py-1 bg-purple-100 text-purple-800 text-xs rounded-full">
                            Recurring
                          </span>
                        )}
                      </div>
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div>
                      <div className="text-sm font-medium text-gray-900">{invoice.clientName}</div>
                      <div className="text-sm text-gray-500">{invoice.clientEmail}</div>
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                    {formatCurrency(invoice.total, invoice.currency)}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`inline-flex items-center space-x-1 px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(invoice.status)}`}>
                      {getStatusIcon(invoice.status)}
                      <span className="capitalize">{invoice.status}</span>
                    </span>
                    {invoice.status === 'overdue' && (
                      <div className="text-xs text-red-600 mt-1">
                        {calculateOverdueDays(invoice.dueDate)} days overdue
                      </div>
                    )}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    {invoice.dueDate.toLocaleDateString()}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm space-x-2">
                    <button
                      onClick={() => setSelectedInvoice(invoice)}
                      className="text-blue-600 hover:text-blue-900"
                      title="View Invoice"
                    >
                      <EyeIcon className="w-4 h-4" />
                    </button>
                    <button
                      className="text-green-600 hover:text-green-900"
                      title="Download PDF"
                    >
                      <ArrowDownTrayIcon className="w-4 h-4" />
                    </button>
                    {invoice.status === 'draft' && (
                      <button
                        onClick={() => handleSendInvoice(invoice.id)}
                        className="text-purple-600 hover:text-purple-900"
                        title="Send Invoice"
                      >
                        <PaperAirplaneIcon className="w-4 h-4" />
                      </button>
                    )}
                    {(invoice.status === 'sent' || invoice.status === 'overdue') && (
                      <button
                        onClick={() => handleMarkAsPaid(invoice.id)}
                        className="text-green-600 hover:text-green-900"
                        title="Mark as Paid"
                      >
                        <CreditCardIcon className="w-4 h-4" />
                      </button>
                    )}
                  </td>
                </motion.tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    </div>
  )

  const renderCreateInvoice = () => (
    <div className="space-y-6">
      <h3 className="text-lg font-semibold text-gray-900">Create New Invoice</h3>
      
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
        <div className="space-y-6">
          {/* Client Information */}
          <div className="bg-white p-6 rounded-lg border border-gray-200">
            <h4 className="font-medium text-gray-900 mb-4 flex items-center">
              <UserIcon className="w-5 h-5 mr-2" />
              Client Information
            </h4>
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Client Name</label>
                <input
                  type="text"
                  className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Enter client name"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Email</label>
                <input
                  type="email"
                  className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="client@example.com"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Address</label>
                <textarea
                  rows={3}
                  className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Enter client address"
                />
              </div>
            </div>
          </div>

          {/* Invoice Details */}
          <div className="bg-white p-6 rounded-lg border border-gray-200">
            <h4 className="font-medium text-gray-900 mb-4 flex items-center">
              <DocumentTextIcon className="w-5 h-5 mr-2" />
              Invoice Details
            </h4>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Invoice Number</label>
                <input
                  type="text"
                  value={`INV-${new Date().getFullYear()}-${String(invoices.length + 1).padStart(3, '0')}`}
                  className="w-full p-3 border border-gray-300 rounded-lg bg-gray-50"
                  readOnly
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Currency</label>
                <select className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                  <option value="PKR">PKR (Pakistani Rupee)</option>
                  <option value="USD">USD (US Dollar)</option>
                </select>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Issue Date</label>
                <input
                  type="date"
                  defaultValue={new Date().toISOString().split('T')[0]}
                  className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Due Date</label>
                <input
                  type="date"
                  defaultValue={new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]}
                  className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                />
              </div>
            </div>
          </div>
        </div>

        <div className="space-y-6">
          {/* Templates */}
          <div className="bg-white p-6 rounded-lg border border-gray-200">
            <h4 className="font-medium text-gray-900 mb-4">Quick Templates</h4>
            <div className="space-y-2">
              {INVOICE_TEMPLATES.map((template) => (
                <button
                  key={template.id}
                  className="w-full text-left p-3 border border-gray-200 rounded-lg hover:bg-blue-50 hover:border-blue-300 transition-colors"
                >
                  <div className="font-medium text-gray-900">{template.name}</div>
                  <div className="text-sm text-gray-600">{template.description}</div>
                  <div className="text-sm text-blue-600 mt-1">
                    {formatCurrency(template.items.reduce((sum, item) => sum + item.amount, 0))}
                  </div>
                </button>
              ))}
            </div>
          </div>

          {/* Recurring Settings */}
          <div className="bg-white p-6 rounded-lg border border-gray-200">
            <h4 className="font-medium text-gray-900 mb-4 flex items-center">
              <CalendarIcon className="w-5 h-5 mr-2" />
              Recurring Invoice
            </h4>
            <div className="space-y-4">
              <label className="flex items-center">
                <input type="checkbox" className="h-4 w-4 text-blue-600 rounded border-gray-300 focus:ring-blue-500" />
                <span className="ml-2 text-sm text-gray-700">Make this a recurring invoice</span>
              </label>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Frequency</label>
                  <select className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                    <option value="monthly">Monthly</option>
                    <option value="quarterly">Quarterly</option>
                    <option value="yearly">Yearly</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">End Date</label>
                  <input
                    type="date"
                    className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  />
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Action Buttons */}
      <div className="flex justify-between">
        <button
          onClick={() => setSelectedTab('invoices')}
          className="px-6 py-2 text-gray-600 hover:text-gray-900 transition-colors"
        >
          Cancel
        </button>
        <div className="space-x-3">
          <button className="px-6 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors">
            Save as Draft
          </button>
          <button className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
            Create & Send
          </button>
        </div>
      </div>
    </div>
  )

  if (!session) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-50">
        <div className="text-center">
          <DocumentTextIcon className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <h2 className="text-2xl font-bold text-gray-900 mb-2">Invoice Management</h2>
          <p className="text-gray-600 mb-6">Please sign in to manage your invoices.</p>
          <a
            href="/auth/signin"
            className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            Sign In
          </a>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-gray-900">Invoice Management</h1>
          <p className="text-gray-600 mt-2">Create, manage, and track your invoices with automated billing.</p>
        </div>

        {/* Tab Navigation */}
        <div className="border-b border-gray-200 mb-8">
          <nav className="-mb-px flex space-x-8">
            {[
              { key: 'invoices', label: 'All Invoices', icon: DocumentTextIcon },
              { key: 'create', label: 'Create Invoice', icon: PlusIcon }
            ].map(tab => (
              <button
                key={tab.key}
                onClick={() => setSelectedTab(tab.key as any)}
                className={`flex items-center space-x-2 py-2 px-1 border-b-2 font-medium text-sm ${
                  selectedTab === tab.key
                    ? 'border-blue-500 text-blue-600'
                    : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
              >
                <tab.icon className="w-5 h-5" />
                <span>{tab.label}</span>
              </button>
            ))}
          </nav>
        </div>

        {/* Tab Content */}
        <AnimatePresence mode="wait">
          <motion.div
            key={selectedTab}
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            exit={{ opacity: 0, y: -20 }}
            transition={{ duration: 0.2 }}
          >
            {selectedTab === 'invoices' && renderInvoicesList()}
            {selectedTab === 'create' && renderCreateInvoice()}
          </motion.div>
        </AnimatePresence>
      </div>

      {/* Invoice Preview Modal */}
      {selectedInvoice && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <motion.div
            initial={{ scale: 0.9, opacity: 0 }}
            animate={{ scale: 1, opacity: 1 }}
            className="bg-white rounded-lg max-w-4xl w-full max-h-[90vh] overflow-y-auto"
          >
            <div className="p-6 border-b border-gray-200">
              <div className="flex items-center justify-between">
                <h3 className="text-lg font-semibold text-gray-900">Invoice Preview</h3>
                <button
                  onClick={() => setSelectedInvoice(null)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  <span className="sr-only">Close</span>
                  ×
                </button>
              </div>
            </div>
            
            <div className="p-6">
              {/* Invoice content would go here */}
              <div className="text-center py-12">
                <DocumentTextIcon className="w-16 h-16 text-gray-400 mx-auto mb-4" />
                <h4 className="text-lg font-semibold text-gray-900 mb-2">
                  {selectedInvoice.invoiceNumber}
                </h4>
                <p className="text-gray-600">
                  Full invoice preview would be implemented here with PDF generation.
                </p>
              </div>
            </div>
          </motion.div>
        </div>
      )}
    </div>
  )
}