'use client'

import { useState, useEffect } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import {
  CalculatorIcon,
  CurrencyDollarIcon,
  ClockIcon,
  ChartBarIcon,
  CheckCircleIcon,
  InformationCircleIcon,
  SparklesIcon,
  AdjustmentsHorizontalIcon
} from '@heroicons/react/24/outline'

interface ProjectRequirement {
  id: string
  category: string
  feature: string
  description: string
  basePrice: number
  complexity: 'basic' | 'standard' | 'advanced' | 'enterprise'
  timeImpact: number // in days
  dependencies: string[]
  popular?: boolean
}

interface PriceEstimate {
  basePrice: number
  complexityMultiplier: number
  featuresTotal: number
  urgencyMultiplier: number
  maintenanceAnnual: number
  total: number
  timeline: {
    development: number
    testing: number
    deployment: number
    total: number
  }
  breakdown: PriceBreakdownItem[]
}

interface PriceBreakdownItem {
  category: string
  items: string[]
  price: number
  percentage: number
}

const PROJECT_REQUIREMENTS: Record<string, ProjectRequirement[]> = {
  'website-development': [
    {
      id: 'responsive-design',
      category: 'Design',
      feature: 'Responsive Design',
      description: 'Mobile-friendly design that adapts to all screen sizes',
      basePrice: 15000,
      complexity: 'standard',
      timeImpact: 5,
      dependencies: [],
      popular: true
    },
    {
      id: 'cms-integration',
      category: 'Development',
      feature: 'Content Management System',
      description: 'Easy-to-use admin panel for content updates',
      basePrice: 25000,
      complexity: 'standard',
      timeImpact: 8,
      dependencies: ['responsive-design']
    },
    {
      id: 'ecommerce-functionality',
      category: 'Development',
      feature: 'E-commerce Functionality',
      description: 'Shopping cart, payment gateway, inventory management',
      basePrice: 75000,
      complexity: 'advanced',
      timeImpact: 20,
      dependencies: ['cms-integration', 'payment-integration']
    },
    {
      id: 'seo-optimization',
      category: 'Marketing',
      feature: 'SEO Optimization',
      description: 'Search engine optimization for better rankings',
      basePrice: 20000,
      complexity: 'standard',
      timeImpact: 7,
      dependencies: [],
      popular: true
    },
    {
      id: 'payment-integration',
      category: 'Development',
      feature: 'Payment Gateway Integration',
      description: 'Secure payment processing with multiple methods',
      basePrice: 30000,
      complexity: 'advanced',
      timeImpact: 10,
      dependencies: []
    },
    {
      id: 'user-authentication',
      category: 'Development',
      feature: 'User Authentication',
      description: 'User registration, login, and profile management',
      basePrice: 20000,
      complexity: 'standard',
      timeImpact: 6,
      dependencies: []
    },
    {
      id: 'multilingual-support',
      category: 'Development',
      feature: 'Multi-language Support',
      description: 'Support for multiple languages including Urdu',
      basePrice: 35000,
      complexity: 'advanced',
      timeImpact: 12,
      dependencies: ['cms-integration']
    },
    {
      id: 'analytics-integration',
      category: 'Marketing',
      feature: 'Analytics Integration',
      description: 'Google Analytics and advanced tracking setup',
      basePrice: 10000,
      complexity: 'basic',
      timeImpact: 3,
      dependencies: []
    }
  ],
  'mobile-app-development': [
    {
      id: 'native-ios',
      category: 'Development',
      feature: 'Native iOS App',
      description: 'Native iOS application with platform-specific features',
      basePrice: 150000,
      complexity: 'advanced',
      timeImpact: 30,
      dependencies: []
    },
    {
      id: 'native-android',
      category: 'Development',
      feature: 'Native Android App',
      description: 'Native Android application with platform-specific features',
      basePrice: 150000,
      complexity: 'advanced',
      timeImpact: 30,
      dependencies: []
    },
    {
      id: 'cross-platform',
      category: 'Development',
      feature: 'Cross-Platform Development',
      description: 'Single codebase for both iOS and Android',
      basePrice: 200000,
      complexity: 'advanced',
      timeImpact: 35,
      dependencies: []
    },
    {
      id: 'push-notifications',
      category: 'Features',
      feature: 'Push Notifications',
      description: 'Real-time notifications to engage users',
      basePrice: 25000,
      complexity: 'standard',
      timeImpact: 7,
      dependencies: []
    },
    {
      id: 'offline-functionality',
      category: 'Features',
      feature: 'Offline Functionality',
      description: 'App works without internet connection',
      basePrice: 40000,
      complexity: 'advanced',
      timeImpact: 15,
      dependencies: []
    },
    {
      id: 'app-store-optimization',
      category: 'Marketing',
      feature: 'App Store Optimization',
      description: 'ASO for better app store visibility',
      basePrice: 15000,
      complexity: 'standard',
      timeImpact: 5,
      dependencies: []
    }
  ]
}

const COMPLEXITY_MULTIPLIERS = {
  basic: 1.0,
  standard: 1.2,
  advanced: 1.5,
  enterprise: 2.0
}

const URGENCY_MULTIPLIERS = {
  flexible: 1.0,
  standard: 1.2,
  urgent: 1.5,
  rush: 2.0
}

const AI_PRICING_ENGINE = {
  calculateEstimate: (
    serviceType: string,
    selectedFeatures: string[],
    complexity: keyof typeof COMPLEXITY_MULTIPLIERS,
    urgency: keyof typeof URGENCY_MULTIPLIERS,
    teamSize: number = 1
  ): PriceEstimate => {
    const requirements = PROJECT_REQUIREMENTS[serviceType] || []
    const selectedReqs = requirements.filter(req => selectedFeatures.includes(req.id))
    
    // Base calculations
    const basePrice = selectedReqs.reduce((sum, req) => sum + req.basePrice, 0)
    const complexityMultiplier = COMPLEXITY_MULTIPLIERS[complexity]
    const urgencyMultiplier = URGENCY_MULTIPLIERS[urgency]
    const featuresTotal = basePrice * complexityMultiplier
    const maintenanceAnnual = featuresTotal * 0.2 // 20% annual maintenance
    
    // Timeline calculations
    const developmentDays = selectedReqs.reduce((sum, req) => sum + req.timeImpact, 0)
    const testingDays = Math.ceil(developmentDays * 0.3)
    const deploymentDays = Math.ceil(developmentDays * 0.1)
    const totalDays = Math.ceil((developmentDays + testingDays + deploymentDays) / urgencyMultiplier)
    
    const total = featuresTotal * urgencyMultiplier
    
    // Breakdown
    const breakdown: PriceBreakdownItem[] = [
      {
        category: 'Development',
        items: selectedReqs.filter(req => req.category === 'Development').map(req => req.feature),
        price: selectedReqs.filter(req => req.category === 'Development').reduce((sum, req) => sum + req.basePrice, 0) * complexityMultiplier,
        percentage: 0
      },
      {
        category: 'Design',
        items: selectedReqs.filter(req => req.category === 'Design').map(req => req.feature),
        price: selectedReqs.filter(req => req.category === 'Design').reduce((sum, req) => sum + req.basePrice, 0) * complexityMultiplier,
        percentage: 0
      },
      {
        category: 'Features',
        items: selectedReqs.filter(req => req.category === 'Features').map(req => req.feature),
        price: selectedReqs.filter(req => req.category === 'Features').reduce((sum, req) => sum + req.basePrice, 0) * complexityMultiplier,
        percentage: 0
      },
      {
        category: 'Marketing',
        items: selectedReqs.filter(req => req.category === 'Marketing').map(req => req.feature),
        price: selectedReqs.filter(req => req.category === 'Marketing').reduce((sum, req) => sum + req.basePrice, 0) * complexityMultiplier,
        percentage: 0
      }
    ].filter(item => item.items.length > 0)
    
    // Calculate percentages
    breakdown.forEach(item => {
      item.percentage = Math.round((item.price / featuresTotal) * 100)
    })
    
    return {
      basePrice,
      complexityMultiplier,
      featuresTotal,
      urgencyMultiplier,
      maintenanceAnnual,
      total,
      timeline: {
        development: developmentDays,
        testing: testingDays,
        deployment: deploymentDays,
        total: totalDays
      },
      breakdown
    }
  },

  generateRecommendations: (serviceType: string, budget: number): string[] => {
    const requirements = PROJECT_REQUIREMENTS[serviceType] || []
    const popularFeatures = requirements.filter(req => req.popular).map(req => req.id)
    
    // Budget-based recommendations
    const affordableFeatures = requirements
      .filter(req => req.basePrice <= budget * 0.3) // Each feature should be max 30% of budget
      .sort((a, b) => b.basePrice - a.basePrice)
      .slice(0, 5)
      .map(req => req.id)
    
    return [...new Set([...popularFeatures, ...affordableFeatures])]
  }
}

export default function AutomatedPriceEstimator() {
  const [selectedService, setSelectedService] = useState<string>('website-development')
  const [selectedFeatures, setSelectedFeatures] = useState<string[]>([])
  const [complexity, setComplexity] = useState<keyof typeof COMPLEXITY_MULTIPLIERS>('standard')
  const [urgency, setUrgency] = useState<keyof typeof URGENCY_MULTIPLIERS>('standard')
  const [estimate, setEstimate] = useState<PriceEstimate | null>(null)
  const [showRecommendations, setShowRecommendations] = useState(false)

  useEffect(() => {
    if (selectedFeatures.length > 0) {
      const newEstimate = AI_PRICING_ENGINE.calculateEstimate(
        selectedService,
        selectedFeatures,
        complexity,
        urgency
      )
      setEstimate(newEstimate)
    } else {
      setEstimate(null)
    }
  }, [selectedService, selectedFeatures, complexity, urgency])

  const formatCurrency = (amount: number): string => {
    return `Rs. ${amount.toLocaleString('en-PK')}`
  }

  const handleFeatureToggle = (featureId: string) => {
    setSelectedFeatures(prev =>
      prev.includes(featureId)
        ? prev.filter(id => id !== featureId)
        : [...prev, featureId]
    )
  }

  const getRecommendedFeatures = () => {
    const budget = estimate?.total || 100000
    return AI_PRICING_ENGINE.generateRecommendations(selectedService, budget)
  }

  const requirements = PROJECT_REQUIREMENTS[selectedService] || []

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center mb-8">
          <CalculatorIcon className="w-16 h-16 text-blue-600 mx-auto mb-4" />
          <h1 className="text-3xl font-bold text-gray-900 mb-2">AI Price Estimator</h1>
          <p className="text-gray-600">Get instant, accurate project estimates using our intelligent pricing engine</p>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Configuration Panel */}
          <div className="lg:col-span-2 space-y-6">
            {/* Service Type Selection */}
            <div className="bg-white rounded-lg p-6 shadow-sm">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Select Service Type</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                {[
                  { id: 'website-development', name: 'Website Development', desc: 'Custom websites and web apps' },
                  { id: 'mobile-app-development', name: 'Mobile App Development', desc: 'iOS and Android applications' }
                ].map((service) => (
                  <button
                    key={service.id}
                    onClick={() => {
                      setSelectedService(service.id)
                      setSelectedFeatures([])
                    }}
                    className={`p-4 border-2 rounded-lg text-left transition-all ${
                      selectedService === service.id
                        ? 'border-blue-500 bg-blue-50'
                        : 'border-gray-200 hover:border-gray-300'
                    }`}
                  >
                    <h4 className="font-medium text-gray-900">{service.name}</h4>
                    <p className="text-sm text-gray-600 mt-1">{service.desc}</p>
                  </button>
                ))}
              </div>
            </div>

            {/* Features Selection */}
            <div className="bg-white rounded-lg p-6 shadow-sm">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold text-gray-900">Select Features</h3>
                <button
                  onClick={() => setShowRecommendations(!showRecommendations)}
                  className="flex items-center space-x-2 text-blue-600 hover:text-blue-800 text-sm"
                >
                  <SparklesIcon className="w-4 h-4" />
                  <span>AI Recommendations</span>
                </button>
              </div>

              {showRecommendations && (
                <motion.div
                  initial={{ opacity: 0, height: 0 }}
                  animate={{ opacity: 1, height: 'auto' }}
                  className="mb-4 p-4 bg-blue-50 border border-blue-200 rounded-lg"
                >
                  <h4 className="font-medium text-blue-900 mb-2">Recommended for your project:</h4>
                  <div className="flex flex-wrap gap-2">
                    {getRecommendedFeatures().map(featureId => {
                      const feature = requirements.find(req => req.id === featureId)
                      return feature ? (
                        <button
                          key={featureId}
                          onClick={() => handleFeatureToggle(featureId)}
                          className="px-3 py-1 bg-blue-100 text-blue-800 text-sm rounded-full hover:bg-blue-200 transition-colors"
                        >
                          {feature.feature}
                        </button>
                      ) : null
                    })}
                  </div>
                </motion.div>
              )}

              <div className="space-y-3">
                {requirements.map((req) => (
                  <motion.div
                    key={req.id}
                    className={`p-4 border-2 rounded-lg cursor-pointer transition-all ${
                      selectedFeatures.includes(req.id)
                        ? 'border-blue-500 bg-blue-50'
                        : 'border-gray-200 hover:border-gray-300'
                    }`}
                    onClick={() => handleFeatureToggle(req.id)}
                    whileHover={{ scale: 1.01 }}
                    whileTap={{ scale: 0.99 }}
                  >
                    <div className="flex items-start justify-between">
                      <div className="flex-1">
                        <div className="flex items-center space-x-3 mb-2">
                          <h4 className="font-medium text-gray-900">{req.feature}</h4>
                          {req.popular && (
                            <span className="px-2 py-1 bg-green-100 text-green-800 text-xs rounded-full">
                              Popular
                            </span>
                          )}
                          <span className="px-2 py-1 bg-gray-100 text-gray-600 text-xs rounded-full">
                            {req.category}
                          </span>
                        </div>
                        <p className="text-sm text-gray-600 mb-2">{req.description}</p>
                        <div className="flex items-center space-x-4 text-sm text-gray-500">
                          <span>Base: {formatCurrency(req.basePrice)}</span>
                          <span>Time: {req.timeImpact} days</span>
                        </div>
                      </div>
                      <input
                        type="checkbox"
                        checked={selectedFeatures.includes(req.id)}
                        onChange={() => handleFeatureToggle(req.id)}
                        className="h-5 w-5 text-blue-600 rounded border-gray-300 focus:ring-blue-500"
                      />
                    </div>
                  </motion.div>
                ))}
              </div>
            </div>

            {/* Project Parameters */}
            <div className="bg-white rounded-lg p-6 shadow-sm">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">Project Parameters</h3>
              
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-3">Project Complexity</label>
                  <div className="space-y-2">
                    {[
                      { id: 'basic', label: 'Basic', desc: 'Simple implementation', multiplier: '1.0x' },
                      { id: 'standard', label: 'Standard', desc: 'Professional features', multiplier: '1.2x' },
                      { id: 'advanced', label: 'Advanced', desc: 'Complex functionality', multiplier: '1.5x' },
                      { id: 'enterprise', label: 'Enterprise', desc: 'Large-scale solution', multiplier: '2.0x' }
                    ].map((level) => (
                      <label key={level.id} className="flex items-center space-x-3">
                        <input
                          type="radio"
                          checked={complexity === level.id}
                          onChange={() => setComplexity(level.id as any)}
                          className="h-4 w-4 text-blue-600 border-gray-300 focus:ring-blue-500"
                        />
                        <div>
                          <span className="text-sm font-medium text-gray-900">{level.label}</span>
                          <span className="text-xs text-blue-600 ml-2">{level.multiplier}</span>
                          <p className="text-xs text-gray-600">{level.desc}</p>
                        </div>
                      </label>
                    ))}
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-3">Timeline Urgency</label>
                  <div className="space-y-2">
                    {[
                      { id: 'flexible', label: 'Flexible', desc: '3+ months', multiplier: '1.0x' },
                      { id: 'standard', label: 'Standard', desc: '2-3 months', multiplier: '1.2x' },
                      { id: 'urgent', label: 'Urgent', desc: '1-2 months', multiplier: '1.5x' },
                      { id: 'rush', label: 'Rush', desc: 'Under 1 month', multiplier: '2.0x' }
                    ].map((timeline) => (
                      <label key={timeline.id} className="flex items-center space-x-3">
                        <input
                          type="radio"
                          checked={urgency === timeline.id}
                          onChange={() => setUrgency(timeline.id as any)}
                          className="h-4 w-4 text-blue-600 border-gray-300 focus:ring-blue-500"
                        />
                        <div>
                          <span className="text-sm font-medium text-gray-900">{timeline.label}</span>
                          <span className="text-xs text-orange-600 ml-2">{timeline.multiplier}</span>
                          <p className="text-xs text-gray-600">{timeline.desc}</p>
                        </div>
                      </label>
                    ))}
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Estimate Display */}
          <div className="space-y-6">
            {estimate ? (
              <>
                {/* Price Summary */}
                <div className="bg-white rounded-lg p-6 shadow-sm">
                  <h3 className="text-lg font-semibold text-gray-900 mb-4">Price Estimate</h3>
                  
                  <div className="space-y-4">
                    <div className="p-4 bg-blue-50 rounded-lg">
                      <div className="text-center">
                        <div className="text-3xl font-bold text-blue-900">
                          {formatCurrency(estimate.total)}
                        </div>
                        <div className="text-sm text-blue-600">Total Project Cost</div>
                      </div>
                    </div>

                    <div className="space-y-3">
                      <div className="flex justify-between text-sm">
                        <span className="text-gray-600">Base Features:</span>
                        <span className="font-medium">{formatCurrency(estimate.basePrice)}</span>
                      </div>
                      <div className="flex justify-between text-sm">
                        <span className="text-gray-600">Complexity ({complexity}):</span>
                        <span className="font-medium">×{estimate.complexityMultiplier}</span>
                      </div>
                      <div className="flex justify-between text-sm">
                        <span className="text-gray-600">Urgency ({urgency}):</span>
                        <span className="font-medium">×{estimate.urgencyMultiplier}</span>
                      </div>
                      <div className="border-t pt-3 flex justify-between">
                        <span className="font-medium text-gray-900">Subtotal:</span>
                        <span className="font-bold text-gray-900">{formatCurrency(estimate.total)}</span>
                      </div>
                    </div>

                    <div className="p-3 bg-yellow-50 border border-yellow-200 rounded-lg">
                      <div className="flex items-center space-x-2">
                        <InformationCircleIcon className="w-4 h-4 text-yellow-600" />
                        <span className="text-sm text-yellow-800">
                          Annual maintenance: {formatCurrency(estimate.maintenanceAnnual)}
                        </span>
                      </div>
                    </div>
                  </div>
                </div>

                {/* Timeline */}
                <div className="bg-white rounded-lg p-6 shadow-sm">
                  <h3 className="text-lg font-semibold text-gray-900 mb-4">Timeline</h3>
                  
                  <div className="space-y-3">
                    <div className="flex justify-between items-center">
                      <span className="text-sm text-gray-600">Development:</span>
                      <span className="font-medium">{estimate.timeline.development} days</span>
                    </div>
                    <div className="flex justify-between items-center">
                      <span className="text-sm text-gray-600">Testing:</span>
                      <span className="font-medium">{estimate.timeline.testing} days</span>
                    </div>
                    <div className="flex justify-between items-center">
                      <span className="text-sm text-gray-600">Deployment:</span>
                      <span className="font-medium">{estimate.timeline.deployment} days</span>
                    </div>
                    <div className="border-t pt-3 flex justify-between items-center">
                      <span className="font-medium text-gray-900">Total Timeline:</span>
                      <span className="font-bold text-gray-900">{estimate.timeline.total} days</span>
                    </div>
                  </div>
                </div>

                {/* Breakdown */}
                {estimate.breakdown.length > 0 && (
                  <div className="bg-white rounded-lg p-6 shadow-sm">
                    <h3 className="text-lg font-semibold text-gray-900 mb-4">Cost Breakdown</h3>
                    
                    <div className="space-y-4">
                      {estimate.breakdown.map((item, index) => (
                        <div key={index}>
                          <div className="flex justify-between items-center mb-2">
                            <span className="font-medium text-gray-900">{item.category}</span>
                            <span className="text-sm text-gray-600">{item.percentage}%</span>
                          </div>
                          <div className="text-sm text-gray-600 mb-2">
                            {formatCurrency(item.price)}
                          </div>
                          <div className="w-full bg-gray-200 rounded-full h-2">
                            <div
                              className="bg-blue-600 h-2 rounded-full"
                              style={{ width: `${item.percentage}%` }}
                            ></div>
                          </div>
                          {item.items.length > 0 && (
                            <div className="mt-2 text-xs text-gray-500">
                              {item.items.join(', ')}
                            </div>
                          )}
                        </div>
                      ))}
                    </div>
                  </div>
                )}

                {/* Actions */}
                <div className="space-y-3">
                  <button className="w-full px-4 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                    Get Detailed Quote
                  </button>
                  <button className="w-full px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors">
                    Save Estimate
                  </button>
                </div>
              </>
            ) : (
              <div className="bg-white rounded-lg p-8 shadow-sm text-center">
                <AdjustmentsHorizontalIcon className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">Configure Your Project</h3>
                <p className="text-gray-600">Select features and parameters to see your price estimate</p>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  )
}