'use client'

import { useState, useEffect } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import {
  SparklesIcon,
  ChartBarIcon,
  CurrencyDollarIcon,
  ClockIcon,
  LightBulbIcon,
  BuildingOfficeIcon,
  TrophyIcon,
  CheckCircleIcon,
  ArrowRightIcon,
  InformationCircleIcon,
  AdjustmentsHorizontalIcon,
  DocumentTextIcon
} from '@heroicons/react/24/outline'

interface BusinessProfile {
  businessType: string
  industry: string
  size: string
  budget: number
  goals: string[]
  currentWebsite: boolean
  targetAudience: string
  timeline: string
}

interface CompetitorInsights {
  averageInvestment: number
  keyFeatures: string[]
  competitorAnalysis: string[]
  marketGaps: string[]
  successFactors: string[]
  recommendedDifferentiators: string[]
}

interface ROIProjection {
  investmentAmount: number
  projectedRevenue: number
  roiPercentage: number
  paybackPeriod: number
  conversionImprovement: number
  metrics: {
    trafficIncrease: string
    leadGeneration: string
    brandRecognition: string
    marketShare: string
  }
}

interface BudgetOptimization {
  totalBudget: number
  allocations: Array<{
    goal: string
    allocatedBudget: number
    recommendedServices: string[]
    expectedImpact: string
  }>
  savings: number
  bundleRecommendations: string[]
}

interface ServiceRecommendation {
  serviceId: string
  serviceName: string
  priority: 'critical' | 'high' | 'medium' | 'low'
  confidence: number
  estimatedPrice: { min: number; max: number }
  timeline: string
  benefits: string[]
  reasoning: string
  roi: number
}

const COMPETITOR_ANALYSIS_ENGINE = {
  analyzeMarket: (industry: string, businessType: string): CompetitorInsights => {
    const competitorData: Record<string, any> = {
      'e-commerce': {
        avgWebsiteCost: 150000,
        essentialFeatures: ['Payment Gateway', 'Inventory Management', 'Mobile Optimization', 'SEO'],
        marketLeaders: ['Daraz', 'Amazon', 'Shopify stores'],
        successFactors: ['Fast loading', 'Secure payments', 'Mobile-first design', 'Social proof'],
        gaps: ['Poor mobile experience', 'Limited payment options', 'Weak SEO']
      },
      'service-provider': {
        avgWebsiteCost: 80000,
        essentialFeatures: ['Service Portfolio', 'Contact Forms', 'Testimonials', 'Booking System'],
        marketLeaders: ['Upwork', 'Fiverr', 'Local service providers'],
        successFactors: ['Trust signals', 'Clear pricing', 'Easy contact', 'Professional design'],
        gaps: ['Generic templates', 'Poor mobile UX', 'No online booking']
      },
      'tech-startup': {
        avgWebsiteCost: 200000,
        essentialFeatures: ['Product Demo', 'Mobile App', 'Analytics', 'Scalable Infrastructure'],
        marketLeaders: ['Successful startups', 'Tech companies'],
        successFactors: ['Innovation showcase', 'Scalability', 'User experience', 'Growth metrics'],
        gaps: ['Complex user flows', 'Poor onboarding', 'Limited analytics']
      }
    }

    const data = competitorData[businessType] || competitorData['service-provider']
    
    return {
      averageInvestment: data.avgWebsiteCost,
      keyFeatures: data.essentialFeatures,
      competitorAnalysis: data.marketLeaders,
      marketGaps: data.gaps,
      successFactors: data.successFactors,
      recommendedDifferentiators: [
        'Superior mobile experience',
        'Advanced analytics integration',
        'Personalized user journey',
        'Local market optimization'
      ]
    }
  }
}

const ADVANCED_PRICING_ENGINE = {
  calculateROI: (investment: number, businessType: string, size: string): ROIProjection => {
    const multipliers = {
      'e-commerce': { revenue: 3.5, conversion: 2.8 },
      'service-provider': { revenue: 2.5, conversion: 3.2 },
      'tech-startup': { revenue: 4.0, conversion: 2.5 },
      'education': { revenue: 2.0, conversion: 2.0 },
      'healthcare': { revenue: 2.8, conversion: 2.5 },
      'creative': { revenue: 2.2, conversion: 2.8 }
    }

    const sizeMultipliers = {
      'startup': 1.0,
      'small': 1.2,
      'medium': 1.5,
      'large': 2.0,
      'enterprise': 3.0
    }

    const baseMultiplier = multipliers[businessType as keyof typeof multipliers] || multipliers['service-provider']
    const sizeMultiplier = sizeMultipliers[size as keyof typeof sizeMultipliers] || 1.0

    const projectedRevenue = investment * baseMultiplier.revenue * sizeMultiplier
    const conversionImprovement = baseMultiplier.conversion * 100

    return {
      investmentAmount: investment,
      projectedRevenue,
      roiPercentage: Math.round(((projectedRevenue - investment) / investment) * 100),
      paybackPeriod: Math.ceil(12 / baseMultiplier.revenue),
      conversionImprovement,
      metrics: {
        trafficIncrease: '150-300%',
        leadGeneration: '200-400%',
        brandRecognition: '180-250%',
        marketShare: '20-45%'
      }
    }
  },

  optimizeBudget: (budget: number, goals: string[], businessType: string): BudgetOptimization => {
    const allocations: Record<string, number> = {
      'increase-sales': 0.4,
      'brand-awareness': 0.3,
      'customer-engagement': 0.25,
      'operational-efficiency': 0.35,
      'market-expansion': 0.4,
      'digital-transformation': 0.45
    }

    const goalServices: Record<string, string[]> = {
      'increase-sales': ['E-commerce Development', 'Digital Marketing', 'SEO Optimization'],
      'brand-awareness': ['Branding', 'Digital Marketing', 'Social Media'],
      'customer-engagement': ['Mobile App Development', 'CMS Development', 'Social Media'],
      'operational-efficiency': ['Custom Software', 'Automation', 'CMS Development'],
      'market-expansion': ['Website Development', 'Digital Marketing', 'SEO Optimization'],
      'digital-transformation': ['Website Development', 'Mobile App Development', 'Cloud Migration']
    }

    const priorityServices = goals.map(goal => {
      const allocation = allocations[goal] || 0.3
      const amount = budget * allocation

      return {
        goal,
        allocatedBudget: amount,
        recommendedServices: goalServices[goal] || ['Website Development'],
        expectedImpact: `${Math.round(allocation * 200)}% improvement in ${goal.replace('-', ' ')}`
      }
    })

    return {
      totalBudget: budget,
      allocations: priorityServices,
      savings: budget * 0.15,
      bundleRecommendations: [
        'Website + SEO Package: 20% discount',
        'Mobile App + Marketing: 25% discount',
        'Complete Digital Transformation: 30% discount'
      ]
    }
  }
}

const AI_INTELLIGENCE_ENGINE = {
  generateRecommendations: (profile: BusinessProfile): ServiceRecommendation[] => {
    const serviceDatabase = [
      {
        serviceId: 'website-development',
        serviceName: 'Website Development',
        basePrice: { min: 50000, max: 200000 },
        timeline: '4-8 weeks',
        benefits: ['Professional online presence', 'Lead generation', 'Brand credibility', '24/7 availability']
      },
      {
        serviceId: 'mobile-app-development',
        serviceName: 'Mobile App Development',
        basePrice: { min: 150000, max: 500000 },
        timeline: '8-16 weeks',
        benefits: ['Direct customer access', 'Push notifications', 'Offline functionality', 'App store presence']
      },
      {
        serviceId: 'seo-optimization',
        serviceName: 'SEO Optimization',
        basePrice: { min: 30000, max: 100000 },
        timeline: '3-6 months',
        benefits: ['Increased visibility', 'Organic traffic growth', 'Cost-effective marketing', 'Long-term results']
      },
      {
        serviceId: 'ui-ux-design',
        serviceName: 'UI/UX Design',
        basePrice: { min: 40000, max: 150000 },
        timeline: '3-6 weeks',
        benefits: ['Better user experience', 'Higher conversions', 'Brand consistency', 'Reduced bounce rate']
      },
      {
        serviceId: 'digital-marketing',
        serviceName: 'Digital Marketing',
        basePrice: { min: 25000, max: 100000 },
        timeline: '2-4 weeks setup',
        benefits: ['Targeted reach', 'Measurable results', 'Cost control', 'Brand awareness']
      }
    ]

    const businessPriorities = {
      'e-commerce': ['website-development', 'seo-optimization', 'digital-marketing', 'ui-ux-design'],
      'service-provider': ['website-development', 'seo-optimization', 'ui-ux-design', 'digital-marketing'],
      'tech-startup': ['mobile-app-development', 'ui-ux-design', 'website-development', 'digital-marketing'],
      'education': ['website-development', 'digital-marketing', 'ui-ux-design', 'seo-optimization'],
      'healthcare': ['website-development', 'ui-ux-design', 'seo-optimization', 'digital-marketing'],
      'creative': ['ui-ux-design', 'website-development', 'digital-marketing', 'seo-optimization']
    }

    const priorities = businessPriorities[profile.businessType as keyof typeof businessPriorities] || businessPriorities['service-provider']
    
    return serviceDatabase.map((service, index) => {
      const priorityIndex = priorities.indexOf(service.serviceId)
      const priority = priorityIndex === 0 ? 'critical' : priorityIndex === 1 ? 'high' : priorityIndex === 2 ? 'medium' : 'low'
      const confidence = Math.max(60, 95 - (priorityIndex * 10))
      
      // Budget-based pricing adjustment
      const budgetMultiplier = profile.budget >= 500000 ? 1.5 : profile.budget >= 200000 ? 1.2 : profile.budget >= 100000 ? 1.0 : 0.8
      
      const roi = ADVANCED_PRICING_ENGINE.calculateROI(
        service.basePrice.min * budgetMultiplier,
        profile.businessType,
        profile.size
      ).roiPercentage

      return {
        serviceId: service.serviceId,
        serviceName: service.serviceName,
        priority: priority as any,
        confidence,
        estimatedPrice: {
          min: Math.round(service.basePrice.min * budgetMultiplier),
          max: Math.round(service.basePrice.max * budgetMultiplier)
        },
        timeline: service.timeline,
        benefits: service.benefits,
        reasoning: `Based on your ${profile.businessType} business type and ${profile.size} company size, this service aligns with your goals and budget.`,
        roi
      }
    }).sort((a, b) => {
      const priorityOrder: Record<string, number> = { critical: 4, high: 3, medium: 2, low: 1 }
      return (priorityOrder[b.priority] || 0) - (priorityOrder[a.priority] || 0)
    })
  }
}

export default function ComprehensiveBusinessIntelligence() {
  const [activeTab, setActiveTab] = useState('profile')
  const [businessProfile, setBusinessProfile] = useState<BusinessProfile>({
    businessType: '',
    industry: '',
    size: '',
    budget: 0,
    goals: [],
    currentWebsite: false,
    targetAudience: '',
    timeline: ''
  })
  const [recommendations, setRecommendations] = useState<ServiceRecommendation[]>([])
  const [competitorInsights, setCompetitorInsights] = useState<CompetitorInsights | null>(null)
  const [roiProjection, setRoiProjection] = useState<ROIProjection | null>(null)
  const [budgetOptimization, setBudgetOptimization] = useState<BudgetOptimization | null>(null)

  const handleProfileUpdate = (field: keyof BusinessProfile, value: any) => {
    setBusinessProfile(prev => ({ ...prev, [field]: value }))
  }

  const analyzeProfile = () => {
    if (!businessProfile.businessType || !businessProfile.budget) return

    // Generate recommendations
    const serviceRecommendations = AI_INTELLIGENCE_ENGINE.generateRecommendations(businessProfile)
    setRecommendations(serviceRecommendations)

    // Competitor analysis
    const insights = COMPETITOR_ANALYSIS_ENGINE.analyzeMarket(businessProfile.industry, businessProfile.businessType)
    setCompetitorInsights(insights)

    // ROI projection
    const roi = ADVANCED_PRICING_ENGINE.calculateROI(businessProfile.budget, businessProfile.businessType, businessProfile.size)
    setRoiProjection(roi)

    // Budget optimization
    const budgetOpt = ADVANCED_PRICING_ENGINE.optimizeBudget(businessProfile.budget, businessProfile.goals, businessProfile.businessType)
    setBudgetOptimization(budgetOpt)

    setActiveTab('analysis')
  }

  const formatCurrency = (amount: number) => `Rs. ${amount.toLocaleString('en-PK')}`

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center mb-8">
          <SparklesIcon className="w-16 h-16 text-blue-600 mx-auto mb-4" />
          <h1 className="text-3xl font-bold text-gray-900 mb-2">AI Business Intelligence</h1>
          <p className="text-gray-600">Advanced analytics and recommendations for your business growth</p>
        </div>

        {/* Navigation Tabs */}
        <div className="border-b border-gray-200 mb-8">
          <nav className="-mb-px flex space-x-8">
            {[
              { id: 'profile', name: 'Business Profile', icon: BuildingOfficeIcon },
              { id: 'analysis', name: 'AI Analysis', icon: ChartBarIcon },
              { id: 'recommendations', name: 'Recommendations', icon: LightBulbIcon },
              { id: 'competitor', name: 'Market Insights', icon: TrophyIcon }
            ].map((tab) => (
              <button
                key={tab.id}
                onClick={() => setActiveTab(tab.id)}
                className={`flex items-center space-x-2 py-4 px-1 border-b-2 font-medium text-sm ${
                  activeTab === tab.id
                    ? 'border-blue-500 text-blue-600'
                    : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
              >
                <tab.icon className="w-5 h-5" />
                <span>{tab.name}</span>
              </button>
            ))}
          </nav>
        </div>

        {/* Business Profile Tab */}
        {activeTab === 'profile' && (
          <div className="bg-white rounded-lg p-8 shadow-sm">
            <h2 className="text-2xl font-bold text-gray-900 mb-6">Tell us about your business</h2>
            
            <div className="space-y-8">
              {/* Business Type */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-4">Business Type</label>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  {[
                    { id: 'e-commerce', name: 'E-commerce', desc: 'Online retail and sales' },
                    { id: 'service-provider', name: 'Service Provider', desc: 'Professional services' },
                    { id: 'tech-startup', name: 'Tech Startup', desc: 'Technology innovation' },
                    { id: 'education', name: 'Education', desc: 'Learning institutions' },
                    { id: 'healthcare', name: 'Healthcare', desc: 'Medical services' },
                    { id: 'creative', name: 'Creative', desc: 'Design and media' }
                  ].map((type) => (
                    <button
                      key={type.id}
                      onClick={() => handleProfileUpdate('businessType', type.id)}
                      className={`p-4 border-2 rounded-lg text-left ${
                        businessProfile.businessType === type.id
                          ? 'border-blue-500 bg-blue-50'
                          : 'border-gray-200 hover:border-gray-300'
                      }`}
                    >
                      <h3 className="font-medium text-gray-900">{type.name}</h3>
                      <p className="text-sm text-gray-600">{type.desc}</p>
                    </button>
                  ))}
                </div>
              </div>

              {/* Budget */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-4">Investment Budget</label>
                <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                  {[
                    { amount: 100000, label: 'Starter', range: 'Rs. 50K - 150K' },
                    { amount: 300000, label: 'Growth', range: 'Rs. 150K - 500K' },
                    { amount: 700000, label: 'Scale', range: 'Rs. 500K - 1M' },
                    { amount: 1500000, label: 'Enterprise', range: 'Rs. 1M+' }
                  ].map((budget) => (
                    <button
                      key={budget.amount}
                      onClick={() => handleProfileUpdate('budget', budget.amount)}
                      className={`p-4 border-2 rounded-lg text-center ${
                        businessProfile.budget === budget.amount
                          ? 'border-blue-500 bg-blue-50'
                          : 'border-gray-200 hover:border-gray-300'
                      }`}
                    >
                      <h3 className="font-medium text-gray-900">{budget.label}</h3>
                      <p className="text-sm text-gray-600">{budget.range}</p>
                    </button>
                  ))}
                </div>
              </div>

              {/* Company Size */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-4">Company Size</label>
                <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                  {[
                    { id: 'startup', label: 'Startup', desc: '1-10 employees' },
                    { id: 'small', label: 'Small', desc: '11-50 employees' },
                    { id: 'medium', label: 'Medium', desc: '51-200 employees' },
                    { id: 'large', label: 'Large', desc: '200+ employees' }
                  ].map((size) => (
                    <button
                      key={size.id}
                      onClick={() => handleProfileUpdate('size', size.id)}
                      className={`p-4 border-2 rounded-lg text-center ${
                        businessProfile.size === size.id
                          ? 'border-blue-500 bg-blue-50'
                          : 'border-gray-200 hover:border-gray-300'
                      }`}
                    >
                      <h3 className="font-medium text-gray-900">{size.label}</h3>
                      <p className="text-sm text-gray-600">{size.desc}</p>
                    </button>
                  ))}
                </div>
              </div>

              {/* Business Goals */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-4">Primary Goals (Select up to 3)</label>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
                  {[
                    'increase-sales',
                    'brand-awareness',
                    'customer-engagement',
                    'operational-efficiency',
                    'market-expansion',
                    'digital-transformation'
                  ].map((goal) => (
                    <label key={goal} className="flex items-center space-x-3">
                      <input
                        type="checkbox"
                        checked={businessProfile.goals.includes(goal)}
                        onChange={(e) => {
                          if (e.target.checked && businessProfile.goals.length < 3) {
                            handleProfileUpdate('goals', [...businessProfile.goals, goal])
                          } else if (!e.target.checked) {
                            handleProfileUpdate('goals', businessProfile.goals.filter(g => g !== goal))
                          }
                        }}
                        className="h-4 w-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                      />
                      <span className="text-sm text-gray-700 capitalize">{goal.replace('-', ' ')}</span>
                    </label>
                  ))}
                </div>
              </div>

              <button
                onClick={analyzeProfile}
                disabled={!businessProfile.businessType || !businessProfile.budget}
                className="w-full px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:bg-gray-300 disabled:cursor-not-allowed transition-colors"
              >
                Generate AI Analysis
              </button>
            </div>
          </div>
        )}

        {/* AI Analysis Tab */}
        {activeTab === 'analysis' && roiProjection && (
          <div className="space-y-8">
            {/* ROI Projection */}
            <div className="bg-white rounded-lg p-6 shadow-sm">
              <h3 className="text-xl font-bold text-gray-900 mb-6">ROI Projection</h3>
              
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                <div className="text-center p-4 bg-green-50 rounded-lg">
                  <div className="text-3xl font-bold text-green-600">{roiProjection.roiPercentage}%</div>
                  <div className="text-sm text-green-700">Expected ROI</div>
                </div>
                <div className="text-center p-4 bg-blue-50 rounded-lg">
                  <div className="text-3xl font-bold text-blue-600">{formatCurrency(roiProjection.projectedRevenue)}</div>
                  <div className="text-sm text-blue-700">Projected Revenue</div>
                </div>
                <div className="text-center p-4 bg-purple-50 rounded-lg">
                  <div className="text-3xl font-bold text-purple-600">{roiProjection.paybackPeriod}</div>
                  <div className="text-sm text-purple-700">Months to Payback</div>
                </div>
              </div>

              <div className="mt-6 grid grid-cols-2 md:grid-cols-4 gap-4">
                {Object.entries(roiProjection.metrics).map(([key, value]) => (
                  <div key={key} className="text-center">
                    <div className="font-semibold text-gray-900">{value}</div>
                    <div className="text-xs text-gray-600 capitalize">{key.replace(/([A-Z])/g, ' $1')}</div>
                  </div>
                ))}
              </div>
            </div>

            {/* Budget Optimization */}
            {budgetOptimization && (
              <div className="bg-white rounded-lg p-6 shadow-sm">
                <h3 className="text-xl font-bold text-gray-900 mb-6">Budget Optimization</h3>
                
                <div className="space-y-4">
                  {budgetOptimization.allocations.map((allocation, index) => (
                    <div key={index} className="p-4 border border-gray-200 rounded-lg">
                      <div className="flex justify-between items-start mb-2">
                        <h4 className="font-medium text-gray-900 capitalize">{allocation.goal.replace('-', ' ')}</h4>
                        <span className="text-blue-600 font-semibold">{formatCurrency(allocation.allocatedBudget)}</span>
                      </div>
                      <p className="text-sm text-gray-600 mb-2">{allocation.expectedImpact}</p>
                      <div className="flex flex-wrap gap-2">
                        {allocation.recommendedServices.map((service, idx) => (
                          <span key={idx} className="px-2 py-1 bg-blue-100 text-blue-800 text-xs rounded-full">
                            {service}
                          </span>
                        ))}
                      </div>
                    </div>
                  ))}
                </div>

                <div className="mt-6 p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                  <h4 className="font-medium text-yellow-800 mb-2">Bundle Savings: {formatCurrency(budgetOptimization.savings)}</h4>
                  <ul className="text-sm text-yellow-700 space-y-1">
                    {budgetOptimization.bundleRecommendations.map((bundle, index) => (
                      <li key={index}>• {bundle}</li>
                    ))}
                  </ul>
                </div>
              </div>
            )}
          </div>
        )}

        {/* Recommendations Tab */}
        {activeTab === 'recommendations' && recommendations.length > 0 && (
          <div className="space-y-6">
            {recommendations.map((rec, index) => (
              <motion.div
                key={rec.serviceId}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
                className={`bg-white rounded-lg p-6 shadow-sm border-l-4 ${
                  rec.priority === 'critical' ? 'border-red-500' :
                  rec.priority === 'high' ? 'border-orange-500' :
                  rec.priority === 'medium' ? 'border-yellow-500' : 'border-gray-300'
                }`}
              >
                <div className="flex justify-between items-start mb-4">
                  <div>
                    <h3 className="text-xl font-bold text-gray-900">{rec.serviceName}</h3>
                    <div className="flex items-center space-x-4 mt-2">
                      <span className={`px-3 py-1 text-xs font-medium rounded-full ${
                        rec.priority === 'critical' ? 'bg-red-100 text-red-800' :
                        rec.priority === 'high' ? 'bg-orange-100 text-orange-800' :
                        rec.priority === 'medium' ? 'bg-yellow-100 text-yellow-800' : 'bg-gray-100 text-gray-800'
                      }`}>
                        {rec.priority.toUpperCase()} PRIORITY
                      </span>
                      <span className="text-sm text-gray-600">Confidence: {rec.confidence}%</span>
                    </div>
                  </div>
                  <div className="text-right">
                    <div className="text-2xl font-bold text-gray-900">
                      {formatCurrency(rec.estimatedPrice.min)} - {formatCurrency(rec.estimatedPrice.max)}
                    </div>
                    <div className="text-sm text-gray-600">{rec.timeline}</div>
                  </div>
                </div>

                <p className="text-gray-700 mb-4">{rec.reasoning}</p>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div>
                    <h4 className="font-medium text-gray-900 mb-2">Key Benefits</h4>
                    <ul className="space-y-1">
                      {rec.benefits.map((benefit, idx) => (
                        <li key={idx} className="flex items-center space-x-2">
                          <CheckCircleIcon className="w-4 h-4 text-green-500" />
                          <span className="text-sm text-gray-700">{benefit}</span>
                        </li>
                      ))}
                    </ul>
                  </div>
                  <div>
                    <div className="p-4 bg-green-50 rounded-lg">
                      <div className="text-center">
                        <div className="text-2xl font-bold text-green-600">{rec.roi}%</div>
                        <div className="text-sm text-green-700">Expected ROI</div>
                      </div>
                    </div>
                  </div>
                </div>

                <div className="mt-6 flex space-x-4">
                  <button className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                    Get Detailed Quote
                  </button>
                  <button className="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50">
                    Learn More
                  </button>
                </div>
              </motion.div>
            ))}
          </div>
        )}

        {/* Market Insights Tab */}
        {activeTab === 'competitor' && competitorInsights && (
          <div className="space-y-8">
            <div className="bg-white rounded-lg p-6 shadow-sm">
              <h3 className="text-xl font-bold text-gray-900 mb-6">Market Analysis</h3>
              
              <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                <div>
                  <h4 className="font-medium text-gray-900 mb-3">Market Leaders</h4>
                  <ul className="space-y-2">
                    {competitorInsights.competitorAnalysis.map((competitor, index) => (
                      <li key={index} className="flex items-center space-x-2">
                        <TrophyIcon className="w-4 h-4 text-yellow-500" />
                        <span className="text-gray-700">{competitor}</span>
                      </li>
                    ))}
                  </ul>
                </div>

                <div>
                  <h4 className="font-medium text-gray-900 mb-3">Success Factors</h4>
                  <ul className="space-y-2">
                    {competitorInsights.successFactors.map((factor, index) => (
                      <li key={index} className="flex items-center space-x-2">
                        <CheckCircleIcon className="w-4 h-4 text-green-500" />
                        <span className="text-gray-700">{factor}</span>
                      </li>
                    ))}
                  </ul>
                </div>

                <div>
                  <h4 className="font-medium text-gray-900 mb-3">Market Gaps</h4>
                  <ul className="space-y-2">
                    {competitorInsights.marketGaps.map((gap, index) => (
                      <li key={index} className="flex items-center space-x-2">
                        <InformationCircleIcon className="w-4 h-4 text-red-500" />
                        <span className="text-gray-700">{gap}</span>
                      </li>
                    ))}
                  </ul>
                </div>

                <div>
                  <h4 className="font-medium text-gray-900 mb-3">Recommended Differentiators</h4>
                  <ul className="space-y-2">
                    {competitorInsights.recommendedDifferentiators.map((diff, index) => (
                      <li key={index} className="flex items-center space-x-2">
                        <LightBulbIcon className="w-4 h-4 text-blue-500" />
                        <span className="text-gray-700">{diff}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              </div>

              <div className="mt-6 p-4 bg-blue-50 border border-blue-200 rounded-lg">
                <h4 className="font-medium text-blue-900 mb-2">Average Market Investment</h4>
                <div className="text-2xl font-bold text-blue-600">
                  {formatCurrency(competitorInsights.averageInvestment)}
                </div>
                <p className="text-sm text-blue-700 mt-1">Based on industry analysis</p>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}