'use client'

import { useState } from 'react'
import { MagnifyingGlassIcon, CheckCircleIcon, XCircleIcon } from '@heroicons/react/24/outline'
import { motion } from 'framer-motion'
import Image from 'next/image'

const popularExtensions = ['.com', '.net', '.org', '.info', '.biz', '.co']

const tldImages = [
  { ext: '.com', image: '/images/domains/com-clean.svg', color: 'from-blue-500 to-blue-600' },
  { ext: '.net', image: '/images/domains/net-clean.svg', color: 'from-green-500 to-green-600' },
  { ext: '.org', image: '/images/domains/org-clean.svg', color: 'from-orange-500 to-orange-600' },
  { ext: '.pk', image: '/images/domains/pk-clean.svg', color: 'from-emerald-600 to-emerald-700' },
  { ext: '.co', image: '/images/domains/co-clean.svg', color: 'from-purple-500 to-purple-600' },
  { ext: '.io', image: '/images/domains/io-clean.svg', color: 'from-red-500 to-red-600' },
]

interface DomainResult {
  domain: string
  extension: string
  available: boolean
  price: number | null
}

export default function DomainChecker() {
  const [domain, setDomain] = useState('')
  const [isChecking, setIsChecking] = useState(false)
  const [results, setResults] = useState<DomainResult[]>([])
  const [hasSearched, setHasSearched] = useState(false)
  const [error, setError] = useState('')

  const checkDomain = async () => {
    if (!domain.trim()) return

    setIsChecking(true)
    setHasSearched(true)
    setError('')
    
    try {
      const response = await fetch('/api/domains/check', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ domain: domain.trim() }),
      })
      
      if (!response.ok) {
        throw new Error('Failed to check domain availability')
      }
      
      const data = await response.json()
      setResults(data.results || [])
    } catch (err) {
      setError('Failed to check domain availability. Please try again.')
      console.error('Domain check error:', err)
    } finally {
      setIsChecking(false)
    }
  }

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    checkDomain()
  }

  return (
    <motion.div
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ delay: 0.5 }}
      className="w-full mx-auto"
    >
      {/* Single Line Domain Search */}
      <motion.div
        initial={{ opacity: 0, y: 30 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 0.7, duration: 0.6 }}
        className="mb-4"
      >
        <div className="bg-white rounded-xl shadow-md border border-gray-100 p-3">
          <form onSubmit={handleSubmit}>
            <div className="flex items-center gap-3">
              {/* Search Input */}
              <div className="relative flex-1">
                <input
                  type="text"
                  value={domain}
                  onChange={(e) => setDomain(e.target.value)}
                  placeholder="Enter domain name (e.g., mycompany)"
                  className="w-full px-4 py-2.5 pl-10 pr-4 border-2 border-gray-200 rounded-lg focus:border-blue-500 focus:outline-none transition-colors text-sm"
                  disabled={isChecking}
                />
                <MagnifyingGlassIcon className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-gray-400" />
              </div>
              
              {/* TLD Quick Select Badges */}
              <div className="flex items-center gap-1.5">
                {tldImages.map((tld, index) => (
                  <motion.button
                    key={tld.ext}
                    initial={{ opacity: 0, scale: 0.8 }}
                    animate={{ opacity: 1, scale: 1 }}
                    transition={{ delay: 0.8 + index * 0.05, duration: 0.3 }}
                    type="button"
                    onClick={() => {
                      const currentDomain = domain || 'yoursite'
                      setDomain(currentDomain)
                      setTimeout(() => {
                        const input = document.querySelector('input[type="text"]') as HTMLInputElement
                        input?.focus()
                      }, 100)
                    }}
                    className={`group relative px-2.5 py-2 bg-gradient-to-r ${tld.color} text-white rounded-lg text-xs font-bold hover:shadow-lg hover:scale-105 transition-all duration-200 whitespace-nowrap`}
                  >
                    {tld.ext}
                    <span className="absolute -bottom-8 left-1/2 transform -translate-x-1/2 bg-gray-900 text-white text-[10px] px-2 py-0.5 rounded opacity-0 group-hover:opacity-100 transition-opacity duration-200 whitespace-nowrap pointer-events-none z-10">
                      Search {tld.ext}
                    </span>
                  </motion.button>
                ))}
              </div>
              
              {/* Search Button */}
              <button
                type="submit"
                disabled={isChecking || !domain.trim()}
                className="px-6 py-2.5 bg-blue-600 hover:bg-blue-700 disabled:bg-gray-300 text-white rounded-lg font-semibold text-sm transition-colors whitespace-nowrap"
              >
                {isChecking ? 'Checking...' : 'Search'}
              </button>
            </div>
          </form>
        </div>
      </motion.div>

      {/* Results */}
      {hasSearched && (
        <motion.div
          initial={{ opacity: 0, height: 0 }}
          animate={{ opacity: 1, height: 'auto' }}
          transition={{ duration: 0.3 }}
          className="mt-4"
        >
          {error && (
            <div className="text-center py-3 text-sm text-red-600 bg-red-50 rounded-lg border border-red-200">
              {error}
            </div>
          )}
          
          {isChecking ? (
            <div className="text-center py-6">
              <div className="inline-block animate-spin rounded-full h-7 w-7 border-b-2 border-blue-600"></div>
              <p className="mt-2 text-sm text-gray-600">Checking availability...</p>
            </div>
          ) : results.length > 0 ? (
            <div className="space-y-2">
              <h4 className="font-semibold text-gray-900 mb-2 text-sm">
                Results for "{domain}":
              </h4>
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
                {results.map((result) => (
                  <motion.div
                    key={result.extension}
                    initial={{ opacity: 0, x: -20 }}
                    animate={{ opacity: 1, x: 0 }}
                    className={`flex items-center justify-between p-3 rounded-lg border-2 ${
                      result.available
                        ? 'border-green-200 bg-green-50'
                        : 'border-red-200 bg-red-50'
                    }`}
                  >
                    <div className="flex-1">
                      <div className="font-medium text-gray-900 text-sm">
                        {result.domain}
                      </div>
                      {result.available && result.price && (
                        <div className="text-xs text-green-600 font-semibold">
                          ${result.price}/year
                        </div>
                      )}
                      {!result.available && (
                        <div className="text-xs text-red-600">
                          Not available
                        </div>
                      )}
                    </div>
                    {result.available ? (
                      <CheckCircleIcon className="w-5 h-5 text-green-600 ml-2" />
                    ) : (
                      <XCircleIcon className="w-5 h-5 text-red-600 ml-2" />
                    )}
                  </motion.div>
                ))}
              </div>
              
              {/* Available domains action */}
              {results.some(r => r.available) && (
                <motion.div
                  initial={{ opacity: 0, y: 10 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: 0.3 }}
                  className="mt-4 p-4 bg-blue-50 rounded-lg border border-blue-200"
                >
                  <p className="text-blue-800 font-medium mb-2">
                    🎉 Great! {results.filter(r => r.available).length} domain(s) available.
                  </p>
                  <div className="flex flex-col sm:flex-row gap-2">
                    <button className="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg font-semibold transition-colors">
                      Register Domains
                    </button>
                    <button className="border-2 border-blue-600 text-blue-600 hover:bg-blue-600 hover:text-white px-6 py-2 rounded-lg font-semibold transition-colors">
                      Get Quote
                    </button>
                  </div>
                </motion.div>
              )}
            </div>
          ) : null}
        </motion.div>
      )}
    </motion.div>
  )
}