'use client'

import { Fragment, useEffect, useState } from 'react'
import { Dialog, Transition } from '@headlessui/react'
import { 
  XMarkIcon, 
  MinusIcon, 
  PlusIcon, 
  TrashIcon,
  GiftIcon,
  TagIcon,
  SparklesIcon,
  CheckIcon,
  ExclamationTriangleIcon
} from '@heroicons/react/24/outline'
import { useHydrationSafeCart } from '@/hooks/useHydrationSafeCart'
import { formatPrice } from '@/lib/utils'
import Link from 'next/link'

// Package options for different services
const SERVICE_PACKAGES = {
  '1': [ // Website Development
    { id: 'basic', name: 'Basic', price: 2500, features: ['Responsive Design', 'SEO Optimized', '5 Pages'] },
    { id: 'premium', name: 'Premium', price: 4000, features: ['Everything in Basic', 'E-commerce Integration', 'Advanced SEO'] },
    { id: 'enterprise', name: 'Enterprise', price: 6500, features: ['Everything in Premium', 'Custom Features', 'Priority Support'] }
  ],
  '4': [ // Social Media Marketing
    { id: 'starter', name: 'Starter', price: 500, features: ['2 Platforms', 'Basic Content', '4 Posts/Week'] },
    { id: 'growth', name: 'Growth', price: 800, features: ['4 Platforms', 'Custom Graphics', '6 Posts/Week'] },
    { id: 'pro', name: 'Professional', price: 1200, features: ['All Platforms', 'Video Content', 'Daily Posts'] }
  ]
}

// Add-ons available for services
const SERVICE_ADDONS = {
  '1': [ // Website Development
    { id: 'ssl', name: 'SSL Certificate', price: 100, description: 'Secure your website' },
    { id: 'hosting', name: '1 Year Hosting', price: 150, description: 'Professional hosting included' },
    { id: 'maintenance', name: 'Monthly Maintenance', price: 200, description: 'Keep your site updated' }
  ],
  '4': [ // Social Media Marketing
    { id: 'analytics', name: 'Advanced Analytics', price: 100, description: 'Detailed performance reports' },
    { id: 'influencer', name: 'Influencer Outreach', price: 300, description: 'Connect with influencers' },
    { id: 'ads', name: 'Paid Advertising Setup', price: 250, description: 'Professional ad campaigns' }
  ]
}

export default function EnhancedShoppingCart() {
  const [mounted, setMounted] = useState(false)
  const [promoCode, setPromoCode] = useState('')
  const [promoError, setPromoError] = useState('')
  const [selectedItemForCustomization, setSelectedItemForCustomization] = useState<string | null>(null)
  
  const {
    items,
    bundles,
    isOpen,
    closeCart,
    removeItem,
    updateQuantity,
    updatePackage,
    addAddon,
    removeAddon,
    addBundle,
    removeBundle,
    suggestBundles,
    applyPromoCode,
    removePromoCode,
    appliedPromoCode,
    getTotalPrice,
    getTotalItems,
    getBulkDiscount,
    getTotalWithDiscounts,
    getSavings,
  } = useHydrationSafeCart()

  const totalPrice = getTotalPrice()
  const totalItems = getTotalItems()
  const bulkDiscount = getBulkDiscount()
  const finalPrice = getTotalWithDiscounts()
  const totalSavings = getSavings()
  const suggestedBundles = suggestBundles()

  useEffect(() => {
    setMounted(true)
  }, [])

  const handleApplyPromoCode = () => {
    if (applyPromoCode(promoCode)) {
      setPromoCode('')
      setPromoError('')
    } else {
      setPromoError('Invalid promo code')
    }
  }

  const handlePackageChange = (itemId: string, selectedPackage: any) => {
    updatePackage(itemId, selectedPackage)
  }

  const handleAddAddon = (itemId: string, addon: any) => {
    addAddon(itemId, addon)
  }

  if (!mounted) {
    return null
  }

  return (
    <Transition.Root show={isOpen} as={Fragment}>
      <Dialog as="div" className="relative z-50" onClose={closeCart}>
        <Transition.Child
          as={Fragment}
          enter="ease-in-out duration-500"
          enterFrom="opacity-0"
          enterTo="opacity-100"
          leave="ease-in-out duration-500"
          leaveFrom="opacity-100"
          leaveTo="opacity-0"
        >
          <div className="fixed inset-0 bg-gray-500 bg-opacity-75 transition-opacity" />
        </Transition.Child>

        <div className="fixed inset-0 overflow-hidden">
          <div className="absolute inset-0 overflow-hidden">
            <div className="pointer-events-none fixed inset-y-0 right-0 flex max-w-full pl-10">
              <Transition.Child
                as={Fragment}
                enter="transform transition ease-in-out duration-500 sm:duration-700"
                enterFrom="translate-x-full"
                enterTo="translate-x-0"
                leave="transform transition ease-in-out duration-500 sm:duration-700"
                leaveFrom="translate-x-0"
                leaveTo="translate-x-full"
              >
                <Dialog.Panel className="pointer-events-auto w-screen max-w-2xl">
                  <div className="flex h-full flex-col overflow-y-scroll bg-white shadow-xl">
                    <div className="flex-1 overflow-y-auto px-4 py-6 sm:px-6">
                      <div className="flex items-start justify-between">
                        <Dialog.Title className="text-lg font-medium text-gray-900">
                          Shopping Cart ({totalItems} items)
                        </Dialog.Title>
                        <div className="ml-3 flex h-7 items-center">
                          <button
                            type="button"
                            className="relative -m-2 p-2 text-gray-400 hover:text-gray-500"
                            onClick={closeCart}
                          >
                            <span className="absolute -inset-0.5" />
                            <span className="sr-only">Close panel</span>
                            <XMarkIcon className="h-6 w-6" aria-hidden="true" />
                          </button>
                        </div>
                      </div>

                      {/* Savings Banner */}
                      {totalSavings > 0 && (
                        <div className="mt-4 bg-green-50 border border-green-200 rounded-lg p-4">
                          <div className="flex items-center">
                            <SparklesIcon className="h-5 w-5 text-green-600 mr-2" />
                            <span className="text-green-800 font-medium">
                              You're saving {formatPrice(totalSavings)}!
                            </span>
                          </div>
                        </div>
                      )}

                      {/* Bulk Discount Info */}
                      {bulkDiscount > 0 && (
                        <div className="mt-4 bg-blue-50 border border-blue-200 rounded-lg p-4">
                          <div className="flex items-center">
                            <GiftIcon className="h-5 w-5 text-blue-600 mr-2" />
                            <span className="text-blue-800 font-medium">
                              Bulk discount applied: {formatPrice(bulkDiscount)} off
                            </span>
                          </div>
                        </div>
                      )}

                      {/* Cart Items */}
                      <div className="mt-8">
                        <div className="flow-root">
                          {items.length === 0 && bundles.length === 0 ? (
                            <div className="text-center py-8">
                              <p className="text-gray-500">Your cart is empty</p>
                              <Link
                                href="/services"
                                className="mt-4 inline-block bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 transition-colors"
                                onClick={closeCart}
                              >
                                Browse Services
                              </Link>
                            </div>
                          ) : (
                            <>
                              {/* Regular Items */}
                              {items.length > 0 && (
                                <ul className="-my-6 divide-y divide-gray-200">
                                  {items.map((item) => (
                                    <li key={item.id} className="py-6">
                                      <div className="flex">
                                        <div className="h-20 w-20 flex-shrink-0 overflow-hidden rounded-md border border-gray-200">
                                          <div className="h-full w-full bg-gradient-to-br from-blue-100 to-purple-100 flex items-center justify-center">
                                            <span className="text-blue-600 font-semibold text-lg">
                                              {item.name.charAt(0).toUpperCase()}
                                            </span>
                                          </div>
                                        </div>

                                        <div className="ml-4 flex flex-1 flex-col">
                                          <div>
                                            <div className="flex justify-between text-base font-medium text-gray-900">
                                              <h3>{item.name}</h3>
                                              <div className="text-right">
                                                <p className="ml-4">{formatPrice((item.selectedPackage?.price || item.price) * item.quantity)}</p>
                                                {item.selectedPackage && (
                                                  <p className="text-sm text-blue-600">{item.selectedPackage.name} Package</p>
                                                )}
                                              </div>
                                            </div>

                                            {/* Package Selection */}
                                            {SERVICE_PACKAGES[item.serviceId as keyof typeof SERVICE_PACKAGES] && (
                                              <div className="mt-2">
                                                <label className="text-sm font-medium text-gray-700">Package:</label>
                                                <select
                                                  value={item.selectedPackage?.id || 'basic'}
                                                  onChange={(e) => {
                                                    const packages = SERVICE_PACKAGES[item.serviceId as keyof typeof SERVICE_PACKAGES]
                                                    const selectedPackage = packages.find(p => p.id === e.target.value)
                                                    if (selectedPackage) {
                                                      handlePackageChange(item.id, selectedPackage)
                                                    }
                                                  }}
                                                  className="mt-1 block w-full text-sm border-gray-300 rounded-md"
                                                >
                                                  {SERVICE_PACKAGES[item.serviceId as keyof typeof SERVICE_PACKAGES].map((pkg) => (
                                                    <option key={pkg.id} value={pkg.id}>
                                                      {pkg.name} - {formatPrice(pkg.price)}
                                                    </option>
                                                  ))}
                                                </select>
                                              </div>
                                            )}

                                            {/* Add-ons */}
                                            {item.addons && item.addons.length > 0 && (
                                              <div className="mt-2">
                                                <div className="text-sm text-gray-600">
                                                  Add-ons: 
                                                  {item.addons.map((addon, index) => (
                                                    <span key={addon.id} className="inline-flex items-center ml-2">
                                                      {addon.name} (+{formatPrice(addon.price)})
                                                      <button
                                                        onClick={() => removeAddon(item.id, addon.id)}
                                                        className="ml-1 text-red-600 hover:text-red-800"
                                                      >
                                                        <XMarkIcon className="h-3 w-3" />
                                                      </button>
                                                    </span>
                                                  ))}
                                                </div>
                                              </div>
                                            )}

                                            {/* Available Add-ons */}
                                            {SERVICE_ADDONS[item.serviceId as keyof typeof SERVICE_ADDONS] && (
                                              <div className="mt-2">
                                                <details className="text-sm">
                                                  <summary className="cursor-pointer text-blue-600 hover:text-blue-800">
                                                    Add optional services
                                                  </summary>
                                                  <div className="mt-2 space-y-1">
                                                    {SERVICE_ADDONS[item.serviceId as keyof typeof SERVICE_ADDONS]
                                                      .filter(addon => !item.addons?.find(a => a.id === addon.id))
                                                      .map((addon) => (
                                                        <div key={addon.id} className="flex items-center justify-between">
                                                          <div>
                                                            <span className="font-medium">{addon.name}</span>
                                                            <p className="text-xs text-gray-500">{addon.description}</p>
                                                          </div>
                                                          <button
                                                            onClick={() => handleAddAddon(item.id, addon)}
                                                            className="text-blue-600 hover:text-blue-800 text-xs px-2 py-1 border border-blue-300 rounded"
                                                          >
                                                            Add +{formatPrice(addon.price)}
                                                          </button>
                                                        </div>
                                                      ))}
                                                  </div>
                                                </details>
                                              </div>
                                            )}
                                          </div>
                                          
                                          <div className="flex flex-1 items-end justify-between text-sm">
                                            <div className="flex items-center space-x-2">
                                              <button
                                                onClick={() => updateQuantity(item.id, item.quantity - 1)}
                                                className="p-1 text-gray-500 hover:text-gray-700"
                                              >
                                                <MinusIcon className="h-4 w-4" />
                                              </button>
                                              <span className="text-gray-500 min-w-[2rem] text-center">
                                                Qty {item.quantity}
                                              </span>
                                              <button
                                                onClick={() => updateQuantity(item.id, item.quantity + 1)}
                                                className="p-1 text-gray-500 hover:text-gray-700"
                                              >
                                                <PlusIcon className="h-4 w-4" />
                                              </button>
                                            </div>

                                            <div className="flex">
                                              <button
                                                type="button"
                                                className="font-medium text-red-600 hover:text-red-500"
                                                onClick={() => removeItem(item.id)}
                                              >
                                                <TrashIcon className="h-4 w-4" />
                                              </button>
                                            </div>
                                          </div>
                                        </div>
                                      </div>
                                    </li>
                                  ))}
                                </ul>
                              )}

                              {/* Bundles */}
                              {bundles.length > 0 && (
                                <div className="mt-8">
                                  <h3 className="text-lg font-medium text-gray-900 mb-4">Service Bundles</h3>
                                  <div className="space-y-4">
                                    {bundles.map((bundle) => (
                                      <div key={bundle.id} className="bg-green-50 border border-green-200 rounded-lg p-4">
                                        <div className="flex justify-between items-start">
                                          <div>
                                            <h4 className="font-medium text-green-800">{bundle.name}</h4>
                                            <p className="text-sm text-green-600">{bundle.description}</p>
                                            <p className="text-lg font-semibold text-green-800 mt-2">
                                              {formatPrice(bundle.price)}
                                              <span className="text-sm font-normal text-green-600 ml-2">
                                                ({bundle.discount}% off)
                                              </span>
                                            </p>
                                          </div>
                                          <button
                                            onClick={() => removeBundle(bundle.id)}
                                            className="text-red-600 hover:text-red-800"
                                          >
                                            <TrashIcon className="h-4 w-4" />
                                          </button>
                                        </div>
                                      </div>
                                    ))}
                                  </div>
                                </div>
                              )}

                              {/* Bundle Suggestions */}
                              {suggestedBundles.length > 0 && (
                                <div className="mt-8">
                                  <h3 className="text-lg font-medium text-gray-900 mb-4">Recommended Bundles</h3>
                                  <div className="space-y-3">
                                    {suggestedBundles.map((bundle) => (
                                      <div key={bundle.id} className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                                        <div className="flex justify-between items-start">
                                          <div>
                                            <h4 className="font-medium text-yellow-800">{bundle.name}</h4>
                                            <p className="text-sm text-yellow-600">{bundle.description}</p>
                                            <p className="text-lg font-semibold text-yellow-800 mt-1">
                                              {formatPrice(bundle.price)} 
                                              <span className="text-sm font-normal text-yellow-600 ml-2">
                                                (Save {bundle.discount}%)
                                              </span>
                                            </p>
                                          </div>
                                          <button
                                            onClick={() => addBundle(bundle)}
                                            className="bg-yellow-600 text-white px-3 py-1 rounded text-sm hover:bg-yellow-700"
                                          >
                                            Add Bundle
                                          </button>
                                        </div>
                                      </div>
                                    ))}
                                  </div>
                                </div>
                              )}
                            </>
                          )}
                        </div>
                      </div>
                    </div>

                    {(items.length > 0 || bundles.length > 0) && (
                      <div className="border-t border-gray-200 px-4 py-6 sm:px-6">
                        {/* Promo Code */}
                        <div className="mb-6">
                          <div className="flex space-x-2">
                            <input
                              type="text"
                              placeholder="Enter promo code"
                              value={promoCode}
                              onChange={(e) => setPromoCode(e.target.value)}
                              className="flex-1 border border-gray-300 rounded-md px-3 py-2 text-sm"
                            />
                            <button
                              onClick={handleApplyPromoCode}
                              className="bg-blue-600 text-white px-4 py-2 rounded-md text-sm hover:bg-blue-700"
                            >
                              Apply
                            </button>
                          </div>
                          {promoError && (
                            <p className="text-red-600 text-xs mt-1 flex items-center">
                              <ExclamationTriangleIcon className="h-3 w-3 mr-1" />
                              {promoError}
                            </p>
                          )}
                          {appliedPromoCode && (
                            <p className="text-green-600 text-xs mt-1 flex items-center">
                              <CheckIcon className="h-3 w-3 mr-1" />
                              Promo code "{appliedPromoCode}" applied
                              <button
                                onClick={removePromoCode}
                                className="ml-2 text-red-600 hover:text-red-800"
                              >
                                <XMarkIcon className="h-3 w-3" />
                              </button>
                            </p>
                          )}
                        </div>

                        {/* Price Breakdown */}
                        <div className="space-y-2 text-sm">
                          <div className="flex justify-between">
                            <p>Subtotal ({totalItems} items)</p>
                            <p>{formatPrice(totalPrice)}</p>
                          </div>
                          {bulkDiscount > 0 && (
                            <div className="flex justify-between text-green-600">
                              <p>Bulk discount</p>
                              <p>-{formatPrice(bulkDiscount)}</p>
                            </div>
                          )}
                          {appliedPromoCode && (
                            <div className="flex justify-between text-green-600">
                              <p>Promo code discount</p>
                              <p>-{formatPrice(totalPrice * (10 / 100))}</p>
                            </div>
                          )}
                        </div>
                        
                        <div className="flex justify-between text-base font-medium text-gray-900 mt-4 pt-4 border-t">
                          <p>Total</p>
                          <div className="text-right">
                            {totalSavings > 0 && (
                              <p className="text-sm text-green-600 line-through">
                                {formatPrice(totalPrice)}
                              </p>
                            )}
                            <p className="text-xl">{formatPrice(finalPrice)}</p>
                          </div>
                        </div>
                        
                        <p className="mt-0.5 text-sm text-gray-500">
                          Shipping and taxes calculated at checkout.
                        </p>
                        
                        <div className="mt-6">
                          <Link
                            href="/checkout"
                            className="flex items-center justify-center rounded-md border border-transparent bg-blue-600 px-6 py-3 text-base font-medium text-white shadow-sm hover:bg-blue-700 transition-colors"
                            onClick={closeCart}
                          >
                            Checkout
                          </Link>
                        </div>
                        
                        <div className="mt-6 flex justify-center text-center text-sm text-gray-500">
                          <p>
                            or{' '}
                            <button
                              type="button"
                              className="font-medium text-blue-600 hover:text-blue-500"
                              onClick={closeCart}
                            >
                              Continue Shopping
                              <span aria-hidden="true"> &rarr;</span>
                            </button>
                          </p>
                        </div>
                      </div>
                    )}
                  </div>
                </Dialog.Panel>
              </Transition.Child>
            </div>
          </div>
        </div>
      </Dialog>
    </Transition.Root>
  )
}