'use client'

import { 
  ShoppingCartIcon, 
  Bars3Icon, 
  XMarkIcon, 
  PhoneIcon,
  CodeBracketIcon,
  DevicePhoneMobileIcon,
  MegaphoneIcon,
  PaintBrushIcon,
  CloudIcon,
  MagnifyingGlassIcon,
  SparklesIcon,
  ServerIcon,
  ServerStackIcon,
  CpuChipIcon,
  GlobeAltIcon
} from '@heroicons/react/24/outline'
import { useHydrationSafeCart } from '@/hooks/useHydrationSafeCart'
import { useUIStore } from '@/store/ui'
import { useSession, signOut } from 'next-auth/react'
import Link from 'next/link'
import { cn } from '@/lib/utils'
import { useHydrationSafe } from '@/hooks/useHydrationSafe'
import { useEffect, useState } from 'react'
import NuwairLogo from '@/components/NuwairLogo'
import NotificationSystem from '@/components/NotificationSystem'

const navigation = [
  { name: 'Home', href: '/' },
  { 
    name: 'Services', 
    href: '/services',
    submenu: [
      { 
        name: 'Website Development', 
        href: '/services/website-development',
        icon: CodeBracketIcon,
        color: 'blue',
        description: 'Custom websites & web apps'
      },
      { 
        name: 'Mobile Apps', 
        href: '/services/mobile-app-development',
        icon: DevicePhoneMobileIcon,
        color: 'purple',
        description: 'iOS & Android applications'
      },
      { 
        name: 'Social Media Marketing', 
        href: '/services/social-media-marketing',
        icon: MegaphoneIcon,
        color: 'pink',
        description: 'Boost your social presence'
      },
      { 
        name: 'UI/UX Design', 
        href: '/services/ui-ux-design',
        icon: PaintBrushIcon,
        color: 'orange',
        description: 'Beautiful user experiences'
      },
      { 
        name: 'Cloud Solutions', 
        href: '/services/cloud-solutions',
        icon: CloudIcon,
        color: 'indigo',
        description: 'Scalable cloud infrastructure'
      },
      { 
        name: 'SEO Services', 
        href: '/services/seo-optimization',
        icon: MagnifyingGlassIcon,
        color: 'green',
        description: 'Rank higher on Google'
      },
    ]
  },
  { 
    name: 'Web Hosting', 
    href: '/shared-hosting',
    submenu: [
      { 
        name: 'Shared Hosting', 
        href: '/shared-hosting',
        icon: ServerIcon,
        color: 'blue',
        description: 'Affordable shared hosting plans'
      },
      { 
        name: 'Reseller Hosting', 
        href: '/reseller-hosting',
        icon: ServerStackIcon,
        color: 'green',
        description: 'Start your hosting business'
      },
      { 
        name: 'Dedicated Server', 
        href: '/dedicated-server',
        icon: CpuChipIcon,
        color: 'red',
        description: 'Powerful dedicated servers'
      },
      { 
        name: 'VPS Hosting', 
        href: '/vps-hosting',
        icon: CloudIcon,
        color: 'purple',
        description: 'Flexible VPS solutions'
      },
    ]
  },
  { name: 'Domain Name', href: '/domains' },
  { name: 'Blog', href: '/blog' },
  { name: 'Contact', href: '/contact' },
]

export default function Header() {
  const { data: session } = useSession()
  const { getTotalItems, openCart } = useHydrationSafeCart()
  const { mobileMenuOpen, setMobileMenuOpen } = useUIStore()
  const isHydrated = useHydrationSafe()
  const [mounted, setMounted] = useState(false)
  const totalItems = getTotalItems()

  // Additional mount check for extra safety
  useEffect(() => {
    setMounted(true)
  }, [])

  // Render with safe defaults during SSR
  const safeItemCount = mounted && isHydrated ? totalItems : 0

  return (
    <header className="bg-white shadow-sm sticky top-0 z-50">
      {/* Corporate Top Bar */}
      <div className="bg-slate-900 text-white py-2.5 text-sm border-b border-slate-800">
        <div className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center">
            <div className="flex items-center space-x-6 font-medium">
              <a href="tel:+923111689247" className="flex items-center space-x-2 hover:text-blue-400 transition-colors group">
                <PhoneIcon className="w-4 h-4 group-hover:scale-110 transition-transform" />
                <span className="tracking-wide">+92 3111 689 247</span>
              </a>
              <span className="text-slate-600 hidden sm:inline">|</span>
              <a href="mailto:sales@nuwair.com" className="hover:text-blue-400 transition-colors tracking-wide hidden sm:inline">
                sales@nuwair.com
              </a>
            </div>
            <div className="hidden lg:flex items-center space-x-4 text-xs uppercase">
              <span className="text-slate-400">🏆 Microsoft & AWS Certified</span>
              <span className="text-slate-600">|</span>
              <span className="text-slate-400">⚡ 99.9% Uptime SLA</span>
              <span className="text-slate-600">|</span>
              <span className="text-slate-400">🌐 Global Network</span>
            </div>
          </div>
        </div>
      </div>
      
      {/* Main Navigation - Corporate Style */}
      <div className="border-b border-slate-200 bg-white">
        <nav className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8" aria-label="Top">
          <div className="flex h-18 items-center justify-between">
            {/* Professional Logo */}
            <div className="flex items-center">
              <Link href="/" className="group relative py-3">
                <NuwairLogo 
                  className="transition-opacity duration-200 group-hover:opacity-80" 
                  width={140} 
                  height={40}
                />
              </Link>
            </div>

            {/* Desktop Navigation - Corporate Professional */}
            <div className="hidden lg:flex lg:items-center lg:space-x-1">
              {navigation.map((item) => (
                <div key={item.name} className="relative group">
                <Link
                  href={item.href}
                  className="relative text-slate-700 hover:text-blue-600 px-4 py-2 text-[15px] font-medium tracking-wide transition-colors duration-200 flex items-center gap-1.5"
                >
                  {item.name}
                  {item.submenu && (
                    <svg className="w-4 h-4 transition-transform group-hover:rotate-180 duration-200" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                    </svg>
                  )}
                </Link>
                {item.submenu && (
                  <div className="absolute left-1/2 -translate-x-1/2 mt-2 w-max opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-200 transform group-hover:translate-y-0 translate-y-2">
                    <div className="bg-white rounded-xl shadow-2xl border border-slate-200 p-3 min-w-[240px]">
                      <div className="space-y-1">
                        {item.submenu.map((subItem) => {
                          const Icon = subItem.icon
                          const colorClasses = {
                            blue: 'text-blue-600 group-hover/item:bg-blue-50',
                            purple: 'text-purple-600 group-hover/item:bg-purple-50',
                            pink: 'text-pink-600 group-hover/item:bg-pink-50',
                            orange: 'text-orange-600 group-hover/item:bg-orange-50',
                            indigo: 'text-indigo-600 group-hover/item:bg-indigo-50',
                            green: 'text-green-600 group-hover/item:bg-green-50',
                            red: 'text-red-600 group-hover/item:bg-red-50',
                          }
                          return (
                            <Link
                              key={subItem.name}
                              href={subItem.href}
                              className="group/item flex items-center gap-3 px-4 py-3 rounded-lg hover:bg-slate-50 transition-all duration-200"
                            >
                              {Icon && (
                                <div className={`w-10 h-10 rounded-lg flex items-center justify-center ${colorClasses[subItem.color as keyof typeof colorClasses]} transition-all duration-200`}>
                                  <Icon className="w-5 h-5" strokeWidth={2} />
                                </div>
                              )}
                              <div className="flex-1">
                                <div className="font-medium text-sm text-slate-900 group-hover/item:text-blue-600 transition-colors">
                                  {subItem.name}
                                </div>
                                <div className="text-xs text-slate-500 mt-0.5">
                                  {subItem.description}
                                </div>
                              </div>
                            </Link>
                          )
                        })}
                      </div>
                    </div>
                  </div>
                )}
              </div>
            ))}
          </div>

          {/* Right side - Corporate Professional */}
          <div className="flex items-center space-x-4">
            {/* Cart with artistic badge */}
            <button
              onClick={openCart}
              className="relative p-2 text-slate-600 hover:text-blue-600 transition-colors duration-200 group"
            >
              <ShoppingCartIcon className="h-6 w-6" />
              {safeItemCount > 0 && (
                <span className="absolute -top-1 -right-1 h-5 w-5 rounded-full bg-blue-600 text-white text-xs flex items-center justify-center font-semibold shadow-md">
                  {safeItemCount}
                </span>
              )}
            </button>

            {/* Notifications */}
            {session && <NotificationSystem />}

            {/* Auth buttons - Corporate */}
            {session ? (
              <div className="hidden lg:flex items-center space-x-3">
                <span className="text-sm text-slate-600 bg-slate-100 px-3 py-1.5 rounded-lg font-medium">
                  {session.user?.name || session.user?.email}
                </span>
                <Link
                  href={session.user?.role === 'ADMIN' ? '/admin' : '/dashboard'}
                  className="text-sm font-medium text-slate-700 hover:text-blue-600 px-4 py-2 hover:bg-blue-50 rounded-lg transition-colors"
                >
                  {session.user?.role === 'ADMIN' ? 'Admin Panel' : 'Dashboard'}
                </Link>
                <button
                  onClick={() => signOut()}
                  className="text-sm font-medium text-slate-700 hover:text-red-600 px-4 py-2 hover:bg-red-50 rounded-lg transition-colors"
                >
                  Sign Out
                </button>
              </div>
            ) : (
              <div className="hidden lg:flex items-center space-x-2">
                <Link
                  href="/auth/signin"
                  className="text-sm font-medium text-slate-700 hover:text-blue-600 px-4 py-2 hover:bg-blue-50 rounded-lg transition-colors"
                >
                  Sign In
                </Link>
                <Link
                  href="/auth/signup"
                  className="bg-blue-600 text-white px-6 py-2 rounded-lg text-sm font-semibold hover:bg-blue-700 transition-colors shadow-sm"
                >
                  Get Started
                </Link>
              </div>
            )}

            {/* Mobile menu button - Corporate */}
            <button
              className="lg:hidden p-2 text-slate-700 hover:bg-slate-100 rounded-lg transition-colors"
              onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
            >
              {mobileMenuOpen ? (
                <XMarkIcon className="h-6 w-6" />
              ) : (
                <Bars3Icon className="h-6 w-6" />
              )}
            </button>
          </div>
        </div>

        {/* Mobile Navigation */}
        <div className={cn(
          'lg:hidden overflow-hidden transition-all duration-300',
          mobileMenuOpen ? 'max-h-96 opacity-100' : 'max-h-0 opacity-0'
        )}>
          <div className="pt-4 pb-6 space-y-2">
            {navigation.map((item) => (
              <div key={item.name}>
                <Link
                  href={item.href}
                  className="block px-3 py-2 text-base font-medium text-gray-700 hover:text-blue-600 hover:bg-gray-50 rounded-md"
                  onClick={() => setMobileMenuOpen(false)}
                >
                  {item.name}
                </Link>
                {item.submenu && (
                  <div className="ml-2 mt-2 space-y-2 bg-gradient-to-br from-gray-50 to-blue-50/30 rounded-xl p-2">
                    {item.submenu.map((subItem) => {
                      const Icon = subItem.icon
                      const colorClasses = {
                        blue: 'bg-blue-100 text-blue-600',
                        purple: 'bg-purple-100 text-purple-600',
                        pink: 'bg-pink-100 text-pink-600',
                        orange: 'bg-orange-100 text-orange-600',
                        indigo: 'bg-indigo-100 text-indigo-600',
                        green: 'bg-green-100 text-green-600',
                      }
                      return (
                        <Link
                          key={subItem.name}
                          href={subItem.href}
                          className="flex items-center gap-3 px-3 py-2.5 text-sm text-gray-700 hover:text-blue-600 hover:bg-white rounded-lg transition-all duration-200 group/mobile"
                          onClick={() => setMobileMenuOpen(false)}
                        >
                          <div className={`w-9 h-9 rounded-lg flex items-center justify-center transition-all duration-300 ${colorClasses[subItem.color as keyof typeof colorClasses]}`}>
                            {Icon && <Icon className="w-5 h-5" />}
                          </div>
                          <div className="flex-1">
                            <div className="font-semibold">{subItem.name}</div>
                            <div className="text-xs text-gray-500">{subItem.description}</div>
                          </div>
                        </Link>
                      )
                    })}
                  </div>
                )}
              </div>
            ))}
            
            {/* Mobile auth buttons */}
            <div className="pt-4 border-t border-gray-200">
              {session ? (
                <div className="space-y-2">
                  <Link
                    href="/dashboard"
                    className="block px-3 py-2 text-base font-medium text-gray-700 hover:text-blue-600 hover:bg-gray-50 rounded-md"
                    onClick={() => setMobileMenuOpen(false)}
                  >
                    Dashboard
                  </Link>
                  <button
                    onClick={() => {
                      signOut()
                      setMobileMenuOpen(false)
                    }}
                    className="block w-full text-left px-3 py-2 text-base font-medium text-gray-700 hover:text-blue-600 hover:bg-gray-50 rounded-md"
                  >
                    Sign Out
                  </button>
                </div>
              ) : (
                <div className="space-y-2">
                  <Link
                    href="/auth/signin"
                    className="block px-3 py-2 text-base font-medium text-gray-700 hover:text-blue-600 hover:bg-gray-50 rounded-md"
                    onClick={() => setMobileMenuOpen(false)}
                  >
                    Sign In
                  </Link>
                  <Link
                    href="/auth/signup"
                    className="block px-3 py-2 text-base font-medium bg-blue-600 text-white hover:bg-blue-700 rounded-md"
                    onClick={() => setMobileMenuOpen(false)}
                  >
                    Get Started
                  </Link>
                </div>
              )}
            </div>
          </div>
        </div>
        </nav>
      </div>
    </header>
  )
}
