'use client'

import { useHydrationSafe } from '@/hooks/useHydrationSafe'
import { ReactNode } from 'react'

interface HydrationBoundaryProps {
  children: ReactNode
  fallback?: ReactNode
  suppressWarning?: boolean
}

/**
 * A comprehensive hydration boundary component that prevents hydration mismatches
 * by ensuring content only renders after client-side hydration is complete.
 * 
 * This component should wrap any elements that:
 * - Use Date.now(), Math.random(), or other dynamic values
 * - Depend on browser APIs or window object
 * - May be modified by browser extensions
 * - Have client-side only state or functionality
 */
export default function HydrationBoundary({ 
  children, 
  fallback = null, 
  suppressWarning = true 
}: HydrationBoundaryProps) {
  const isHydrated = useHydrationSafe()

  if (!isHydrated) {
    return (
      <div suppressHydrationWarning={suppressWarning}>
        {fallback}
      </div>
    )
  }

  return (
    <div suppressHydrationWarning={suppressWarning}>
      {children}
    </div>
  )
}