'use client'

import { useState } from 'react'
import { motion } from 'framer-motion'
import {
  DocumentTextIcon,
  EnvelopeIcon,
  PrinterIcon,
  ArrowDownTrayIcon,
  CurrencyDollarIcon,
  CalendarIcon,
  UserIcon,
} from '@heroicons/react/24/outline'

interface InvoiceData {
  id: string
  invoiceNumber: string
  description: string
  amount: number
  subtotal: number
  taxAmount: number
  taxRate: number
  discountAmount: number
  currency: string
  status: 'PENDING' | 'SENT' | 'PAID' | 'OVERDUE' | 'CANCELLED'
  dueDate: string
  paidDate?: string
  billingName: string
  billingEmail: string
  billingAddress: any
  createdAt: string
  order?: {
    id: string
    items: Array<{
      service: {
        name: string
        description: string
      }
      quantity: number
      price: number
    }>
  }
}

interface InvoiceViewerProps {
  invoice: InvoiceData
  onSendInvoice?: (invoiceId: string) => void
  onDownloadPDF?: (invoiceId: string) => void
  isAdmin?: boolean
}

const InvoiceViewer: React.FC<InvoiceViewerProps> = ({
  invoice,
  onSendInvoice,
  onDownloadPDF,
  isAdmin = false
}) => {
  const [isSending, setIsSending] = useState(false)

  const handleSendInvoice = async () => {
    if (!onSendInvoice) return
    setIsSending(true)
    try {
      await onSendInvoice(invoice.id)
    } finally {
      setIsSending(false)
    }
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'PAID':
        return 'bg-green-100 text-green-800'
      case 'SENT':
        return 'bg-blue-100 text-blue-800'
      case 'OVERDUE':
        return 'bg-red-100 text-red-800'
      case 'CANCELLED':
        return 'bg-gray-100 text-gray-800'
      default:
        return 'bg-yellow-100 text-yellow-800'
    }
  }

  return (
    <motion.div
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      className="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden"
    >
      {/* Header */}
      <div className="bg-gradient-to-r from-blue-600 to-blue-700 text-white p-6">
        <div className="flex justify-between items-start">
          <div>
            <h1 className="text-2xl font-bold mb-2">INVOICE</h1>
            <p className="text-blue-100">#{invoice.invoiceNumber}</p>
          </div>
          <div className="text-right">
            <div className="mb-2">
              <span className={`inline-flex px-3 py-1 text-sm font-semibold rounded-full ${getStatusColor(invoice.status)}`}>
                {invoice.status}
              </span>
            </div>
            <p className="text-blue-100 text-sm">
              Created: {new Date(invoice.createdAt).toLocaleDateString()}
            </p>
          </div>
        </div>
      </div>

      {/* Company Info */}
      <div className="p-6 border-b border-gray-200">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div>
            <h3 className="text-lg font-semibold text-gray-900 mb-3">From:</h3>
            <div className="text-gray-600">
              <p className="font-semibold text-gray-900">Nuwair Systems</p>
              <p>Digital Solutions & Hosting</p>
              <p>support@nuwairsystems.com</p>
              <p>+1 (555) 123-4567</p>
            </div>
          </div>
          <div>
            <h3 className="text-lg font-semibold text-gray-900 mb-3">Bill To:</h3>
            <div className="text-gray-600">
              <p className="font-semibold text-gray-900">{invoice.billingName}</p>
              <p>{invoice.billingEmail}</p>
              {typeof invoice.billingAddress === 'object' && (
                <>
                  <p>{invoice.billingAddress.street}</p>
                  <p>{invoice.billingAddress.city}, {invoice.billingAddress.state} {invoice.billingAddress.zipCode}</p>
                  <p>{invoice.billingAddress.country}</p>
                </>
              )}
            </div>
          </div>
        </div>
      </div>

      {/* Invoice Details */}
      <div className="p-6">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
          <div className="flex items-center">
            <CalendarIcon className="w-5 h-5 text-gray-400 mr-2" />
            <div>
              <p className="text-sm text-gray-500">Due Date</p>
              <p className="font-medium text-gray-900">
                {new Date(invoice.dueDate).toLocaleDateString()}
              </p>
            </div>
          </div>
          {invoice.paidDate && (
            <div className="flex items-center">
              <CurrencyDollarIcon className="w-5 h-5 text-green-500 mr-2" />
              <div>
                <p className="text-sm text-gray-500">Paid Date</p>
                <p className="font-medium text-gray-900">
                  {new Date(invoice.paidDate).toLocaleDateString()}
                </p>
              </div>
            </div>
          )}
          <div className="flex items-center">
            <DocumentTextIcon className="w-5 h-5 text-blue-500 mr-2" />
            <div>
              <p className="text-sm text-gray-500">Amount</p>
              <p className="font-bold text-lg text-gray-900">
                ${invoice.amount.toFixed(2)} {invoice.currency.toUpperCase()}
              </p>
            </div>
          </div>
        </div>

        {/* Description */}
        <div className="mb-6">
          <h4 className="text-sm font-medium text-gray-700 mb-2">Description</h4>
          <p className="text-gray-600">{invoice.description}</p>
        </div>

        {/* Line Items */}
        {invoice.order?.items && invoice.order.items.length > 0 && (
          <div className="mb-6">
            <h4 className="text-sm font-medium text-gray-700 mb-3">Items</h4>
            <div className="overflow-x-auto">
              <table className="min-w-full border border-gray-200 rounded-lg">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Service
                    </th>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Quantity
                    </th>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Price
                    </th>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Total
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {invoice.order.items.map((item, index) => (
                    <tr key={index}>
                      <td className="px-4 py-3">
                        <div>
                          <p className="text-sm font-medium text-gray-900">{item.service.name}</p>
                          <p className="text-sm text-gray-500">{item.service.description}</p>
                        </div>
                      </td>
                      <td className="px-4 py-3 text-sm text-gray-900">{item.quantity}</td>
                      <td className="px-4 py-3 text-sm text-gray-900">${item.price.toFixed(2)}</td>
                      <td className="px-4 py-3 text-sm font-medium text-gray-900">
                        ${(item.price * item.quantity).toFixed(2)}
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}

        {/* Totals */}
        <div className="border-t border-gray-200 pt-6">
          <div className="space-y-2 max-w-xs ml-auto">
            <div className="flex justify-between text-sm">
              <span className="text-gray-600">Subtotal:</span>
              <span className="text-gray-900">${invoice.subtotal.toFixed(2)}</span>
            </div>
            {invoice.discountAmount > 0 && (
              <div className="flex justify-between text-sm">
                <span className="text-gray-600">Discount:</span>
                <span className="text-red-600">-${invoice.discountAmount.toFixed(2)}</span>
              </div>
            )}
            {invoice.taxAmount > 0 && (
              <div className="flex justify-between text-sm">
                <span className="text-gray-600">Tax ({invoice.taxRate}%):</span>
                <span className="text-gray-900">${invoice.taxAmount.toFixed(2)}</span>
              </div>
            )}
            <div className="flex justify-between text-lg font-bold border-t border-gray-200 pt-2">
              <span>Total:</span>
              <span>${invoice.amount.toFixed(2)} {invoice.currency.toUpperCase()}</span>
            </div>
          </div>
        </div>

        {/* Actions */}
        {isAdmin && (
          <div className="flex flex-wrap gap-3 mt-6 pt-6 border-t border-gray-200">
            <button
              onClick={handleSendInvoice}
              disabled={isSending || invoice.status === 'PAID'}
              className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
            >
              <EnvelopeIcon className="w-4 h-4 mr-2" />
              {isSending ? 'Sending...' : 'Send Invoice'}
            </button>
            <button
              onClick={() => onDownloadPDF?.(invoice.id)}
              className="flex items-center px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700 focus:outline-none focus:ring-2 focus:ring-gray-500 focus:ring-offset-2 transition-colors"
            >
              <ArrowDownTrayIcon className="w-4 h-4 mr-2" />
              Download PDF
            </button>
            <button
              onClick={() => window.print()}
              className="flex items-center px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2 transition-colors"
            >
              <PrinterIcon className="w-4 h-4 mr-2" />
              Print
            </button>
          </div>
        )}
      </div>
    </motion.div>
  )
}

export default InvoiceViewer