'use client'

import { useState, useEffect, useRef } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { useSession } from 'next-auth/react'
import {
  ChatBubbleLeftRightIcon,
  XMarkIcon,
  PaperAirplaneIcon,
  PaperClipIcon,
  FaceSmileIcon,
  UserIcon,
  PhoneIcon,
  VideoCameraIcon,
  ClockIcon,
  CheckIcon
} from '@heroicons/react/24/outline'

interface Message {
  id: string
  content: string
  sender: 'user' | 'agent' | 'ai' | 'system'
  timestamp: Date
  type: 'text' | 'image' | 'file' | 'system'
  metadata?: {
    agentName?: string
    agentAvatar?: string
    fileName?: string
    fileSize?: string
    isTyping?: boolean
  }
}

interface ChatAgent {
  id: string
  name: string
  avatar: string
  status: 'online' | 'away' | 'busy'
  specialty: string
  responseTime: string
}

const CHAT_AGENTS: ChatAgent[] = [
  {
    id: 'agent-1',
    name: 'Sarah Ahmed',
    avatar: '/agents/sarah.jpg',
    status: 'online',
    specialty: 'Web Development',
    responseTime: '< 2 min'
  },
  {
    id: 'agent-2', 
    name: 'Ali Hassan',
    avatar: '/agents/ali.jpg',
    status: 'online',
    specialty: 'Digital Marketing',
    responseTime: '< 5 min'
  },
  {
    id: 'agent-3',
    name: 'Fatima Khan',
    avatar: '/agents/fatima.jpg',
    status: 'away',
    specialty: 'Technical Support',
    responseTime: '< 10 min'
  }
]

const AI_RESPONSES = [
  "Hello! I'm here to help you with any questions about our services. How can I assist you today?",
  "Thank you for your message! I'm connecting you with one of our specialists who will be with you shortly.",
  "I understand you're interested in our web development services. Let me get you connected with our technical team.",
  "For urgent matters, feel free to call us at +92-300-1234567 or schedule a video consultation.",
  "I've created a support ticket for your inquiry. One of our team members will follow up with you within 24 hours."
]

export default function LiveChatWidget() {
  const { data: session } = useSession()
  const [isOpen, setIsOpen] = useState(false)
  const [messages, setMessages] = useState<Message[]>([])
  const [inputValue, setInputValue] = useState('')
  const [isTyping, setIsTyping] = useState(false)
  const [currentAgent, setCurrentAgent] = useState<ChatAgent | null>(null)
  const [chatStage, setChatStage] = useState<'initial' | 'ai' | 'agent'>('initial')
  const [showAgentSelection, setShowAgentSelection] = useState(false)
  const messagesEndRef = useRef<HTMLDivElement>(null)
  const fileInputRef = useRef<HTMLInputElement>(null)

  useEffect(() => {
    scrollToBottom()
  }, [messages])

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' })
  }

  const initializeChat = () => {
    if (messages.length === 0) {
      const welcomeMessage: Message = {
        id: 'welcome',
        content: session?.user?.name 
          ? `Welcome back, ${session.user.name}! How can we help you today?`
          : "Welcome to Nuwair Systems! I'm your AI assistant. How can I help you today?",
        sender: 'ai',
        timestamp: new Date(),
        type: 'text',
        metadata: {
          agentName: 'AI Assistant',
          agentAvatar: '/ai-avatar.png'
        }
      }
      setMessages([welcomeMessage])
      setChatStage('ai')
    }
  }

  const handleSendMessage = async () => {
    if (!inputValue.trim()) return

    const userMessage: Message = {
      id: `msg-${Date.now()}`,
      content: inputValue,
      sender: 'user',
      timestamp: new Date(),
      type: 'text'
    }

    setMessages(prev => [...prev, userMessage])
    setInputValue('')
    setIsTyping(true)

    // Simulate AI or agent response
    setTimeout(() => {
      let responseContent = ''
      
      if (chatStage === 'ai') {
        // AI response logic
        const input = inputValue.toLowerCase()
        if (input.includes('price') || input.includes('cost')) {
          responseContent = "I can help you with pricing information! Our services start from $250 for graphic design up to $4,500 for mobile app development. Would you like me to connect you with a specialist for a detailed quote?"
        } else if (input.includes('web') || input.includes('website')) {
          responseContent = "Great! Our web development services include responsive design, SEO optimization, and modern technologies. Let me connect you with Sarah, our web development specialist."
          setCurrentAgent(CHAT_AGENTS[0])
          setChatStage('agent')
        } else if (input.includes('marketing') || input.includes('seo')) {
          responseContent = "Perfect! Our digital marketing team can help with SEO, social media, and PPC campaigns. I'll connect you with Ali, our marketing expert."
          setCurrentAgent(CHAT_AGENTS[1])
          setChatStage('agent')
        } else {
          responseContent = AI_RESPONSES[Math.floor(Math.random() * AI_RESPONSES.length)]
        }
      } else {
        // Agent response
        responseContent = `Thanks for your question! I'm reviewing your requirements and will provide you with a detailed proposal shortly. In the meantime, would you like to schedule a video call to discuss your project in detail?`
      }

      const responseMessage: Message = {
        id: `response-${Date.now()}`,
        content: responseContent,
        sender: chatStage === 'ai' ? 'ai' : 'agent',
        timestamp: new Date(),
        type: 'text',
        metadata: {
          agentName: chatStage === 'ai' ? 'AI Assistant' : currentAgent?.name,
          agentAvatar: chatStage === 'ai' ? '/ai-avatar.png' : currentAgent?.avatar
        }
      }

      setMessages(prev => [...prev, responseMessage])
      setIsTyping(false)
    }, 2000 + Math.random() * 2000)
  }

  const handleFileUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0]
    if (!file) return

    const fileMessage: Message = {
      id: `file-${Date.now()}`,
      content: `Uploaded: ${file.name}`,
      sender: 'user',
      timestamp: new Date(),
      type: 'file',
      metadata: {
        fileName: file.name,
        fileSize: `${(file.size / 1024 / 1024).toFixed(1)} MB`
      }
    }

    setMessages(prev => [...prev, fileMessage])
  }

  const requestHumanAgent = () => {
    setShowAgentSelection(true)
  }

  const selectAgent = (agent: ChatAgent) => {
    setCurrentAgent(agent)
    setChatStage('agent')
    setShowAgentSelection(false)
    
    const transferMessage: Message = {
      id: `transfer-${Date.now()}`,
      content: `You've been connected with ${agent.name}, our ${agent.specialty} specialist. They'll be with you shortly!`,
      sender: 'system',
      timestamp: new Date(),
      type: 'system'
    }
    
    setMessages(prev => [...prev, transferMessage])
  }

  const scheduleVideoCall = () => {
    // Integration with calendar scheduling
    window.open('https://calendly.com/nuwairsystems/consultation', '_blank')
  }

  return (
    <>
      {/* Chat Toggle Button */}
      <motion.button
        onClick={() => {
          setIsOpen(true)
          initializeChat()
        }}
        className={`fixed bottom-6 right-6 z-50 bg-gradient-to-r from-blue-600 to-purple-600 text-white rounded-full p-4 shadow-lg hover:shadow-xl transition-all duration-300 ${isOpen ? 'hidden' : 'block'}`}
        whileHover={{ scale: 1.1 }}
        whileTap={{ scale: 0.9 }}
        initial={{ opacity: 0, y: 100 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ delay: 2 }}
      >
        <ChatBubbleLeftRightIcon className="w-6 h-6" />
        <div className="absolute -top-1 -right-1 w-3 h-3 bg-green-500 rounded-full animate-pulse"></div>
      </motion.button>

      {/* Chat Window */}
      <AnimatePresence>
        {isOpen && (
          <motion.div
            initial={{ opacity: 0, y: 20, scale: 0.9 }}
            animate={{ opacity: 1, y: 0, scale: 1 }}
            exit={{ opacity: 0, y: 20, scale: 0.9 }}
            className="fixed bottom-6 right-6 z-50 w-96 h-[600px] bg-white rounded-lg shadow-2xl flex flex-col overflow-hidden"
          >
            {/* Chat Header */}
            <div className="bg-gradient-to-r from-blue-600 to-purple-600 text-white p-4 flex items-center justify-between">
              <div className="flex items-center space-x-3">
                <div className="relative">
                  <div className="w-10 h-10 bg-white/20 rounded-full flex items-center justify-center">
                    {currentAgent ? (
                      <img 
                        src={currentAgent.avatar} 
                        alt={currentAgent.name}
                        className="w-8 h-8 rounded-full"
                        onError={(e) => {
                          e.currentTarget.src = '/placeholder-agent.png'
                        }}
                      />
                    ) : (
                      <ChatBubbleLeftRightIcon className="w-5 h-5" />
                    )}
                  </div>
                  <div className="absolute -bottom-1 -right-1 w-3 h-3 bg-green-400 rounded-full border-2 border-white"></div>
                </div>
                <div>
                  <h3 className="font-semibold">
                    {currentAgent ? currentAgent.name : 'Nuwair Support'}
                  </h3>
                  <p className="text-xs opacity-90">
                    {currentAgent ? `${currentAgent.specialty} • ${currentAgent.responseTime}` : 'We\'re online and ready to help!'}
                  </p>
                </div>
              </div>
              <div className="flex items-center space-x-2">
                {currentAgent && (
                  <>
                    <button
                      onClick={scheduleVideoCall}
                      className="p-2 hover:bg-white/20 rounded-full transition-colors"
                      title="Schedule Video Call"
                    >
                      <VideoCameraIcon className="w-4 h-4" />
                    </button>
                    <button
                      className="p-2 hover:bg-white/20 rounded-full transition-colors"
                      title="Call"
                    >
                      <PhoneIcon className="w-4 h-4" />
                    </button>
                  </>
                )}
                <button
                  onClick={() => setIsOpen(false)}
                  className="p-2 hover:bg-white/20 rounded-full transition-colors"
                >
                  <XMarkIcon className="w-4 h-4" />
                </button>
              </div>
            </div>

            {/* Agent Selection Modal */}
            {showAgentSelection && (
              <div className="absolute inset-0 bg-white z-10 flex flex-col">
                <div className="p-4 border-b">
                  <h3 className="font-semibold text-gray-900">Choose a Specialist</h3>
                  <p className="text-sm text-gray-600">Select an expert to help with your needs</p>
                </div>
                <div className="flex-1 p-4 space-y-3">
                  {CHAT_AGENTS.map((agent) => (
                    <button
                      key={agent.id}
                      onClick={() => selectAgent(agent)}
                      className="w-full p-3 border rounded-lg hover:bg-gray-50 transition-colors text-left"
                    >
                      <div className="flex items-center space-x-3">
                        <div className="relative">
                          <img 
                            src={agent.avatar} 
                            alt={agent.name}
                            className="w-10 h-10 rounded-full"
                            onError={(e) => {
                              e.currentTarget.src = '/placeholder-agent.png'
                            }}
                          />
                          <div className={`absolute -bottom-1 -right-1 w-3 h-3 rounded-full border-2 border-white ${
                            agent.status === 'online' ? 'bg-green-400' :
                            agent.status === 'away' ? 'bg-yellow-400' : 'bg-red-400'
                          }`}></div>
                        </div>
                        <div className="flex-1">
                          <h4 className="font-medium text-gray-900">{agent.name}</h4>
                          <p className="text-sm text-gray-600">{agent.specialty}</p>
                          <p className="text-xs text-gray-500">Response time: {agent.responseTime}</p>
                        </div>
                        <div className={`px-2 py-1 rounded-full text-xs font-medium ${
                          agent.status === 'online' ? 'bg-green-100 text-green-800' :
                          agent.status === 'away' ? 'bg-yellow-100 text-yellow-800' : 'bg-red-100 text-red-800'
                        }`}>
                          {agent.status}
                        </div>
                      </div>
                    </button>
                  ))}
                </div>
                <div className="p-4 border-t">
                  <button
                    onClick={() => setShowAgentSelection(false)}
                    className="w-full py-2 text-gray-600 hover:text-gray-800"
                  >
                    Continue with AI Assistant
                  </button>
                </div>
              </div>
            )}

            {/* Messages Area */}
            <div className="flex-1 overflow-y-auto p-4 space-y-4 bg-gray-50">
              {messages.map((message) => (
                <div
                  key={message.id}
                  className={`flex ${message.sender === 'user' ? 'justify-end' : 'justify-start'}`}
                >
                  <div className={`max-w-[80%] ${
                    message.sender === 'user' 
                      ? 'bg-blue-600 text-white' 
                      : message.sender === 'system'
                      ? 'bg-yellow-100 text-yellow-800 border border-yellow-200'
                      : 'bg-white text-gray-900 border'
                  } rounded-lg p-3 shadow-sm`}>
                    {message.sender !== 'user' && message.metadata?.agentName && (
                      <div className="flex items-center space-x-2 mb-2">
                        <img 
                          src={message.metadata.agentAvatar || '/placeholder-agent.png'} 
                          alt={message.metadata.agentName}
                          className="w-6 h-6 rounded-full"
                          onError={(e) => {
                            e.currentTarget.src = '/placeholder-agent.png'
                          }}
                        />
                        <span className="text-xs font-medium text-gray-600">
                          {message.metadata.agentName}
                        </span>
                      </div>
                    )}
                    
                    {message.type === 'file' && (
                      <div className="flex items-center space-x-2 text-sm">
                        <PaperClipIcon className="w-4 h-4" />
                        <span>{message.metadata?.fileName}</span>
                        <span className="text-xs opacity-75">({message.metadata?.fileSize})</span>
                      </div>
                    )}
                    
                    <p className="text-sm">{message.content}</p>
                    
                    <div className="flex items-center justify-between mt-2">
                      <span className="text-xs opacity-75">
                        {message.timestamp.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                      </span>
                      {message.sender === 'user' && (
                        <CheckIcon className="w-3 h-3 opacity-75" />
                      )}
                    </div>
                  </div>
                </div>
              ))}
              
              {/* Typing Indicator */}
              {isTyping && (
                <div className="flex justify-start">
                  <div className="bg-white border rounded-lg p-3 shadow-sm">
                    <div className="flex items-center space-x-2">
                      <div className="flex space-x-1">
                        <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce"></div>
                        <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce" style={{ animationDelay: '0.1s' }}></div>
                        <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce" style={{ animationDelay: '0.2s' }}></div>
                      </div>
                      <span className="text-xs text-gray-500">typing...</span>
                    </div>
                  </div>
                </div>
              )}
              
              <div ref={messagesEndRef} />
            </div>

            {/* Quick Actions */}
            <div className="px-4 py-2 bg-white border-t">
              <div className="flex space-x-2 text-xs">
                <button
                  onClick={requestHumanAgent}
                  className="px-3 py-1 bg-gray-100 text-gray-600 rounded-full hover:bg-gray-200 transition-colors"
                >
                  <UserIcon className="w-3 h-3 inline mr-1" />
                  Talk to Human
                </button>
                <button
                  onClick={scheduleVideoCall}
                  className="px-3 py-1 bg-gray-100 text-gray-600 rounded-full hover:bg-gray-200 transition-colors"
                >
                  <VideoCameraIcon className="w-3 h-3 inline mr-1" />
                  Video Call
                </button>
                <button className="px-3 py-1 bg-gray-100 text-gray-600 rounded-full hover:bg-gray-200 transition-colors">
                  <ClockIcon className="w-3 h-3 inline mr-1" />
                  Business Hours
                </button>
              </div>
            </div>

            {/* Input Area */}
            <div className="p-4 bg-white border-t">
              <div className="flex items-center space-x-2">
                <input
                  type="file"
                  ref={fileInputRef}
                  onChange={handleFileUpload}
                  className="hidden"
                  accept="image/*,.pdf,.doc,.docx"
                />
                <button
                  onClick={() => fileInputRef.current?.click()}
                  className="p-2 text-gray-400 hover:text-gray-600 transition-colors"
                >
                  <PaperClipIcon className="w-5 h-5" />
                </button>
                <input
                  type="text"
                  value={inputValue}
                  onChange={(e) => setInputValue(e.target.value)}
                  onKeyPress={(e) => e.key === 'Enter' && handleSendMessage()}
                  placeholder="Type your message..."
                  className="flex-1 p-2 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500"
                />
                <button
                  className="p-2 text-gray-400 hover:text-gray-600 transition-colors"
                >
                  <FaceSmileIcon className="w-5 h-5" />
                </button>
                <button
                  onClick={handleSendMessage}
                  disabled={!inputValue.trim()}
                  className="p-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                >
                  <PaperAirplaneIcon className="w-5 h-5" />
                </button>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>
    </>
  )
}