'use client'

import { useState, useEffect } from 'react'
import { useSession } from 'next-auth/react'
import { motion, AnimatePresence } from 'framer-motion'
import {
  BellIcon,
  XMarkIcon,
  CheckIcon,
  ExclamationTriangleIcon,
  InformationCircleIcon,
  ChatBubbleLeftRightIcon,
  TicketIcon,
  CreditCardIcon,
  ServerIcon
} from '@heroicons/react/24/outline'

interface Notification {
  id: string
  type: 'message' | 'ticket' | 'payment' | 'system' | 'project'
  title: string
  content: string
  timestamp: Date
  read: boolean
  urgent: boolean
  actionUrl?: string
  metadata?: {
    projectId?: string
    ticketId?: string
    invoiceId?: string
    fromUser?: string
  }
}

const MOCK_NOTIFICATIONS: Notification[] = [
  {
    id: 'notif-1',
    type: 'message',
    title: 'New Project Message',
    content: 'Sarah Ahmed replied to your message in E-commerce Website Development project',
    timestamp: new Date('2025-10-31T10:30:00'),
    read: false,
    urgent: false,
    actionUrl: '/dashboard/communication',
    metadata: {
      projectId: 'WEB-001',
      fromUser: 'Sarah Ahmed'
    }
  },
  {
    id: 'notif-2',
    type: 'ticket',
    title: 'Support Ticket Updated',
    content: 'Your ticket "SSL Certificate Issue" has been resolved',
    timestamp: new Date('2025-10-31T09:15:00'),
    read: false,
    urgent: false,
    actionUrl: '/dashboard/tickets',
    metadata: {
      ticketId: 'TKT-001'
    }
  },
  {
    id: 'notif-3',
    type: 'payment',
    title: 'Payment Reminder',
    content: 'Invoice #INV-2025-003 is due in 3 days',
    timestamp: new Date('2025-10-31T08:00:00'),
    read: true,
    urgent: true,
    actionUrl: '/dashboard/invoices',
    metadata: {
      invoiceId: 'INV-2025-003'
    }
  },
  {
    id: 'notif-4',
    type: 'project',
    title: 'Project Milestone',
    content: 'Design phase completed for Mobile App Development project',
    timestamp: new Date('2025-10-30T16:45:00'),
    read: true,
    urgent: false,
    actionUrl: '/dashboard/communication',
    metadata: {
      projectId: 'APP-001'
    }
  },
  {
    id: 'notif-5',
    type: 'system',
    title: 'System Maintenance',
    content: 'Scheduled maintenance on November 1st from 2:00 AM to 4:00 AM PST',
    timestamp: new Date('2025-10-30T12:00:00'),
    read: true,
    urgent: false
  }
]

interface NotificationSystemProps {
  showBadge?: boolean
  className?: string
}

export default function NotificationSystem({ showBadge = true, className = '' }: NotificationSystemProps) {
  const { data: session } = useSession()
  const [notifications, setNotifications] = useState<Notification[]>([])
  const [isOpen, setIsOpen] = useState(false)
  const [filter, setFilter] = useState<'all' | 'unread' | 'urgent'>('all')

  useEffect(() => {
    if (session?.user) {
      setNotifications(MOCK_NOTIFICATIONS)
    }
  }, [session])

  const unreadCount = notifications.filter(n => !n.read).length
  const urgentCount = notifications.filter(n => n.urgent && !n.read).length

  const filteredNotifications = notifications.filter(notification => {
    switch (filter) {
      case 'unread':
        return !notification.read
      case 'urgent':
        return notification.urgent
      default:
        return true
    }
  })

  const markAsRead = (notificationId: string) => {
    setNotifications(prev =>
      prev.map(n =>
        n.id === notificationId ? { ...n, read: true } : n
      )
    )
  }

  const markAllAsRead = () => {
    setNotifications(prev =>
      prev.map(n => ({ ...n, read: true }))
    )
  }

  const deleteNotification = (notificationId: string) => {
    setNotifications(prev =>
      prev.filter(n => n.id !== notificationId)
    )
  }

  const getNotificationIcon = (type: Notification['type']) => {
    switch (type) {
      case 'message':
        return <ChatBubbleLeftRightIcon className="w-5 h-5 text-blue-600" />
      case 'ticket':
        return <TicketIcon className="w-5 h-5 text-green-600" />
      case 'payment':
        return <CreditCardIcon className="w-5 h-5 text-red-600" />
      case 'project':
        return <ServerIcon className="w-5 h-5 text-purple-600" />
      case 'system':
        return <InformationCircleIcon className="w-5 h-5 text-yellow-600" />
      default:
        return <BellIcon className="w-5 h-5 text-gray-600" />
    }
  }

  const getTypeColor = (type: Notification['type']) => {
    switch (type) {
      case 'message':
        return 'bg-blue-50 border-blue-200'
      case 'ticket':
        return 'bg-green-50 border-green-200'
      case 'payment':
        return 'bg-red-50 border-red-200'
      case 'project':
        return 'bg-purple-50 border-purple-200'
      case 'system':
        return 'bg-yellow-50 border-yellow-200'
      default:
        return 'bg-gray-50 border-gray-200'
    }
  }

  const formatTimeAgo = (timestamp: Date) => {
    const now = new Date()
    const diffInMinutes = Math.floor((now.getTime() - timestamp.getTime()) / (1000 * 60))
    
    if (diffInMinutes < 1) return 'Just now'
    if (diffInMinutes < 60) return `${diffInMinutes}m ago`
    if (diffInMinutes < 1440) return `${Math.floor(diffInMinutes / 60)}h ago`
    return `${Math.floor(diffInMinutes / 1440)}d ago`
  }

  if (!session) return null

  return (
    <div className={`relative ${className}`}>
      {/* Notification Bell */}
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="relative p-2 text-gray-400 hover:text-gray-600 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 rounded-lg transition-colors"
      >
        <BellIcon className="w-6 h-6" />
        {showBadge && unreadCount > 0 && (
          <span className="absolute -top-1 -right-1 inline-flex items-center justify-center px-2 py-1 text-xs font-bold leading-none text-white transform translate-x-1/2 -translate-y-1/2 bg-red-600 rounded-full">
            {unreadCount > 99 ? '99+' : unreadCount}
          </span>
        )}
        {urgentCount > 0 && (
          <span className="absolute -top-0.5 -right-0.5 w-3 h-3 bg-orange-500 rounded-full animate-pulse"></span>
        )}
      </button>

      {/* Notification Panel */}
      <AnimatePresence>
        {isOpen && (
          <>
            {/* Backdrop */}
            <div
              className="fixed inset-0 z-40"
              onClick={() => setIsOpen(false)}
            />
            
            {/* Panel */}
            <motion.div
              initial={{ opacity: 0, scale: 0.95, y: -10 }}
              animate={{ opacity: 1, scale: 1, y: 0 }}
              exit={{ opacity: 0, scale: 0.95, y: -10 }}
              className="absolute right-0 mt-2 w-80 md:w-96 bg-white rounded-lg shadow-xl border border-gray-200 z-50 max-h-[80vh] flex flex-col"
            >
              {/* Header */}
              <div className="p-4 border-b border-gray-200">
                <div className="flex items-center justify-between mb-3">
                  <h3 className="text-lg font-semibold text-gray-900">Notifications</h3>
                  <button
                    onClick={() => setIsOpen(false)}
                    className="text-gray-400 hover:text-gray-600 transition-colors"
                  >
                    <XMarkIcon className="w-5 h-5" />
                  </button>
                </div>

                {/* Filter Tabs */}
                <div className="flex space-x-1 bg-gray-100 rounded-lg p-1">
                  {[
                    { key: 'all', label: 'All' },
                    { key: 'unread', label: `Unread (${unreadCount})` },
                    { key: 'urgent', label: `Urgent (${urgentCount})` }
                  ].map(tab => (
                    <button
                      key={tab.key}
                      onClick={() => setFilter(tab.key as any)}
                      className={`flex-1 px-3 py-1.5 text-sm font-medium rounded-md transition-colors ${
                        filter === tab.key
                          ? 'bg-white text-gray-900 shadow-sm'
                          : 'text-gray-600 hover:text-gray-900'
                      }`}
                    >
                      {tab.label}
                    </button>
                  ))}
                </div>
              </div>

              {/* Actions */}
              {unreadCount > 0 && (
                <div className="px-4 py-2 border-b border-gray-100">
                  <button
                    onClick={markAllAsRead}
                    className="text-sm text-blue-600 hover:text-blue-800 transition-colors"
                  >
                    Mark all as read
                  </button>
                </div>
              )}

              {/* Notifications List */}
              <div className="flex-1 overflow-y-auto">
                {filteredNotifications.length === 0 ? (
                  <div className="p-8 text-center">
                    <BellIcon className="w-12 h-12 text-gray-300 mx-auto mb-3" />
                    <p className="text-gray-500">
                      {filter === 'unread' ? 'No unread notifications' :
                       filter === 'urgent' ? 'No urgent notifications' :
                       'No notifications'}
                    </p>
                  </div>
                ) : (
                  <div className="divide-y divide-gray-100">
                    {filteredNotifications.map((notification) => (
                      <motion.div
                        key={notification.id}
                        initial={{ opacity: 0, x: 20 }}
                        animate={{ opacity: 1, x: 0 }}
                        className={`p-4 hover:bg-gray-50 transition-colors cursor-pointer border-l-4 ${
                          !notification.read ? 'bg-blue-50/50 border-l-blue-500' :
                          notification.urgent ? 'border-l-red-500' :
                          'border-l-transparent'
                        } ${getTypeColor(notification.type)}`}
                        onClick={() => {
                          if (!notification.read) {
                            markAsRead(notification.id)
                          }
                          if (notification.actionUrl) {
                            window.location.href = notification.actionUrl
                          }
                        }}
                      >
                        <div className="flex items-start space-x-3">
                          <div className="flex-shrink-0 mt-0.5">
                            {getNotificationIcon(notification.type)}
                          </div>
                          
                          <div className="flex-1 min-w-0">
                            <div className="flex items-center justify-between mb-1">
                              <p className={`text-sm font-medium ${
                                !notification.read ? 'text-gray-900' : 'text-gray-700'
                              }`}>
                                {notification.title}
                              </p>
                              {notification.urgent && (
                                <ExclamationTriangleIcon className="w-4 h-4 text-red-500 flex-shrink-0" />
                              )}
                            </div>
                            
                            <p className="text-sm text-gray-600 line-clamp-2 mb-2">
                              {notification.content}
                            </p>
                            
                            <div className="flex items-center justify-between">
                              <span className="text-xs text-gray-500">
                                {formatTimeAgo(notification.timestamp)}
                              </span>
                              
                              <div className="flex items-center space-x-2">
                                {!notification.read && (
                                  <button
                                    onClick={(e) => {
                                      e.stopPropagation()
                                      markAsRead(notification.id)
                                    }}
                                    className="text-xs text-blue-600 hover:text-blue-800 transition-colors"
                                  >
                                    <CheckIcon className="w-3 h-3" />
                                  </button>
                                )}
                                <button
                                  onClick={(e) => {
                                    e.stopPropagation()
                                    deleteNotification(notification.id)
                                  }}
                                  className="text-xs text-gray-400 hover:text-red-600 transition-colors"
                                >
                                  <XMarkIcon className="w-3 h-3" />
                                </button>
                              </div>
                            </div>
                          </div>
                        </div>
                      </motion.div>
                    ))}
                  </div>
                )}
              </div>

              {/* Footer */}
              <div className="p-3 border-t border-gray-200 bg-gray-50 rounded-b-lg">
                <a
                  href="/dashboard/notifications"
                  className="block text-center text-sm text-blue-600 hover:text-blue-800 transition-colors"
                >
                  View all notifications
                </a>
              </div>
            </motion.div>
          </>
        )}
      </AnimatePresence>
    </div>
  )
}