"use client";
import React from "react";
interface NuwairLogoProps {
  className?: string
  width?: number
  height?: number
  variant?: 'default' | 'dark' | 'light'
  format?: 'png' | 'svg'
}

// Professional, minimalist logo with modern design
export function NuwairLogoSVG({ 
  className = '', 
  width = 180, 
  height = 50,
  variant = 'default' 
}: NuwairLogoProps) {
  const isDark = variant === 'dark'
  
  return (
    <svg 
      width={width} 
      height={height} 
      viewBox="0 0 200 60" 
      className={className}
      xmlns="http://www.w3.org/2000/svg"
    >
      {/* Gradient Definitions */}
      <defs>
        {/* Premium brand gradient */}
        <linearGradient id="brandGradient" x1="0%" y1="0%" x2="100%" y2="100%">
          <stop offset="0%" stopColor="#0284c7" />
          <stop offset="50%" stopColor="#0ea5e9" />
          <stop offset="100%" stopColor="#06b6d4" />
        </linearGradient>
        
        {/* Accent gradient */}
        <linearGradient id="accentGradient" x1="0%" y1="0%" x2="100%" y2="0%">
          <stop offset="0%" stopColor="#3b82f6" />
          <stop offset="100%" stopColor="#6366f1" />
        </linearGradient>
      </defs>
      
      {/* Professional Icon - Stylized "N" with tech aesthetic */}
      <g transform="translate(0, 8)">
        {/* Main N shape with modern design */}
        <path 
          d="M 8,40 L 8,8 C 8,8 8,8 10,8 L 18,8 L 32,30 L 32,8 C 32,8 32,8 34,8 L 40,8 C 40,8 40,8 40,10 L 40,40 C 40,40 40,40 38,40 L 30,40 L 16,18 L 16,40 C 16,40 16,40 14,40 L 8,40 Z" 
          fill="url(#brandGradient)"
          stroke="none"
        />
        
        {/* Accent element - modern geometric shape */}
        <rect 
          x="42" 
          y="8" 
          width="3" 
          height="32" 
          rx="1.5"
          fill="url(#accentGradient)"
          opacity="0.8"
        />
        
        {/* Small decorative dots */}
        <circle cx="48" cy="10" r="2" fill="url(#accentGradient)" opacity="0.6" />
        <circle cx="48" cy="38" r="2" fill="url(#accentGradient)" opacity="0.6" />
      </g>
      
      {/* Company Name - Clean, professional typography */}
      <text 
        x="56" 
        y="28" 
        fontSize="22" 
        fontWeight="700" 
        fill={isDark ? '#ffffff' : '#0f172a'}
        fontFamily="Inter, -apple-system, BlinkMacSystemFont, Segoe UI, sans-serif"
        letterSpacing="-0.5"
      >
        NUWAIR
      </text>
      
      {/* Tagline - Professional subtitle */}
      <text 
        x="56" 
        y="44" 
        fontSize="9" 
        fontWeight="600"
        fill={isDark ? '#94a3b8' : '#64748b'}
        fontFamily="Inter, -apple-system, BlinkMacSystemFont, Segoe UI, sans-serif"
        letterSpacing="1.5"
      >
        SYSTEMS
      </text>
      
      {/* Subtle accent line */}
      <rect
        x="56"
        y="46"
        width="50"
        height="1.5"
        rx="0.75"
        fill="url(#brandGradient)"
        opacity="0.3"
      />
    </svg>
  )
}

export default function NuwairLogo({ 
  className = '', 
  width = 180, 
  height = 50, 
  variant = 'default',
}: NuwairLogoProps) {
  const [imgError, setImgError] = React.useState(false);
  const getImageStyle = () => {
    switch (variant) {
      case 'dark':
        return { filter: 'brightness(1.1) contrast(1.05)' }
      case 'light':
        return { filter: 'brightness(0.95) contrast(1.05)' }
      default:
        return {}
    }
  }
  const logoSrc = '/images/nuwair-logo.png'
  return (
    <div className={`flex items-center ${className}`} suppressHydrationWarning>
      {!imgError ? (
        <img 
          src={logoSrc}
          alt="NuwairSystems Logo" 
          width={width}
          height={height}
          className="object-contain max-w-full h-auto"
          style={{ 
            maxWidth: width, 
            maxHeight: height,
            ...getImageStyle()
          }}
          onError={() => setImgError(true)}
          suppressHydrationWarning
        />
      ) : (
        <NuwairLogoSVG width={width} height={height} className={className} variant={variant} />
      )}
    </div>
  )
}