'use client'

import { useState } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import {
  PhotoIcon,
  StarIcon,
  ChartBarIcon,
  GlobeAltIcon,
  DevicePhoneMobileIcon,
  ShoppingCartIcon,
  MagnifyingGlassIcon,
  FunnelIcon,
  CheckCircleIcon,
  ArrowRightIcon,
  XMarkIcon,
  PlayIcon,
} from '@heroicons/react/24/outline'
import { StarIcon as StarIconSolid } from '@heroicons/react/24/solid'

// Types
interface Project {
  id: string
  title: string
  category: 'web' | 'mobile' | 'ecommerce' | 'seo' | 'marketing' | 'branding'
  client: string
  year: string
  description: string
  challenge: string
  solution: string
  results: string[]
  technologies: string[]
  image: string
  images: string[]
  featured: boolean
  videoUrl?: string
}

interface Testimonial {
  id: string
  name: string
  position: string
  company: string
  avatar: string
  rating: number
  text: string
  project: string
  date: string
}

export default function PortfolioShowcase() {
  const [selectedCategory, setSelectedCategory] = useState<string>('all')
  const [searchQuery, setSearchQuery] = useState('')
  const [selectedProject, setSelectedProject] = useState<Project | null>(null)
  const [showFilters, setShowFilters] = useState(false)

  // Sample Projects Data
  const projects: Project[] = [
    {
      id: 'proj-1',
      title: 'TechMart E-Commerce Platform',
      category: 'ecommerce',
      client: 'TechMart Inc.',
      year: '2024',
      description: 'A comprehensive e-commerce platform with AI-powered recommendations and real-time inventory management',
      challenge: 'The client needed a scalable e-commerce solution that could handle high traffic during flash sales and provide personalized shopping experiences',
      solution: 'We built a modern Next.js platform with serverless architecture, Redis caching, and machine learning-based product recommendations',
      results: [
        '300% increase in conversion rate',
        '50% reduction in page load time',
        '24/7 uptime during peak seasons',
        '$2M+ in additional revenue',
      ],
      technologies: ['Next.js', 'TypeScript', 'Prisma', 'PostgreSQL', 'Redis', 'Stripe', 'AWS'],
      image: '/images/portfolio/techmart.jpg',
      images: ['/images/portfolio/techmart-1.jpg', '/images/portfolio/techmart-2.jpg'],
      featured: true,
      videoUrl: 'https://youtube.com/demo',
    },
    {
      id: 'proj-2',
      title: 'FinanceTracker Mobile App',
      category: 'mobile',
      client: 'FinTech Solutions',
      year: '2024',
      description: 'A cross-platform mobile app for personal finance management with AI-powered insights',
      challenge: 'Users needed an intuitive way to track expenses across multiple accounts with real-time synchronization',
      solution: 'Built a React Native app with offline-first architecture, biometric authentication, and intelligent categorization',
      results: [
        '100K+ downloads in 3 months',
        '4.8/5 app store rating',
        '85% daily active users',
        'Featured on App Store',
      ],
      technologies: ['React Native', 'TypeScript', 'Firebase', 'Node.js', 'MongoDB', 'Plaid API'],
      image: '/images/portfolio/financetracker.jpg',
      images: ['/images/portfolio/financetracker-1.jpg'],
      featured: true,
    },
    {
      id: 'proj-3',
      title: 'GlobalMart SEO Optimization',
      category: 'seo',
      client: 'GlobalMart Corp',
      year: '2023',
      description: 'Complete SEO overhaul resulting in 500% increase in organic traffic',
      challenge: 'Client was losing market share to competitors due to poor search visibility',
      solution: 'Comprehensive SEO strategy including technical optimization, content marketing, and link building',
      results: [
        '500% increase in organic traffic',
        '#1 ranking for 50+ keywords',
        '200% increase in qualified leads',
        '150% ROI in 6 months',
      ],
      technologies: ['SEO Tools', 'Google Analytics', 'Schema Markup', 'Content Strategy', 'Technical SEO'],
      image: '/images/portfolio/globalmart.jpg',
      images: [],
      featured: false,
    },
    {
      id: 'proj-4',
      title: 'HealthCare Portal',
      category: 'web',
      client: 'MedCare Solutions',
      year: '2024',
      description: 'HIPAA-compliant patient portal with telemedicine capabilities',
      challenge: 'Healthcare provider needed secure patient communication and appointment management system',
      solution: 'Built enterprise-grade portal with end-to-end encryption, video conferencing, and EHR integration',
      results: [
        '90% reduction in phone calls',
        '50,000+ patients onboarded',
        'Zero security incidents',
        '95% patient satisfaction',
      ],
      technologies: ['React', 'Node.js', 'PostgreSQL', 'WebRTC', 'AWS', 'HIPAA Compliance'],
      image: '/images/portfolio/healthcare.jpg',
      images: [],
      featured: true,
    },
    {
      id: 'proj-5',
      title: 'BrandHub Digital Marketing',
      category: 'marketing',
      client: 'BrandHub Agency',
      year: '2023',
      description: 'Multi-channel digital marketing campaign driving 400% ROI',
      challenge: 'Client needed to increase brand awareness and lead generation across digital channels',
      solution: 'Executed integrated campaign with social media, PPC, email marketing, and content strategy',
      results: [
        '400% return on ad spend',
        '2M+ social media impressions',
        '15K+ qualified leads',
        '250% brand awareness increase',
      ],
      technologies: ['Google Ads', 'Facebook Ads', 'Email Marketing', 'Analytics', 'CRM Integration'],
      image: '/images/portfolio/brandhub.jpg',
      images: [],
      featured: false,
    },
    {
      id: 'proj-6',
      title: 'LuxuryBrand Identity',
      category: 'branding',
      client: 'LuxuryBrand Co',
      year: '2024',
      description: 'Complete brand identity redesign including logo, website, and marketing materials',
      challenge: 'Outdated brand failing to resonate with modern luxury market',
      solution: 'Comprehensive rebrand with modern design language, premium positioning, and cohesive visual identity',
      results: [
        '300% increase in brand recall',
        'Award-winning design',
        '80% increase in premium sales',
        'Featured in major publications',
      ],
      technologies: ['Adobe Suite', 'Figma', 'Brand Strategy', 'UI/UX Design', 'Print Design'],
      image: '/images/portfolio/luxury.jpg',
      images: [],
      featured: false,
    },
  ]

  // Sample Testimonials Data
  const testimonials: Testimonial[] = [
    {
      id: 'test-1',
      name: 'Sarah Johnson',
      position: 'CEO',
      company: 'TechMart Inc.',
      avatar: '/images/avatars/sarah.jpg',
      rating: 5,
      text: 'NuwairSystems transformed our e-commerce platform beyond our expectations. The AI-powered recommendations increased our sales by 300% in just 3 months. Their team is professional, responsive, and truly understands modern web technologies.',
      project: 'TechMart E-Commerce Platform',
      date: 'February 2024',
    },
    {
      id: 'test-2',
      name: 'Michael Chen',
      position: 'Product Manager',
      company: 'FinTech Solutions',
      avatar: '/images/avatars/michael.jpg',
      rating: 5,
      text: 'The FinanceTracker app exceeded all our requirements. The offline-first architecture and beautiful UI made it a hit with our users. We reached 100K downloads faster than we ever imagined possible.',
      project: 'FinanceTracker Mobile App',
      date: 'January 2024',
    },
    {
      id: 'test-3',
      name: 'Emily Rodriguez',
      position: 'Marketing Director',
      company: 'GlobalMart Corp',
      avatar: '/images/avatars/emily.jpg',
      rating: 5,
      text: 'Their SEO expertise is unmatched. We went from page 5 to ranking #1 for our most important keywords. Organic traffic increased by 500% and our lead quality improved dramatically. Best investment we\'ve made.',
      project: 'GlobalMart SEO Optimization',
      date: 'December 2023',
    },
    {
      id: 'test-4',
      name: 'Dr. James Wilson',
      position: 'Chief Medical Officer',
      company: 'MedCare Solutions',
      avatar: '/images/avatars/james.jpg',
      rating: 5,
      text: 'Security and compliance were our top priorities, and NuwairSystems delivered perfectly. The patient portal is intuitive, secure, and has dramatically improved our patient engagement. Highly recommended!',
      project: 'HealthCare Portal',
      date: 'March 2024',
    },
    {
      id: 'test-5',
      name: 'Lisa Anderson',
      position: 'Founder',
      company: 'BrandHub Agency',
      avatar: '/images/avatars/lisa.jpg',
      rating: 5,
      text: 'The digital marketing campaign delivered exceptional results. 400% ROI and thousands of qualified leads. Their data-driven approach and creative execution made all the difference.',
      project: 'BrandHub Digital Marketing',
      date: 'November 2023',
    },
  ]

  const categories = [
    { id: 'all', label: 'All Projects', icon: PhotoIcon },
    { id: 'web', label: 'Web Development', icon: GlobeAltIcon },
    { id: 'mobile', label: 'Mobile Apps', icon: DevicePhoneMobileIcon },
    { id: 'ecommerce', label: 'E-Commerce', icon: ShoppingCartIcon },
    { id: 'seo', label: 'SEO', icon: MagnifyingGlassIcon },
    { id: 'marketing', label: 'Marketing', icon: ChartBarIcon },
    { id: 'branding', label: 'Branding', icon: StarIcon },
  ]

  const filteredProjects = projects.filter((project) => {
    const matchesCategory = selectedCategory === 'all' || project.category === selectedCategory
    const matchesSearch =
      project.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
      project.description.toLowerCase().includes(searchQuery.toLowerCase()) ||
      project.client.toLowerCase().includes(searchQuery.toLowerCase())
    return matchesCategory && matchesSearch
  })

  const getCategoryIcon = (categoryId: string) => {
    const category = categories.find((c) => c.id === categoryId)
    return category?.icon || PhotoIcon
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-white to-blue-50 py-16">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Hero Section */}
        <motion.div
          initial={{ opacity: 0, y: -30 }}
          animate={{ opacity: 1, y: 0 }}
          className="text-center mb-16"
        >
          <h1 className="text-5xl md:text-7xl font-bold text-gray-900 mb-6">
            Our <span className="text-gradient-modern">Portfolio</span>
          </h1>
          <p className="text-xl md:text-2xl text-gray-600 max-w-3xl mx-auto">
            Explore our success stories and see how we&apos;ve helped businesses achieve their digital goals
          </p>
        </motion.div>

        {/* Search and Filter */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          className="mb-12"
        >
          <div className="bg-white rounded-2xl shadow-lg p-6">
            <div className="flex flex-col md:flex-row gap-4">
              <div className="flex-1 relative">
                <MagnifyingGlassIcon className="absolute left-4 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
                <input
                  type="text"
                  placeholder="Search projects..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="w-full pl-12 pr-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all"
                />
              </div>
              <button
                onClick={() => setShowFilters(!showFilters)}
                className="flex items-center justify-center gap-2 px-6 py-3 bg-blue-600 text-white rounded-xl hover:bg-blue-700 transition-colors"
              >
                <FunnelIcon className="w-5 h-5" />
                Filters
              </button>
            </div>

            {/* Category Filter */}
            <AnimatePresence>
              {showFilters && (
                <motion.div
                  initial={{ opacity: 0, height: 0 }}
                  animate={{ opacity: 1, height: 'auto' }}
                  exit={{ opacity: 0, height: 0 }}
                  className="mt-6 pt-6 border-t border-gray-200"
                >
                  <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-7 gap-3">
                    {categories.map((category) => {
                      const Icon = category.icon
                      return (
                        <button
                          key={category.id}
                          onClick={() => setSelectedCategory(category.id)}
                          className={`flex flex-col items-center gap-2 p-4 rounded-xl border-2 transition-all ${
                            selectedCategory === category.id
                              ? 'border-blue-600 bg-blue-50 text-blue-600'
                              : 'border-gray-200 hover:border-gray-300 text-gray-600'
                          }`}
                        >
                          <Icon className="w-6 h-6" />
                          <span className="text-xs font-medium text-center">{category.label}</span>
                        </button>
                      )
                    })}
                  </div>
                </motion.div>
              )}
            </AnimatePresence>
          </div>
        </motion.div>

        {/* Featured Projects */}
        {selectedCategory === 'all' && (
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="mb-16"
          >
            <h2 className="text-3xl font-bold text-gray-900 mb-8">Featured Projects</h2>
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
              {projects
                .filter((p) => p.featured)
                .map((project, index) => {
                  const CategoryIcon = getCategoryIcon(project.category)
                  return (
                    <motion.div
                      key={project.id}
                      initial={{ opacity: 0, y: 30 }}
                      animate={{ opacity: 1, y: 0 }}
                      transition={{ delay: index * 0.1 }}
                      onClick={() => setSelectedProject(project)}
                      className="group relative bg-white rounded-2xl shadow-lg overflow-hidden cursor-pointer hover:shadow-2xl transition-all duration-300"
                    >
                      <div className="aspect-video bg-gradient-to-br from-blue-500 to-purple-600 relative overflow-hidden">
                        <div className="absolute inset-0 flex items-center justify-center">
                          <PhotoIcon className="w-24 h-24 text-white/30" />
                        </div>
                        <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent" />
                        <div className="absolute top-4 right-4">
                          <span className="px-3 py-1 bg-white/90 backdrop-blur-sm text-blue-600 rounded-full text-xs font-semibold">
                            Featured
                          </span>
                        </div>
                        <div className="absolute bottom-4 left-4 right-4">
                          <h3 className="text-2xl font-bold text-white mb-2">{project.title}</h3>
                          <p className="text-white/90 text-sm">{project.client}</p>
                        </div>
                      </div>
                      <div className="p-6">
                        <div className="flex items-center gap-2 mb-4">
                          <CategoryIcon className="w-5 h-5 text-blue-600" />
                          <span className="text-sm font-medium text-gray-600 capitalize">{project.category}</span>
                          <span className="text-gray-400">•</span>
                          <span className="text-sm text-gray-500">{project.year}</span>
                        </div>
                        <p className="text-gray-600 mb-4 line-clamp-2">{project.description}</p>
                        <div className="flex items-center justify-between">
                          <div className="flex flex-wrap gap-2">
                            {project.technologies.slice(0, 3).map((tech) => (
                              <span key={tech} className="px-2 py-1 bg-blue-50 text-blue-600 rounded-md text-xs font-medium">
                                {tech}
                              </span>
                            ))}
                            {project.technologies.length > 3 && (
                              <span className="px-2 py-1 bg-gray-100 text-gray-600 rounded-md text-xs font-medium">
                                +{project.technologies.length - 3} more
                              </span>
                            )}
                          </div>
                          <ArrowRightIcon className="w-5 h-5 text-blue-600 group-hover:translate-x-1 transition-transform" />
                        </div>
                      </div>
                    </motion.div>
                  )
                })}
            </div>
          </motion.div>
        )}

        {/* All Projects Grid */}
        <div className="mb-16">
          <div className="flex items-center justify-between mb-8">
            <h2 className="text-3xl font-bold text-gray-900">
              {selectedCategory === 'all' ? 'All Projects' : categories.find((c) => c.id === selectedCategory)?.label}
            </h2>
            <p className="text-gray-600">{filteredProjects.length} projects found</p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {filteredProjects.map((project, index) => {
              const CategoryIcon = getCategoryIcon(project.category)
              return (
                <motion.div
                  key={project.id}
                  initial={{ opacity: 0, scale: 0.95 }}
                  animate={{ opacity: 1, scale: 1 }}
                  transition={{ delay: index * 0.05 }}
                  onClick={() => setSelectedProject(project)}
                  className="group bg-white rounded-2xl shadow-lg overflow-hidden cursor-pointer hover:shadow-2xl transition-all duration-300"
                >
                  <div className="aspect-video bg-gradient-to-br from-blue-400 to-purple-500 relative overflow-hidden">
                    <div className="absolute inset-0 flex items-center justify-center">
                      <PhotoIcon className="w-16 h-16 text-white/30" />
                    </div>
                    {project.featured && (
                      <div className="absolute top-3 right-3">
                        <StarIconSolid className="w-6 h-6 text-yellow-400" />
                      </div>
                    )}
                  </div>
                  <div className="p-6">
                    <div className="flex items-center gap-2 mb-3">
                      <CategoryIcon className="w-4 h-4 text-blue-600" />
                      <span className="text-xs font-medium text-gray-600 capitalize">{project.category}</span>
                    </div>
                    <h3 className="text-lg font-bold text-gray-900 mb-2 group-hover:text-blue-600 transition-colors">
                      {project.title}
                    </h3>
                    <p className="text-sm text-gray-600 mb-4 line-clamp-2">{project.description}</p>
                    <div className="flex items-center justify-between text-xs text-gray-500">
                      <span>{project.client}</span>
                      <span>{project.year}</span>
                    </div>
                  </div>
                </motion.div>
              )
            })}
          </div>
        </div>

        {/* Client Testimonials */}
        <motion.div
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          className="mb-16"
        >
          <h2 className="text-3xl font-bold text-gray-900 mb-8 text-center">What Our Clients Say</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {testimonials.map((testimonial, index) => (
              <motion.div
                key={testimonial.id}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
                className="bg-white rounded-2xl shadow-lg p-6 hover:shadow-xl transition-shadow"
              >
                <div className="flex items-center gap-4 mb-4">
                  <div className="w-12 h-12 bg-gradient-to-br from-blue-500 to-purple-500 rounded-full flex items-center justify-center text-white font-bold">
                    {testimonial.name.charAt(0)}
                  </div>
                  <div>
                    <h4 className="font-semibold text-gray-900">{testimonial.name}</h4>
                    <p className="text-sm text-gray-600">
                      {testimonial.position}, {testimonial.company}
                    </p>
                  </div>
                </div>

                <div className="flex items-center gap-1 mb-4">
                  {[...Array(testimonial.rating)].map((_, i) => (
                    <StarIconSolid key={i} className="w-5 h-5 text-yellow-400" />
                  ))}
                </div>

                <p className="text-gray-600 mb-4 italic">&ldquo;{testimonial.text}&rdquo;</p>

                <div className="flex items-center justify-between text-xs text-gray-500 pt-4 border-t border-gray-100">
                  <span>{testimonial.project}</span>
                  <span>{testimonial.date}</span>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.div>

        {/* CTA Section */}
        <motion.div
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          className="bg-gradient-to-r from-blue-600 to-purple-600 rounded-3xl p-12 text-center text-white"
        >
          <h2 className="text-4xl font-bold mb-4">Ready to Start Your Success Story?</h2>
          <p className="text-xl text-blue-100 mb-8 max-w-2xl mx-auto">
            Let&apos;s discuss how we can help you achieve remarkable results like our featured clients
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <button className="px-8 py-4 bg-white text-blue-600 rounded-xl font-semibold hover:bg-blue-50 transition-colors">
              Get Free Consultation
            </button>
            <button className="px-8 py-4 border-2 border-white text-white rounded-xl font-semibold hover:bg-white/10 transition-colors">
              View All Services
            </button>
          </div>
        </motion.div>
      </div>

      {/* Project Detail Modal */}
      <AnimatePresence>
        {selectedProject && (
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            exit={{ opacity: 0 }}
            onClick={() => setSelectedProject(null)}
            className="fixed inset-0 bg-black/60 backdrop-blur-sm z-50 flex items-center justify-center p-4 overflow-y-auto"
          >
            <motion.div
              initial={{ scale: 0.9, opacity: 0 }}
              animate={{ scale: 1, opacity: 1 }}
              exit={{ scale: 0.9, opacity: 0 }}
              onClick={(e) => e.stopPropagation()}
              className="bg-white rounded-3xl shadow-2xl max-w-4xl w-full my-8 overflow-hidden"
            >
              <div className="relative aspect-video bg-gradient-to-br from-blue-500 to-purple-600">
                <div className="absolute inset-0 flex items-center justify-center">
                  <PhotoIcon className="w-32 h-32 text-white/30" />
                </div>
                {selectedProject.videoUrl && (
                  <div className="absolute inset-0 flex items-center justify-center">
                    <button className="w-16 h-16 bg-white rounded-full flex items-center justify-center hover:scale-110 transition-transform">
                      <PlayIcon className="w-8 h-8 text-blue-600 ml-1" />
                    </button>
                  </div>
                )}
                <button
                  onClick={() => setSelectedProject(null)}
                  className="absolute top-4 right-4 w-10 h-10 bg-white/90 backdrop-blur-sm rounded-full flex items-center justify-center hover:bg-white transition-colors"
                >
                  <XMarkIcon className="w-6 h-6 text-gray-900" />
                </button>
              </div>

              <div className="p-8">
                <div className="flex items-start justify-between mb-6">
                  <div>
                    <h2 className="text-3xl font-bold text-gray-900 mb-2">{selectedProject.title}</h2>
                    <p className="text-lg text-gray-600">
                      {selectedProject.client} • {selectedProject.year}
                    </p>
                  </div>
                  <span className="px-4 py-2 bg-blue-100 text-blue-600 rounded-full text-sm font-semibold capitalize">
                    {selectedProject.category}
                  </span>
                </div>

                <p className="text-gray-600 mb-8">{selectedProject.description}</p>

                <div className="space-y-6 mb-8">
                  <div>
                    <h3 className="text-xl font-bold text-gray-900 mb-3">The Challenge</h3>
                    <p className="text-gray-600">{selectedProject.challenge}</p>
                  </div>

                  <div>
                    <h3 className="text-xl font-bold text-gray-900 mb-3">Our Solution</h3>
                    <p className="text-gray-600">{selectedProject.solution}</p>
                  </div>

                  <div>
                    <h3 className="text-xl font-bold text-gray-900 mb-3">Results</h3>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      {selectedProject.results.map((result, index) => (
                        <div key={index} className="flex items-start gap-3 p-4 bg-green-50 rounded-xl">
                          <CheckCircleIcon className="w-6 h-6 text-green-600 flex-shrink-0 mt-0.5" />
                          <span className="text-gray-700 font-medium">{result}</span>
                        </div>
                      ))}
                    </div>
                  </div>

                  <div>
                    <h3 className="text-xl font-bold text-gray-900 mb-3">Technologies Used</h3>
                    <div className="flex flex-wrap gap-2">
                      {selectedProject.technologies.map((tech) => (
                        <span key={tech} className="px-4 py-2 bg-blue-50 text-blue-600 rounded-lg font-medium">
                          {tech}
                        </span>
                      ))}
                    </div>
                  </div>
                </div>

                <div className="flex gap-4">
                  <button
                    onClick={() => setSelectedProject(null)}
                    className="flex-1 px-6 py-3 bg-blue-600 text-white rounded-xl font-semibold hover:bg-blue-700 transition-colors"
                  >
                    Start Similar Project
                  </button>
                  <button
                    onClick={() => setSelectedProject(null)}
                    className="px-6 py-3 border-2 border-gray-300 text-gray-700 rounded-xl font-semibold hover:bg-gray-50 transition-colors"
                  >
                    Close
                  </button>
                </div>
              </div>
            </motion.div>
          </motion.div>
        )}
      </AnimatePresence>
    </div>
  )
}
