'use client'

import { useState, useEffect } from 'react'
import { useSession } from 'next-auth/react'
import { motion, AnimatePresence } from 'framer-motion'
import {
  ChatBubbleLeftRightIcon,
  VideoCameraIcon,
  PhoneIcon,
  DocumentIcon,
  PhotoIcon,
  ShareIcon,
  ClockIcon,
  UserIcon,
  CheckIcon,
  ExclamationTriangleIcon,
  PaperAirplaneIcon,
  PaperClipIcon
} from '@heroicons/react/24/outline'

interface ProjectCommunication {
  id: string
  projectId: string
  projectName: string
  messages: ProjectMessage[]
  participants: ProjectParticipant[]
  lastActivity: Date
  status: 'active' | 'archived' | 'completed'
}

interface ProjectMessage {
  id: string
  content: string
  author: string
  authorType: 'client' | 'team' | 'admin'
  timestamp: Date
  type: 'text' | 'file' | 'system' | 'milestone'
  attachments?: FileAttachment[]
  metadata?: {
    milestone?: string
    priority?: 'low' | 'medium' | 'high'
    category?: string
  }
}

interface ProjectParticipant {
  id: string
  name: string
  role: string
  avatar?: string
  status: 'online' | 'away' | 'offline'
  expertise: string[]
}

interface FileAttachment {
  id: string
  name: string
  size: string
  type: string
  url: string
  uploadedBy: string
  uploadedAt: Date
}

const MOCK_PROJECTS: ProjectCommunication[] = [
  {
    id: 'proj-1',
    projectId: 'WEB-001',
    projectName: 'E-commerce Website Development',
    status: 'active',
    lastActivity: new Date(),
    participants: [
      {
        id: 'sarah-1',
        name: 'Sarah Ahmed',
        role: 'Lead Developer',
        avatar: '/team/sarah.jpg',
        status: 'online',
        expertise: ['React', 'Node.js', 'E-commerce']
      },
      {
        id: 'ali-1',
        name: 'Ali Hassan',
        role: 'UI/UX Designer',
        avatar: '/team/ali.jpg', 
        status: 'away',
        expertise: ['Figma', 'Design Systems', 'User Experience']
      }
    ],
    messages: [
      {
        id: 'msg-1',
        content: 'Project kickoff meeting scheduled for Monday at 10 AM. We\'ll discuss the initial wireframes and project timeline.',
        author: 'Sarah Ahmed',
        authorType: 'team',
        timestamp: new Date('2025-10-31T09:00:00'),
        type: 'system',
        metadata: { milestone: 'Project Kickoff', priority: 'high' }
      },
      {
        id: 'msg-2',
        content: 'I\'ve uploaded the initial design mockups for review. Please let me know your thoughts!',
        author: 'Ali Hassan',
        authorType: 'team',
        timestamp: new Date('2025-10-31T14:30:00'),
        type: 'file',
        attachments: [
          {
            id: 'file-1',
            name: 'Homepage-Design-v1.figma',
            size: '2.4 MB',
            type: 'figma',
            url: '/files/homepage-design-v1.figma',
            uploadedBy: 'Ali Hassan',
            uploadedAt: new Date('2025-10-31T14:30:00')
          }
        ]
      }
    ]
  }
]

export default function ProjectCommunicationHub() {
  const { data: session } = useSession()
  const [selectedProject, setSelectedProject] = useState<ProjectCommunication | null>(null)
  const [projects, setProjects] = useState<ProjectCommunication[]>(MOCK_PROJECTS)
  const [newMessage, setNewMessage] = useState('')
  const [showVideoCall, setShowVideoCall] = useState(false)
  const [isTyping, setIsTyping] = useState(false)

  useEffect(() => {
    if (projects.length > 0) {
      setSelectedProject(projects[0])
    }
  }, [projects])

  const handleSendMessage = () => {
    if (!newMessage.trim() || !selectedProject) return

    const message: ProjectMessage = {
      id: `msg-${Date.now()}`,
      content: newMessage,
      author: session?.user?.name || 'Client',
      authorType: 'client',
      timestamp: new Date(),
      type: 'text'
    }

    const updatedProject = {
      ...selectedProject,
      messages: [...selectedProject.messages, message],
      lastActivity: new Date()
    }

    setProjects(prev => prev.map(p => p.id === selectedProject.id ? updatedProject : p))
    setSelectedProject(updatedProject)
    setNewMessage('')

    // Simulate team response
    setTimeout(() => {
      const teamResponse: ProjectMessage = {
        id: `response-${Date.now()}`,
        content: "Thanks for your message! We're reviewing your feedback and will get back to you shortly with updates.",
        author: 'Sarah Ahmed',
        authorType: 'team',
        timestamp: new Date(),
        type: 'text'
      }

      const responseProject = {
        ...updatedProject,
        messages: [...updatedProject.messages, teamResponse],
        lastActivity: new Date()
      }

      setProjects(prev => prev.map(p => p.id === selectedProject.id ? responseProject : p))
      setSelectedProject(responseProject)
    }, 2000)
  }

  const getStatusColor = (status: ProjectCommunication['status']) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800'
      case 'completed': return 'bg-blue-100 text-blue-800'
      case 'archived': return 'bg-gray-100 text-gray-800'
      default: return 'bg-gray-100 text-gray-800'
    }
  }

  const getParticipantStatusColor = (status: ProjectParticipant['status']) => {
    switch (status) {
      case 'online': return 'bg-green-400'
      case 'away': return 'bg-yellow-400'
      case 'offline': return 'bg-gray-400'
      default: return 'bg-gray-400'
    }
  }

  const scheduleVideoCall = () => {
    // Integration with video call platform
    window.open('https://meet.google.com/new', '_blank')
  }

  if (!session) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-50">
        <div className="text-center">
          <ChatBubbleLeftRightIcon className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <h2 className="text-2xl font-bold text-gray-900 mb-2">Project Communication</h2>
          <p className="text-gray-600 mb-6">Please sign in to access your project communications.</p>
          <a
            href="/auth/signin"
            className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            Sign In
          </a>
        </div>
      </div>
    )
  }

  return (
    <div className="h-screen flex bg-gray-50">
      {/* Projects Sidebar */}
      <div className="w-80 bg-white border-r border-gray-200 flex flex-col">
        <div className="p-4 border-b border-gray-200">
          <h1 className="text-xl font-semibold text-gray-900">Project Communications</h1>
          <p className="text-sm text-gray-600 mt-1">Stay connected with your project teams</p>
        </div>

        <div className="flex-1 overflow-y-auto">
          {projects.map((project) => (
            <motion.div
              key={project.id}
              onClick={() => setSelectedProject(project)}
              className={`p-4 border-b border-gray-100 cursor-pointer transition-colors ${
                selectedProject?.id === project.id ? 'bg-blue-50 border-blue-200' : 'hover:bg-gray-50'
              }`}
              whileHover={{ x: 4 }}
            >
              <div className="flex items-start justify-between mb-2">
                <h3 className="font-medium text-gray-900 text-sm">{project.projectName}</h3>
                <span className={`px-2 py-1 rounded-full text-xs font-medium ${getStatusColor(project.status)}`}>
                  {project.status}
                </span>
              </div>
              <p className="text-xs text-gray-500 mb-2">ID: {project.projectId}</p>
              
              {/* Last Message Preview */}
              {project.messages.length > 0 && (
                <div className="mb-2">
                  <p className="text-sm text-gray-600 line-clamp-2">
                    {project.messages[project.messages.length - 1].content}
                  </p>
                </div>
              )}

              {/* Participants */}
              <div className="flex items-center justify-between">
                <div className="flex -space-x-2">
                  {project.participants.slice(0, 3).map((participant) => (
                    <div key={participant.id} className="relative">
                      <img 
                        src={participant.avatar || '/placeholder-avatar.png'} 
                        alt={participant.name}
                        className="w-6 h-6 rounded-full border-2 border-white"
                        onError={(e) => {
                          e.currentTarget.src = '/placeholder-avatar.png'
                        }}
                      />
                      <div className={`absolute -bottom-1 -right-1 w-2 h-2 rounded-full border border-white ${
                        getParticipantStatusColor(participant.status)
                      }`}></div>
                    </div>
                  ))}
                  {project.participants.length > 3 && (
                    <div className="w-6 h-6 rounded-full bg-gray-200 border-2 border-white flex items-center justify-center">
                      <span className="text-xs text-gray-600">+{project.participants.length - 3}</span>
                    </div>
                  )}
                </div>
                <span className="text-xs text-gray-500">
                  {project.lastActivity.toLocaleDateString()}
                </span>
              </div>
            </motion.div>
          ))}
        </div>
      </div>

      {/* Chat Area */}
      <div className="flex-1 flex flex-col">
        {selectedProject ? (
          <>
            {/* Chat Header */}
            <div className="bg-white border-b border-gray-200 p-4">
              <div className="flex items-center justify-between">
                <div>
                  <h2 className="text-lg font-semibold text-gray-900">{selectedProject.projectName}</h2>
                  <p className="text-sm text-gray-600">Project ID: {selectedProject.projectId}</p>
                </div>
                <div className="flex items-center space-x-3">
                  {/* Participants */}
                  <div className="flex items-center space-x-2">
                    {selectedProject.participants.map((participant) => (
                      <div key={participant.id} className="flex items-center space-x-2">
                        <div className="relative">
                          <img 
                            src={participant.avatar || '/placeholder-avatar.png'} 
                            alt={participant.name}
                            className="w-8 h-8 rounded-full"
                            onError={(e) => {
                              e.currentTarget.src = '/placeholder-avatar.png'
                            }}
                          />
                          <div className={`absolute -bottom-1 -right-1 w-3 h-3 rounded-full border-2 border-white ${
                            getParticipantStatusColor(participant.status)
                          }`}></div>
                        </div>
                        <div className="hidden md:block">
                          <p className="text-sm font-medium text-gray-900">{participant.name}</p>
                          <p className="text-xs text-gray-500">{participant.role}</p>
                        </div>
                      </div>
                    ))}
                  </div>

                  {/* Action Buttons */}
                  <div className="flex items-center space-x-2">
                    <button
                      onClick={scheduleVideoCall}
                      className="p-2 text-gray-400 hover:text-gray-600 hover:bg-gray-100 rounded-lg transition-colors"
                      title="Start Video Call"
                    >
                      <VideoCameraIcon className="w-5 h-5" />
                    </button>
                    <button
                      className="p-2 text-gray-400 hover:text-gray-600 hover:bg-gray-100 rounded-lg transition-colors"
                      title="Voice Call"
                    >
                      <PhoneIcon className="w-5 h-5" />
                    </button>
                    <button
                      className="p-2 text-gray-400 hover:text-gray-600 hover:bg-gray-100 rounded-lg transition-colors"
                      title="Share Files"
                    >
                      <ShareIcon className="w-5 h-5" />
                    </button>
                  </div>
                </div>
              </div>
            </div>

            {/* Messages */}
            <div className="flex-1 overflow-y-auto p-4 space-y-4 bg-gray-50">
              {selectedProject.messages.map((message) => (
                <motion.div
                  key={message.id}
                  initial={{ opacity: 0, y: 10 }}
                  animate={{ opacity: 1, y: 0 }}
                  className={`flex ${message.authorType === 'client' ? 'justify-end' : 'justify-start'}`}
                >
                  <div className={`max-w-[70%] ${
                    message.authorType === 'client' 
                      ? 'bg-blue-600 text-white' 
                      : message.type === 'system'
                      ? 'bg-yellow-100 text-yellow-800 border border-yellow-200'
                      : 'bg-white text-gray-900 border border-gray-200'
                  } rounded-lg p-4 shadow-sm`}>
                    
                    {/* Message Header */}
                    {message.authorType !== 'client' && (
                      <div className="flex items-center space-x-2 mb-2">
                        <UserIcon className="w-4 h-4" />
                        <span className="text-sm font-medium">{message.author}</span>
                        {message.metadata?.priority && (
                          <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                            message.metadata.priority === 'high' ? 'bg-red-100 text-red-800' :
                            message.metadata.priority === 'medium' ? 'bg-yellow-100 text-yellow-800' :
                            'bg-gray-100 text-gray-800'
                          }`}>
                            {message.metadata.priority}
                          </span>
                        )}
                      </div>
                    )}

                    {/* System/Milestone Messages */}
                    {message.type === 'system' && message.metadata?.milestone && (
                      <div className="flex items-center space-x-2 mb-2">
                        <ExclamationTriangleIcon className="w-4 h-4" />
                        <span className="text-sm font-semibold">Milestone: {message.metadata.milestone}</span>
                      </div>
                    )}

                    {/* Message Content */}
                    <p className="text-sm">{message.content}</p>

                    {/* File Attachments */}
                    {message.attachments && message.attachments.length > 0 && (
                      <div className="mt-3 space-y-2">
                        {message.attachments.map((file) => (
                          <div key={file.id} className="flex items-center space-x-3 p-2 bg-black/5 rounded-lg">
                            <div className="p-2 bg-gray-100 rounded-lg">
                              {file.type.includes('image') ? (
                                <PhotoIcon className="w-4 h-4 text-gray-600" />
                              ) : (
                                <DocumentIcon className="w-4 h-4 text-gray-600" />
                              )}
                            </div>
                            <div className="flex-1">
                              <p className="text-sm font-medium">{file.name}</p>
                              <p className="text-xs text-gray-500">{file.size} • {file.uploadedBy}</p>
                            </div>
                            <button className="text-blue-600 hover:text-blue-800 text-sm">
                              Download
                            </button>
                          </div>
                        ))}
                      </div>
                    )}

                    {/* Message Footer */}
                    <div className="flex items-center justify-between mt-3">
                      <span className="text-xs opacity-75">
                        {message.timestamp.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                      </span>
                      {message.authorType === 'client' && (
                        <CheckIcon className="w-3 h-3 opacity-75" />
                      )}
                    </div>
                  </div>
                </motion.div>
              ))}

              {/* Typing Indicator */}
              {isTyping && (
                <div className="flex justify-start">
                  <div className="bg-white border border-gray-200 rounded-lg p-3 shadow-sm">
                    <div className="flex items-center space-x-2">
                      <div className="flex space-x-1">
                        <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce"></div>
                        <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce" style={{ animationDelay: '0.1s' }}></div>
                        <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce" style={{ animationDelay: '0.2s' }}></div>
                      </div>
                      <span className="text-xs text-gray-500">Team is typing...</span>
                    </div>
                  </div>
                </div>
              )}
            </div>

            {/* Message Input */}
            <div className="bg-white border-t border-gray-200 p-4">
              <div className="flex items-center space-x-3">
                <button className="p-2 text-gray-400 hover:text-gray-600 transition-colors">
                  <PaperClipIcon className="w-5 h-5" />
                </button>
                <input
                  type="text"
                  value={newMessage}
                  onChange={(e) => setNewMessage(e.target.value)}
                  onKeyPress={(e) => e.key === 'Enter' && handleSendMessage()}
                  placeholder="Type your message..."
                  className="flex-1 p-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500"
                />
                <button
                  onClick={handleSendMessage}
                  disabled={!newMessage.trim()}
                  className="p-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                >
                  <PaperAirplaneIcon className="w-5 h-5" />
                </button>
              </div>

              {/* Quick Actions */}
              <div className="flex items-center space-x-4 mt-3 text-sm">
                <button
                  onClick={scheduleVideoCall}
                  className="flex items-center space-x-1 text-gray-600 hover:text-blue-600 transition-colors"
                >
                  <VideoCameraIcon className="w-4 h-4" />
                  <span>Schedule Call</span>
                </button>
                <button className="flex items-center space-x-1 text-gray-600 hover:text-blue-600 transition-colors">
                  <ClockIcon className="w-4 h-4" />
                  <span>Set Reminder</span>
                </button>
                <button className="flex items-center space-x-1 text-gray-600 hover:text-blue-600 transition-colors">
                  <ShareIcon className="w-4 h-4" />
                  <span>Share Files</span>
                </button>
              </div>
            </div>
          </>
        ) : (
          <div className="flex-1 flex items-center justify-center bg-gray-50">
            <div className="text-center">
              <ChatBubbleLeftRightIcon className="w-16 h-16 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">Select a Project</h3>
              <p className="text-gray-600">Choose a project to start communicating with your team</p>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}