'use client'

import { useState } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import {
  ChartBarIcon,
  ClockIcon,
  DocumentIcon,
  CheckCircleIcon,
  XCircleIcon,
  ArrowUpTrayIcon,
  UserGroupIcon,
  CalendarIcon,
  FolderIcon,
  ChatBubbleLeftRightIcon,
  EyeIcon,
  ArrowDownTrayIcon,
  TrashIcon,
  PencilIcon,
  PlusIcon,
  ChartPieIcon,
  FlagIcon,
} from '@heroicons/react/24/outline'

// Types
interface Task {
  id: string
  title: string
  description: string
  status: 'todo' | 'in-progress' | 'review' | 'completed'
  priority: 'low' | 'medium' | 'high' | 'critical'
  assignee: string
  dueDate: string
  progress: number
  dependencies?: string[]
}

interface Project {
  id: string
  name: string
  client: string
  status: 'planning' | 'active' | 'on-hold' | 'completed'
  progress: number
  startDate: string
  endDate: string
  budget: number
  spent: number
  tasks: Task[]
}

interface ProjectFile {
  id: string
  name: string
  type: string
  size: string
  uploadedBy: string
  uploadedAt: string
  status: 'pending' | 'approved' | 'rejected'
}

interface Approval {
  id: string
  type: string
  title: string
  description: string
  status: 'pending' | 'approved' | 'rejected'
  requestedAt: string
  requestedBy: string
}

export default function ProjectManagementDashboard() {
  const [activeTab, setActiveTab] = useState<'overview' | 'tasks' | 'gantt' | 'files' | 'approvals'>('overview')
  const [selectedProject, setSelectedProject] = useState<string>('proj-1')

  // Sample data
  const projects: Project[] = [
    {
      id: 'proj-1',
      name: 'E-Commerce Platform Redesign',
      client: 'TechCorp Inc.',
      status: 'active',
      progress: 65,
      startDate: '2024-01-15',
      endDate: '2024-06-30',
      budget: 50000,
      spent: 32500,
      tasks: [
        {
          id: 'task-1',
          title: 'Design Homepage Mockup',
          description: 'Create responsive homepage design',
          status: 'completed',
          priority: 'high',
          assignee: 'Sarah Johnson',
          dueDate: '2024-02-15',
          progress: 100,
        },
        {
          id: 'task-2',
          title: 'Implement Payment Gateway',
          description: 'Integrate Stripe payment system',
          status: 'in-progress',
          priority: 'critical',
          assignee: 'Mike Chen',
          dueDate: '2024-03-20',
          progress: 70,
          dependencies: ['task-1'],
        },
        {
          id: 'task-3',
          title: 'Product Catalog API',
          description: 'Build RESTful API for products',
          status: 'in-progress',
          priority: 'high',
          assignee: 'Alex Kumar',
          dueDate: '2024-03-10',
          progress: 45,
        },
        {
          id: 'task-4',
          title: 'User Authentication',
          description: 'Implement OAuth2 authentication',
          status: 'review',
          priority: 'critical',
          assignee: 'Lisa Park',
          dueDate: '2024-02-28',
          progress: 90,
        },
        {
          id: 'task-5',
          title: 'Mobile App UI',
          description: 'Design mobile application interface',
          status: 'todo',
          priority: 'medium',
          assignee: 'John Smith',
          dueDate: '2024-04-15',
          progress: 0,
        },
      ],
    },
    {
      id: 'proj-2',
      name: 'Mobile Banking App',
      client: 'FinBank Solutions',
      status: 'active',
      progress: 40,
      startDate: '2024-02-01',
      endDate: '2024-08-31',
      budget: 75000,
      spent: 30000,
      tasks: [],
    },
    {
      id: 'proj-3',
      name: 'Corporate Website',
      client: 'Global Industries',
      status: 'planning',
      progress: 15,
      startDate: '2024-03-01',
      endDate: '2024-05-31',
      budget: 30000,
      spent: 4500,
      tasks: [],
    },
  ]

  const projectFiles: ProjectFile[] = [
    {
      id: 'file-1',
      name: 'Homepage_Design_v3.fig',
      type: 'Figma',
      size: '2.4 MB',
      uploadedBy: 'Sarah Johnson',
      uploadedAt: '2024-02-10',
      status: 'approved',
    },
    {
      id: 'file-2',
      name: 'API_Documentation.pdf',
      type: 'PDF',
      size: '1.8 MB',
      uploadedBy: 'Alex Kumar',
      uploadedAt: '2024-02-12',
      status: 'approved',
    },
    {
      id: 'file-3',
      name: 'Payment_Flow_Diagram.png',
      type: 'Image',
      size: '856 KB',
      uploadedBy: 'Mike Chen',
      uploadedAt: '2024-02-14',
      status: 'pending',
    },
    {
      id: 'file-4',
      name: 'User_Testing_Results.xlsx',
      type: 'Excel',
      size: '3.2 MB',
      uploadedBy: 'Lisa Park',
      uploadedAt: '2024-02-15',
      status: 'pending',
    },
  ]

  const approvals: Approval[] = [
    {
      id: 'app-1',
      type: 'Design',
      title: 'Homepage Final Design',
      description: 'Final version of the homepage design for client approval',
      status: 'pending',
      requestedAt: '2024-02-14',
      requestedBy: 'Sarah Johnson',
    },
    {
      id: 'app-2',
      type: 'Milestone',
      title: 'Phase 1 Completion',
      description: 'Request approval for Phase 1 deliverables',
      status: 'approved',
      requestedAt: '2024-02-10',
      requestedBy: 'Project Manager',
    },
    {
      id: 'app-3',
      type: 'Budget',
      title: 'Additional Resources',
      description: 'Request for additional budget allocation',
      status: 'pending',
      requestedAt: '2024-02-15',
      requestedBy: 'Finance Team',
    },
  ]

  const currentProject = projects.find(p => p.id === selectedProject) || projects[0]

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed':
      case 'approved':
        return 'text-green-600 bg-green-100'
      case 'in-progress':
      case 'active':
        return 'text-blue-600 bg-blue-100'
      case 'review':
      case 'pending':
        return 'text-yellow-600 bg-yellow-100'
      case 'todo':
      case 'planning':
        return 'text-gray-600 bg-gray-100'
      case 'on-hold':
      case 'rejected':
        return 'text-red-600 bg-red-100'
      default:
        return 'text-gray-600 bg-gray-100'
    }
  }

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'critical':
        return 'text-red-600 bg-red-100'
      case 'high':
        return 'text-orange-600 bg-orange-100'
      case 'medium':
        return 'text-yellow-600 bg-yellow-100'
      case 'low':
        return 'text-green-600 bg-green-100'
      default:
        return 'text-gray-600 bg-gray-100'
    }
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-white to-blue-50 py-12">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <motion.div
          initial={{ opacity: 0, y: -20 }}
          animate={{ opacity: 1, y: 0 }}
          className="mb-8"
        >
          <h1 className="text-4xl font-bold text-gray-900 mb-2">Project Management Dashboard</h1>
          <p className="text-gray-600">Track progress, manage tasks, and collaborate with your team</p>
        </motion.div>

        {/* Project Selector */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          className="mb-8"
        >
          <div className="bg-white rounded-2xl shadow-lg p-6">
            <label className="block text-sm font-semibold text-gray-700 mb-3">Select Project</label>
            <select
              value={selectedProject}
              onChange={(e) => setSelectedProject(e.target.value)}
              className="w-full md:w-96 px-4 py-3 border border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all"
            >
              {projects.map((project) => (
                <option key={project.id} value={project.id}>
                  {project.name} - {project.client}
                </option>
              ))}
            </select>
          </div>
        </motion.div>

        {/* Tabs */}
        <div className="bg-white rounded-2xl shadow-lg overflow-hidden mb-8">
          <div className="border-b border-gray-200">
            <nav className="flex -mb-px overflow-x-auto">
              {([
                { id: 'overview' as const, label: 'Overview', icon: ChartPieIcon },
                { id: 'tasks' as const, label: 'Tasks', icon: CheckCircleIcon },
                { id: 'gantt' as const, label: 'Timeline', icon: CalendarIcon },
                { id: 'files' as const, label: 'Files', icon: FolderIcon },
                { id: 'approvals' as const, label: 'Approvals', icon: FlagIcon },
              ]).map((tab) => {
                const Icon = tab.icon
                return (
                  <button
                    key={tab.id}
                    onClick={() => setActiveTab(tab.id)}
                    className={`flex items-center gap-2 px-6 py-4 border-b-2 font-medium transition-colors whitespace-nowrap ${
                      activeTab === tab.id
                        ? 'border-blue-600 text-blue-600'
                        : 'border-transparent text-gray-600 hover:text-gray-900 hover:border-gray-300'
                    }`}
                  >
                    <Icon className="w-5 h-5" />
                    {tab.label}
                  </button>
                )
              })}
            </nav>
          </div>

          <div className="p-6">
            <AnimatePresence mode="wait">
              {/* Overview Tab */}
              {activeTab === 'overview' && (
                <motion.div
                  key="overview"
                  initial={{ opacity: 0, x: -20 }}
                  animate={{ opacity: 1, x: 0 }}
                  exit={{ opacity: 0, x: 20 }}
                  className="space-y-6"
                >
                  {/* Project Stats */}
                  <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
                    <div className="bg-gradient-to-br from-blue-50 to-blue-100 rounded-xl p-6">
                      <div className="flex items-center justify-between mb-3">
                        <ChartBarIcon className="w-8 h-8 text-blue-600" />
                        <span className={`px-3 py-1 rounded-full text-xs font-semibold ${getStatusColor(currentProject.status)}`}>
                          {currentProject.status}
                        </span>
                      </div>
                      <h3 className="text-2xl font-bold text-gray-900">{currentProject.progress}%</h3>
                      <p className="text-sm text-gray-600">Project Progress</p>
                    </div>

                    <div className="bg-gradient-to-br from-green-50 to-green-100 rounded-xl p-6">
                      <div className="flex items-center justify-between mb-3">
                        <CheckCircleIcon className="w-8 h-8 text-green-600" />
                      </div>
                      <h3 className="text-2xl font-bold text-gray-900">
                        {currentProject.tasks.filter(t => t.status === 'completed').length}/{currentProject.tasks.length}
                      </h3>
                      <p className="text-sm text-gray-600">Tasks Completed</p>
                    </div>

                    <div className="bg-gradient-to-br from-purple-50 to-purple-100 rounded-xl p-6">
                      <div className="flex items-center justify-between mb-3">
                        <ClockIcon className="w-8 h-8 text-purple-600" />
                      </div>
                      <h3 className="text-2xl font-bold text-gray-900">
                        {Math.ceil((new Date(currentProject.endDate).getTime() - new Date().getTime()) / (1000 * 60 * 60 * 24))} days
                      </h3>
                      <p className="text-sm text-gray-600">Until Deadline</p>
                    </div>

                    <div className="bg-gradient-to-br from-orange-50 to-orange-100 rounded-xl p-6">
                      <div className="flex items-center justify-between mb-3">
                        <ChartPieIcon className="w-8 h-8 text-orange-600" />
                      </div>
                      <h3 className="text-2xl font-bold text-gray-900">
                        ${currentProject.spent.toLocaleString()}
                      </h3>
                      <p className="text-sm text-gray-600">of ${currentProject.budget.toLocaleString()} budget</p>
                    </div>
                  </div>

                  {/* Progress Bar */}
                  <div className="bg-gray-50 rounded-xl p-6">
                    <div className="flex items-center justify-between mb-3">
                      <h3 className="font-semibold text-gray-900">Overall Progress</h3>
                      <span className="text-2xl font-bold text-blue-600">{currentProject.progress}%</span>
                    </div>
                    <div className="w-full h-4 bg-gray-200 rounded-full overflow-hidden">
                      <motion.div
                        initial={{ width: 0 }}
                        animate={{ width: `${currentProject.progress}%` }}
                        transition={{ duration: 1, ease: 'easeOut' }}
                        className="h-full bg-gradient-to-r from-blue-500 to-purple-500"
                      />
                    </div>
                  </div>

                  {/* Budget Overview */}
                  <div className="bg-gray-50 rounded-xl p-6">
                    <h3 className="font-semibold text-gray-900 mb-4">Budget Overview</h3>
                    <div className="space-y-3">
                      <div className="flex items-center justify-between">
                        <span className="text-gray-600">Total Budget</span>
                        <span className="font-semibold text-gray-900">${currentProject.budget.toLocaleString()}</span>
                      </div>
                      <div className="flex items-center justify-between">
                        <span className="text-gray-600">Spent</span>
                        <span className="font-semibold text-orange-600">${currentProject.spent.toLocaleString()}</span>
                      </div>
                      <div className="flex items-center justify-between">
                        <span className="text-gray-600">Remaining</span>
                        <span className="font-semibold text-green-600">
                          ${(currentProject.budget - currentProject.spent).toLocaleString()}
                        </span>
                      </div>
                      <div className="w-full h-3 bg-gray-200 rounded-full overflow-hidden mt-4">
                        <div
                          className="h-full bg-gradient-to-r from-green-500 to-orange-500"
                          style={{ width: `${(currentProject.spent / currentProject.budget) * 100}%` }}
                        />
                      </div>
                    </div>
                  </div>

                  {/* Recent Activity */}
                  <div className="bg-gray-50 rounded-xl p-6">
                    <h3 className="font-semibold text-gray-900 mb-4">Recent Activity</h3>
                    <div className="space-y-4">
                      {[
                        { user: 'Sarah Johnson', action: 'completed task', item: 'Design Homepage Mockup', time: '2 hours ago' },
                        { user: 'Mike Chen', action: 'uploaded file', item: 'Payment_Flow_Diagram.png', time: '5 hours ago' },
                        { user: 'Lisa Park', action: 'requested approval for', item: 'User Authentication', time: '1 day ago' },
                        { user: 'Alex Kumar', action: 'updated task', item: 'Product Catalog API', time: '2 days ago' },
                      ].map((activity, index) => (
                        <div key={index} className="flex items-start gap-4">
                          <div className="w-2 h-2 bg-blue-600 rounded-full mt-2" />
                          <div className="flex-1">
                            <p className="text-sm text-gray-900">
                              <span className="font-semibold">{activity.user}</span> {activity.action}{' '}
                              <span className="font-semibold">{activity.item}</span>
                            </p>
                            <p className="text-xs text-gray-500">{activity.time}</p>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                </motion.div>
              )}

              {/* Tasks Tab */}
              {activeTab === 'tasks' && (
                <motion.div
                  key="tasks"
                  initial={{ opacity: 0, x: -20 }}
                  animate={{ opacity: 1, x: 0 }}
                  exit={{ opacity: 0, x: 20 }}
                >
                  <div className="flex items-center justify-between mb-6">
                    <h2 className="text-2xl font-bold text-gray-900">Task Management</h2>
                    <button
                      className="flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-xl hover:bg-blue-700 transition-colors"
                    >
                      <PlusIcon className="w-5 h-5" />
                      New Task
                    </button>
                  </div>

                  <div className="space-y-4">
                    {currentProject.tasks.map((task) => (
                      <motion.div
                        key={task.id}
                        initial={{ opacity: 0, y: 10 }}
                        animate={{ opacity: 1, y: 0 }}
                        className="bg-white border border-gray-200 rounded-xl p-6 hover:shadow-lg transition-shadow"
                      >
                        <div className="flex items-start justify-between mb-4">
                          <div className="flex-1">
                            <div className="flex items-center gap-3 mb-2">
                              <h3 className="text-lg font-semibold text-gray-900">{task.title}</h3>
                              <span className={`px-3 py-1 rounded-full text-xs font-semibold ${getStatusColor(task.status)}`}>
                                {task.status}
                              </span>
                              <span className={`px-3 py-1 rounded-full text-xs font-semibold ${getPriorityColor(task.priority)}`}>
                                {task.priority}
                              </span>
                            </div>
                            <p className="text-gray-600 text-sm mb-3">{task.description}</p>
                            <div className="flex items-center gap-6 text-sm text-gray-500">
                              <div className="flex items-center gap-2">
                                <UserGroupIcon className="w-4 h-4" />
                                {task.assignee}
                              </div>
                              <div className="flex items-center gap-2">
                                <CalendarIcon className="w-4 h-4" />
                                Due: {new Date(task.dueDate).toLocaleDateString()}
                              </div>
                            </div>
                          </div>
                          <div className="flex items-center gap-2">
                            <button className="p-2 text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-colors">
                              <PencilIcon className="w-5 h-5" />
                            </button>
                            <button className="p-2 text-gray-600 hover:text-red-600 hover:bg-red-50 rounded-lg transition-colors">
                              <TrashIcon className="w-5 h-5" />
                            </button>
                          </div>
                        </div>

                        {/* Task Progress */}
                        <div className="mt-4">
                          <div className="flex items-center justify-between mb-2">
                            <span className="text-sm font-medium text-gray-700">Progress</span>
                            <span className="text-sm font-semibold text-blue-600">{task.progress}%</span>
                          </div>
                          <div className="w-full h-2 bg-gray-200 rounded-full overflow-hidden">
                            <div
                              className="h-full bg-gradient-to-r from-blue-500 to-purple-500"
                              style={{ width: `${task.progress}%` }}
                            />
                          </div>
                        </div>
                      </motion.div>
                    ))}
                  </div>
                </motion.div>
              )}

              {/* Gantt Chart Tab */}
              {activeTab === 'gantt' && (
                <motion.div
                  key="gantt"
                  initial={{ opacity: 0, x: -20 }}
                  animate={{ opacity: 1, x: 0 }}
                  exit={{ opacity: 0, x: 20 }}
                >
                  <h2 className="text-2xl font-bold text-gray-900 mb-6">Project Timeline</h2>
                  
                  <div className="bg-gradient-to-r from-blue-50 to-purple-50 rounded-xl p-8">
                    <div className="text-center">
                      <CalendarIcon className="w-16 h-16 text-blue-600 mx-auto mb-4" />
                      <h3 className="text-xl font-semibold text-gray-900 mb-2">Gantt Chart View</h3>
                      <p className="text-gray-600 mb-6">
                        Visual timeline representation showing task dependencies and project milestones
                      </p>
                      
                      {/* Simple Timeline Visualization */}
                      <div className="bg-white rounded-xl p-6 mt-8">
                        <div className="space-y-4">
                          {currentProject.tasks.slice(0, 5).map((task, index) => (
                            <div key={task.id} className="flex items-center gap-4">
                              <div className="w-32 text-left">
                                <p className="text-sm font-medium text-gray-900 truncate">{task.title}</p>
                              </div>
                              <div className="flex-1 relative h-12 bg-gray-100 rounded-lg overflow-hidden">
                                <motion.div
                                  initial={{ width: 0 }}
                                  animate={{ width: `${60 + index * 5}%` }}
                                  transition={{ duration: 0.8, delay: index * 0.1 }}
                                  className={`absolute left-0 top-0 h-full rounded-lg ${
                                    task.status === 'completed'
                                      ? 'bg-gradient-to-r from-green-500 to-green-600'
                                      : task.status === 'in-progress'
                                      ? 'bg-gradient-to-r from-blue-500 to-purple-500'
                                      : 'bg-gray-300'
                                  }`}
                                >
                                  <span className="absolute right-2 top-1/2 transform -translate-y-1/2 text-xs font-semibold text-white">
                                    {task.progress}%
                                  </span>
                                </motion.div>
                              </div>
                              <div className="w-24 text-right">
                                <span className="text-xs text-gray-500">{new Date(task.dueDate).toLocaleDateString()}</span>
                              </div>
                            </div>
                          ))}
                        </div>
                      </div>
                    </div>
                  </div>
                </motion.div>
              )}

              {/* Files Tab */}
              {activeTab === 'files' && (
                <motion.div
                  key="files"
                  initial={{ opacity: 0, x: -20 }}
                  animate={{ opacity: 1, x: 0 }}
                  exit={{ opacity: 0, x: 20 }}
                >
                  <div className="flex items-center justify-between mb-6">
                    <h2 className="text-2xl font-bold text-gray-900">Project Files</h2>
                    <button
                      className="flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-xl hover:bg-blue-700 transition-colors"
                    >
                      <ArrowUpTrayIcon className="w-5 h-5" />
                      Upload File
                    </button>
                  </div>

                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                    {projectFiles.map((file) => (
                      <motion.div
                        key={file.id}
                        initial={{ opacity: 0, scale: 0.95 }}
                        animate={{ opacity: 1, scale: 1 }}
                        className="bg-white border border-gray-200 rounded-xl p-6 hover:shadow-lg transition-shadow"
                      >
                        <div className="flex items-start justify-between mb-4">
                          <DocumentIcon className="w-12 h-12 text-blue-600" />
                          <span className={`px-3 py-1 rounded-full text-xs font-semibold ${getStatusColor(file.status)}`}>
                            {file.status}
                          </span>
                        </div>
                        <h3 className="font-semibold text-gray-900 mb-2 truncate">{file.name}</h3>
                        <div className="space-y-2 text-sm text-gray-600 mb-4">
                          <p>Type: {file.type}</p>
                          <p>Size: {file.size}</p>
                          <p>Uploaded by: {file.uploadedBy}</p>
                          <p>{file.uploadedAt}</p>
                        </div>
                        <div className="flex items-center gap-2">
                          <button className="flex-1 flex items-center justify-center gap-2 px-3 py-2 bg-blue-50 text-blue-600 rounded-lg hover:bg-blue-100 transition-colors">
                            <EyeIcon className="w-4 h-4" />
                            View
                          </button>
                          <button className="flex-1 flex items-center justify-center gap-2 px-3 py-2 bg-green-50 text-green-600 rounded-lg hover:bg-green-100 transition-colors">
                            <ArrowDownTrayIcon className="w-4 h-4" />
                            Download
                          </button>
                        </div>
                      </motion.div>
                    ))}
                  </div>
                </motion.div>
              )}

              {/* Approvals Tab */}
              {activeTab === 'approvals' && (
                <motion.div
                  key="approvals"
                  initial={{ opacity: 0, x: -20 }}
                  animate={{ opacity: 1, x: 0 }}
                  exit={{ opacity: 0, x: 20 }}
                >
                  <h2 className="text-2xl font-bold text-gray-900 mb-6">Client Approvals</h2>

                  <div className="space-y-4">
                    {approvals.map((approval) => (
                      <motion.div
                        key={approval.id}
                        initial={{ opacity: 0, y: 10 }}
                        animate={{ opacity: 1, y: 0 }}
                        className="bg-white border border-gray-200 rounded-xl p-6 hover:shadow-lg transition-shadow"
                      >
                        <div className="flex items-start justify-between">
                          <div className="flex-1">
                            <div className="flex items-center gap-3 mb-3">
                              <FlagIcon className="w-6 h-6 text-blue-600" />
                              <h3 className="text-lg font-semibold text-gray-900">{approval.title}</h3>
                              <span className={`px-3 py-1 rounded-full text-xs font-semibold ${getStatusColor(approval.status)}`}>
                                {approval.status}
                              </span>
                            </div>
                            <p className="text-gray-600 mb-4">{approval.description}</p>
                            <div className="flex items-center gap-6 text-sm text-gray-500">
                              <span>Type: {approval.type}</span>
                              <span>Requested by: {approval.requestedBy}</span>
                              <span>Date: {approval.requestedAt}</span>
                            </div>
                          </div>
                          {approval.status === 'pending' && (
                            <div className="flex items-center gap-2 ml-4">
                              <button className="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors flex items-center gap-2">
                                <CheckCircleIcon className="w-5 h-5" />
                                Approve
                              </button>
                              <button className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors flex items-center gap-2">
                                <XCircleIcon className="w-5 h-5" />
                                Reject
                              </button>
                            </div>
                          )}
                        </div>
                      </motion.div>
                    ))}
                  </div>
                </motion.div>
              )}
            </AnimatePresence>
          </div>
        </div>

        {/* Team Communication */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          className="bg-gradient-to-r from-blue-600 to-purple-600 rounded-2xl p-8 text-white"
        >
          <div className="flex items-center justify-between">
            <div>
              <h3 className="text-2xl font-bold mb-2">Need to discuss something?</h3>
              <p className="text-blue-100">Connect with your team and clients in real-time</p>
            </div>
            <button className="flex items-center gap-2 px-6 py-3 bg-white text-blue-600 rounded-xl font-semibold hover:bg-blue-50 transition-colors">
              <ChatBubbleLeftRightIcon className="w-5 h-5" />
              Open Team Chat
            </button>
          </div>
        </motion.div>
      </div>
    </div>
  )
}
