'use client'

import { useState } from 'react'
import { motion, AnimatePresence } from 'framer-motion'
import { 
  CheckIcon,
  XMarkIcon,
  PlusIcon,
  MinusIcon,
  ChartBarIcon,
  ClockIcon,
  CurrencyDollarIcon,
  UserGroupIcon
} from '@heroicons/react/24/outline'
import { useCartStore } from '@/store/cart'

interface Service {
  id: number
  title: string
  description: string
  price: number
  duration: string
  category: string
  slug: string
  features: string[]
  popular?: boolean
  packages?: {
    name: string
    price: number
    features: string[]
  }[]
}

interface ServiceComparisonProps {
  availableServices: Service[]
}

export default function ServiceComparison({ availableServices }: ServiceComparisonProps) {
  const [selectedServices, setSelectedServices] = useState<Service[]>([])
  const [showComparison, setShowComparison] = useState(false)
  const addToCart = useCartStore((state) => state.addToCart)
  const openCart = useCartStore((state) => state.openCart)

  const addServiceToComparison = (service: Service) => {
    if (selectedServices.length < 3 && !selectedServices.find(s => s.id === service.id)) {
      setSelectedServices([...selectedServices, service])
      setShowComparison(true)
    }
  }

  const removeServiceFromComparison = (serviceId: number) => {
    setSelectedServices(selectedServices.filter(s => s.id !== serviceId))
    if (selectedServices.length <= 1) {
      setShowComparison(false)
    }
  }

  const handleAddToCart = (service: Service) => {
    addToCart({
      id: service.id,
      title: service.title,
      price: service.price,
      quantity: 1
    })
    setTimeout(() => openCart(), 100)
  }

  // Get all unique features across selected services
  const allFeatures = [...new Set(selectedServices.flatMap(s => s.features))]

  return (
    <>
      {/* Service Selection */}
      <div className="bg-white rounded-2xl shadow-lg p-6 mb-8">
        <div className="flex items-center justify-between mb-6">
          <div>
            <h2 className="text-2xl font-bold text-gray-900">Compare Services</h2>
            <p className="text-gray-600">Select up to 3 services to compare features and pricing</p>
          </div>
          {selectedServices.length > 0 && (
            <button
              onClick={() => setShowComparison(!showComparison)}
              className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors"
            >
              {showComparison ? 'Hide' : 'Show'} Comparison ({selectedServices.length})
            </button>
          )}
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
          {availableServices.map(service => {
            const isSelected = selectedServices.find(s => s.id === service.id)
            const canAdd = selectedServices.length < 3 && !isSelected

            return (
              <div
                key={service.id}
                className={`border rounded-xl p-4 transition-all ${
                  isSelected 
                    ? 'border-blue-500 bg-blue-50' 
                    : 'border-gray-200 hover:border-gray-300'
                }`}
              >
                <div className="flex items-start justify-between mb-3">
                  <div className="flex-1">
                    <h3 className="font-semibold text-gray-900">{service.title}</h3>
                    <p className="text-sm text-gray-600 mt-1">{service.category}</p>
                    <div className="flex items-center gap-4 mt-2 text-sm text-gray-500">
                      <div className="flex items-center gap-1">
                        <CurrencyDollarIcon className="h-4 w-4" />
                        <span>${service.price.toLocaleString()}</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <ClockIcon className="h-4 w-4" />
                        <span>{service.duration}</span>
                      </div>
                    </div>
                  </div>
                  
                  <button
                    onClick={() => 
                      isSelected 
                        ? removeServiceFromComparison(service.id)
                        : addServiceToComparison(service)
                    }
                    disabled={!canAdd && !isSelected}
                    className={`p-2 rounded-lg transition-colors ${
                      isSelected
                        ? 'bg-red-100 text-red-600 hover:bg-red-200'
                        : canAdd
                        ? 'bg-blue-100 text-blue-600 hover:bg-blue-200'
                        : 'bg-gray-100 text-gray-400 cursor-not-allowed'
                    }`}
                  >
                    {isSelected ? (
                      <MinusIcon className="h-5 w-5" />
                    ) : (
                      <PlusIcon className="h-5 w-5" />
                    )}
                  </button>
                </div>

                <div className="space-y-1">
                  {service.features.slice(0, 3).map((feature, index) => (
                    <div key={index} className="flex items-center gap-2 text-sm text-gray-600">
                      <CheckIcon className="h-3 w-3 text-green-500" />
                      <span>{feature}</span>
                    </div>
                  ))}
                  {service.features.length > 3 && (
                    <p className="text-sm text-gray-500">+{service.features.length - 3} more features</p>
                  )}
                </div>
              </div>
            )
          })}
        </div>
      </div>

      {/* Comparison Table */}
      <AnimatePresence>
        {showComparison && selectedServices.length > 0 && (
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            exit={{ opacity: 0, y: -20 }}
            className="bg-white rounded-2xl shadow-lg overflow-hidden mb-8"
          >
            <div className="p-6 border-b border-gray-200">
              <div className="flex items-center justify-between">
                <h3 className="text-xl font-bold text-gray-900">Service Comparison</h3>
                <button
                  onClick={() => setShowComparison(false)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  <XMarkIcon className="h-6 w-6" />
                </button>
              </div>
            </div>

            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-4 text-left text-sm font-medium text-gray-500 uppercase tracking-wider">
                      Features
                    </th>
                    {selectedServices.map(service => (
                      <th key={service.id} className="px-6 py-4 text-center min-w-64">
                        <div className="space-y-2">
                          <h4 className="font-semibold text-gray-900">{service.title}</h4>
                          <div className="text-2xl font-bold text-blue-600">
                            ${service.price.toLocaleString()}
                          </div>
                          <div className="text-sm text-gray-500">{service.duration}</div>
                          {service.popular && (
                            <span className="inline-block bg-orange-100 text-orange-800 text-xs px-2 py-1 rounded-full">
                              Popular
                            </span>
                          )}
                        </div>
                      </th>
                    ))}
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {/* Basic Info */}
                  <tr>
                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                      Category
                    </td>
                    {selectedServices.map(service => (
                      <td key={service.id} className="px-6 py-4 whitespace-nowrap text-center text-sm text-gray-500">
                        {service.category}
                      </td>
                    ))}
                  </tr>

                  {/* Features Comparison */}
                  {allFeatures.map(feature => (
                    <tr key={feature}>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-700">
                        {feature}
                      </td>
                      {selectedServices.map(service => (
                        <td key={service.id} className="px-6 py-4 whitespace-nowrap text-center">
                          {service.features.includes(feature) ? (
                            <CheckIcon className="h-5 w-5 text-green-500 mx-auto" />
                          ) : (
                            <XMarkIcon className="h-5 w-5 text-red-500 mx-auto" />
                          )}
                        </td>
                      ))}
                    </tr>
                  ))}

                  {/* Action Buttons */}
                  <tr className="bg-gray-50">
                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                      Actions
                    </td>
                    {selectedServices.map(service => (
                      <td key={service.id} className="px-6 py-4 whitespace-nowrap text-center">
                        <div className="space-y-2">
                          <button
                            onClick={() => handleAddToCart(service)}
                            className="w-full bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors text-sm"
                          >
                            Add to Cart
                          </button>
                          <button
                            onClick={() => removeServiceFromComparison(service.id)}
                            className="w-full text-gray-500 hover:text-gray-700 text-sm"
                          >
                            Remove
                          </button>
                        </div>
                      </td>
                    ))}
                  </tr>
                </tbody>
              </table>
            </div>

            {/* Comparison Insights */}
            <div className="p-6 bg-gray-50 border-t border-gray-200">
              <h4 className="font-semibold text-gray-900 mb-4">Comparison Insights</h4>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div className="bg-white p-4 rounded-lg">
                  <div className="flex items-center gap-2 mb-2">
                    <CurrencyDollarIcon className="h-5 w-5 text-green-500" />
                    <span className="font-medium text-gray-900">Best Value</span>
                  </div>
                  <p className="text-sm text-gray-600">
                    {selectedServices.reduce((prev, current) => 
                      (prev.price / prev.features.length) < (current.price / current.features.length) ? prev : current
                    ).title} offers the most features per dollar
                  </p>
                </div>

                <div className="bg-white p-4 rounded-lg">
                  <div className="flex items-center gap-2 mb-2">
                    <ChartBarIcon className="h-5 w-5 text-blue-500" />
                    <span className="font-medium text-gray-900">Most Features</span>
                  </div>
                  <p className="text-sm text-gray-600">
                    {selectedServices.reduce((prev, current) => 
                      prev.features.length > current.features.length ? prev : current
                    ).title} includes the most features ({
                      selectedServices.reduce((prev, current) => 
                        prev.features.length > current.features.length ? prev : current
                      ).features.length
                    } total)
                  </p>
                </div>

                <div className="bg-white p-4 rounded-lg">
                  <div className="flex items-center gap-2 mb-2">
                    <ClockIcon className="h-5 w-5 text-orange-500" />
                    <span className="font-medium text-gray-900">Fastest Delivery</span>
                  </div>
                  <p className="text-sm text-gray-600">
                    Compare delivery times to choose the best option for your timeline
                  </p>
                </div>
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>
    </>
  )
}