'use client'

import { useState, useEffect } from 'react'
import { useSession } from 'next-auth/react'
import { motion, AnimatePresence } from 'framer-motion'
import {
  CreditCardIcon,
  CalendarIcon,
  ClockIcon,
  BanknotesIcon,
  CheckCircleIcon,
  XCircleIcon,
  PauseIcon,
  PlayIcon,
  ArrowPathIcon,
  ExclamationTriangleIcon,
  InformationCircleIcon
} from '@heroicons/react/24/outline'

interface Subscription {
  id: string
  serviceId: string
  serviceName: string
  plan: SubscriptionPlan
  status: 'active' | 'paused' | 'cancelled' | 'expired'
  startDate: Date
  nextBillingDate: Date
  lastPaymentDate?: Date
  totalPaid: number
  paymentMethod: string
  autoRenew: boolean
}

interface SubscriptionPlan {
  id: string
  name: string
  price: number
  currency: string
  interval: 'monthly' | 'quarterly' | 'yearly'
  intervalCount: number
  features: string[]
  popular?: boolean
  discountPercent?: number
}

interface BillingHistory {
  id: string
  subscriptionId: string
  amount: number
  currency: string
  status: 'paid' | 'failed' | 'pending' | 'refunded'
  paymentDate: Date
  paymentMethod: string
  invoiceUrl?: string
  failureReason?: string
}

const SUBSCRIPTION_PLANS: SubscriptionPlan[] = [
  {
    id: 'hosting-basic',
    name: 'Basic Hosting',
    price: 999,
    currency: 'PKR',
    interval: 'monthly',
    intervalCount: 1,
    features: ['5GB Storage', '10GB Bandwidth', '1 Domain', 'Email Support'],
    discountPercent: 10
  },
  {
    id: 'hosting-pro',
    name: 'Pro Hosting',
    price: 1999,
    currency: 'PKR',
    interval: 'monthly',
    intervalCount: 1,
    features: ['20GB Storage', '50GB Bandwidth', '5 Domains', 'Priority Support', 'SSL Certificate'],
    popular: true,
    discountPercent: 15
  },
  {
    id: 'maintenance',
    name: 'Website Maintenance',
    price: 5000,
    currency: 'PKR',
    interval: 'monthly',
    intervalCount: 1,
    features: ['Daily Backups', 'Security Updates', 'Performance Monitoring', '24/7 Support'],
    discountPercent: 20
  },
  {
    id: 'seo-monthly',
    name: 'SEO Service',
    price: 15000,
    currency: 'PKR',
    interval: 'monthly',
    intervalCount: 1,
    features: ['Keyword Research', 'Content Optimization', 'Monthly Reports', 'Link Building'],
    discountPercent: 25
  }
]

const MOCK_SUBSCRIPTIONS: Subscription[] = [
  {
    id: 'sub-1',
    serviceId: 'hosting-pro',
    serviceName: 'Pro Hosting Package',
    plan: SUBSCRIPTION_PLANS[1],
    status: 'active',
    startDate: new Date('2024-10-01'),
    nextBillingDate: new Date('2024-11-01'),
    lastPaymentDate: new Date('2024-10-01'),
    totalPaid: 3998,
    paymentMethod: 'Credit Card (**** 1234)',
    autoRenew: true
  },
  {
    id: 'sub-2',
    serviceId: 'maintenance',
    serviceName: 'Website Maintenance',
    plan: SUBSCRIPTION_PLANS[2],
    status: 'active',
    startDate: new Date('2024-09-15'),
    nextBillingDate: new Date('2024-11-15'),
    lastPaymentDate: new Date('2024-10-15'),
    totalPaid: 10000,
    paymentMethod: 'JazzCash',
    autoRenew: true
  }
]

const MOCK_BILLING_HISTORY: BillingHistory[] = [
  {
    id: 'bill-1',
    subscriptionId: 'sub-1',
    amount: 1999,
    currency: 'PKR',
    status: 'paid',
    paymentDate: new Date('2024-10-01'),
    paymentMethod: 'Credit Card',
    invoiceUrl: '/invoices/INV-2024-001.pdf'
  },
  {
    id: 'bill-2',
    subscriptionId: 'sub-1',
    amount: 1999,
    currency: 'PKR',
    status: 'paid',
    paymentDate: new Date('2024-09-01'),
    paymentMethod: 'Credit Card',
    invoiceUrl: '/invoices/INV-2024-002.pdf'
  },
  {
    id: 'bill-3',
    subscriptionId: 'sub-2',
    amount: 5000,
    currency: 'PKR',
    status: 'paid',
    paymentDate: new Date('2024-10-15'),
    paymentMethod: 'JazzCash'
  }
]

export default function SubscriptionBilling() {
  const { data: session } = useSession()
  const [subscriptions, setSubscriptions] = useState<Subscription[]>([])
  const [billingHistory, setBillingHistory] = useState<BillingHistory[]>([])
  const [selectedTab, setSelectedTab] = useState<'subscriptions' | 'plans' | 'history'>('subscriptions')
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    if (session?.user) {
      // Simulate API call
      setTimeout(() => {
        setSubscriptions(MOCK_SUBSCRIPTIONS)
        setBillingHistory(MOCK_BILLING_HISTORY)
        setLoading(false)
      }, 1000)
    }
  }, [session])

  const getStatusColor = (status: Subscription['status']) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800'
      case 'paused': return 'bg-yellow-100 text-yellow-800'
      case 'cancelled': return 'bg-red-100 text-red-800'
      case 'expired': return 'bg-gray-100 text-gray-800'
      default: return 'bg-gray-100 text-gray-800'
    }
  }

  const getStatusIcon = (status: Subscription['status']) => {
    switch (status) {
      case 'active': return <CheckCircleIcon className="w-4 h-4" />
      case 'paused': return <PauseIcon className="w-4 h-4" />
      case 'cancelled': return <XCircleIcon className="w-4 h-4" />
      case 'expired': return <ClockIcon className="w-4 h-4" />
      default: return <ClockIcon className="w-4 h-4" />
    }
  }

  const formatCurrency = (amount: number, currency: string = 'PKR'): string => {
    if (currency === 'PKR') {
      return `Rs. ${amount.toLocaleString('en-PK')}`
    }
    return `$${amount.toLocaleString('en-US')}`
  }

  const calculateYearlyDiscount = (plan: SubscriptionPlan): number => {
    const monthlyTotal = plan.price * 12
    const discount = plan.discountPercent || 0
    return monthlyTotal * (discount / 100)
  }

  const handleSubscriptionAction = async (subscriptionId: string, action: 'pause' | 'resume' | 'cancel') => {
    setSubscriptions(prev => prev.map(sub => 
      sub.id === subscriptionId 
        ? { 
            ...sub, 
            status: action === 'pause' ? 'paused' : action === 'resume' ? 'active' : 'cancelled',
            autoRenew: action === 'cancel' ? false : sub.autoRenew
          }
        : sub
    ))
  }

  const handleToggleAutoRenew = (subscriptionId: string) => {
    setSubscriptions(prev => prev.map(sub => 
      sub.id === subscriptionId 
        ? { ...sub, autoRenew: !sub.autoRenew }
        : sub
    ))
  }

  const renderSubscriptions = () => (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h3 className="text-lg font-semibold text-gray-900">Active Subscriptions</h3>
        <button
          onClick={() => setSelectedTab('plans')}
          className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
        >
          Subscribe to New Plan
        </button>
      </div>

      {loading ? (
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          {[1, 2].map(i => (
            <div key={i} className="border border-gray-200 rounded-lg p-6 animate-pulse">
              <div className="h-4 bg-gray-200 rounded w-3/4 mb-4"></div>
              <div className="space-y-3">
                <div className="h-3 bg-gray-200 rounded"></div>
                <div className="h-3 bg-gray-200 rounded w-1/2"></div>
              </div>
            </div>
          ))}
        </div>
      ) : subscriptions.length === 0 ? (
        <div className="text-center py-12">
          <CreditCardIcon className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <h4 className="text-xl font-semibold text-gray-900 mb-2">No Active Subscriptions</h4>
          <p className="text-gray-600 mb-6">Start with a hosting or maintenance plan today.</p>
          <button
            onClick={() => setSelectedTab('plans')}
            className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            Browse Plans
          </button>
        </div>
      ) : (
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {subscriptions.map((subscription) => (
            <motion.div
              key={subscription.id}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              className="border border-gray-200 rounded-lg p-6 hover:shadow-lg transition-shadow"
            >
              <div className="flex items-start justify-between mb-4">
                <div>
                  <h4 className="font-semibold text-gray-900">{subscription.serviceName}</h4>
                  <p className="text-sm text-gray-600">{subscription.plan.name}</p>
                </div>
                <div className={`flex items-center space-x-1 px-2 py-1 rounded-full text-xs font-medium ${getStatusColor(subscription.status)}`}>
                  {getStatusIcon(subscription.status)}
                  <span className="capitalize">{subscription.status}</span>
                </div>
              </div>

              <div className="space-y-3 mb-4">
                <div className="flex justify-between text-sm">
                  <span className="text-gray-600">Monthly Cost:</span>
                  <span className="font-medium">{formatCurrency(subscription.plan.price)}</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span className="text-gray-600">Next Billing:</span>
                  <span className="font-medium">{subscription.nextBillingDate.toLocaleDateString()}</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span className="text-gray-600">Payment Method:</span>
                  <span className="font-medium">{subscription.paymentMethod}</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span className="text-gray-600">Total Paid:</span>
                  <span className="font-medium">{formatCurrency(subscription.totalPaid)}</span>
                </div>
              </div>

              <div className="flex items-center justify-between mb-4">
                <div className="flex items-center space-x-2">
                  <input
                    type="checkbox"
                    checked={subscription.autoRenew}
                    onChange={() => handleToggleAutoRenew(subscription.id)}
                    className="h-4 w-4 text-blue-600 rounded border-gray-300 focus:ring-blue-500"
                  />
                  <span className="text-sm text-gray-700">Auto-renew</span>
                </div>
                {subscription.status === 'active' && !subscription.autoRenew && (
                  <span className="text-xs text-orange-600 flex items-center">
                    <ExclamationTriangleIcon className="w-3 h-3 mr-1" />
                    Will expire on {subscription.nextBillingDate.toLocaleDateString()}
                  </span>
                )}
              </div>

              <div className="flex space-x-2">
                {subscription.status === 'active' ? (
                  <button
                    onClick={() => handleSubscriptionAction(subscription.id, 'pause')}
                    className="flex-1 px-3 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors text-sm"
                  >
                    Pause
                  </button>
                ) : subscription.status === 'paused' ? (
                  <button
                    onClick={() => handleSubscriptionAction(subscription.id, 'resume')}
                    className="flex-1 px-3 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors text-sm"
                  >
                    Resume
                  </button>
                ) : null}
                
                <button
                  onClick={() => handleSubscriptionAction(subscription.id, 'cancel')}
                  className="flex-1 px-3 py-2 border border-red-300 text-red-700 rounded-lg hover:bg-red-50 transition-colors text-sm"
                  disabled={subscription.status === 'cancelled'}
                >
                  {subscription.status === 'cancelled' ? 'Cancelled' : 'Cancel'}
                </button>
              </div>
            </motion.div>
          ))}
        </div>
      )}
    </div>
  )

  const renderPlans = () => (
    <div className="space-y-6">
      <div>
        <h3 className="text-lg font-semibold text-gray-900 mb-2">Subscription Plans</h3>
        <p className="text-gray-600">Choose a plan that fits your needs. All plans include our standard support.</p>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {SUBSCRIPTION_PLANS.map((plan) => (
          <motion.div
            key={plan.id}
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className={`relative border-2 rounded-lg p-6 ${
              plan.popular ? 'border-blue-500 bg-blue-50' : 'border-gray-200'
            }`}
          >
            {plan.popular && (
              <span className="absolute -top-3 left-1/2 transform -translate-x-1/2 px-3 py-1 bg-blue-500 text-white text-sm rounded-full">
                Most Popular
              </span>
            )}

            <div className="text-center mb-6">
              <h4 className="text-xl font-semibold text-gray-900 mb-2">{plan.name}</h4>
              <div className="mb-4">
                <span className="text-3xl font-bold text-gray-900">{formatCurrency(plan.price)}</span>
                <span className="text-gray-600">/{plan.interval}</span>
              </div>
              
              {plan.discountPercent && (
                <div className="mb-4">
                  <div className="bg-green-100 text-green-800 px-3 py-1 rounded-full text-sm inline-block">
                    Save {plan.discountPercent}% yearly
                  </div>
                  <p className="text-xs text-gray-600 mt-1">
                    Save {formatCurrency(calculateYearlyDiscount(plan))} per year
                  </p>
                </div>
              )}
            </div>

            <ul className="space-y-3 mb-6">
              {plan.features.map((feature, index) => (
                <li key={index} className="flex items-center text-sm">
                  <CheckCircleIcon className="w-4 h-4 text-green-500 mr-2 flex-shrink-0" />
                  <span className="text-gray-700">{feature}</span>
                </li>
              ))}
            </ul>

            <button
              className={`w-full py-3 px-4 rounded-lg font-medium transition-colors ${
                plan.popular
                  ? 'bg-blue-600 text-white hover:bg-blue-700'
                  : 'border border-gray-300 text-gray-700 hover:bg-gray-50'
              }`}
            >
              Subscribe Now
            </button>
          </motion.div>
        ))}
      </div>

      <div className="bg-blue-50 border border-blue-200 rounded-lg p-6">
        <div className="flex items-start space-x-3">
          <InformationCircleIcon className="w-6 h-6 text-blue-600 mt-1" />
          <div>
            <h4 className="font-medium text-blue-900 mb-2">Subscription Benefits</h4>
            <ul className="text-sm text-blue-800 space-y-1">
              <li>• Cancel anytime with 30-day notice</li>
              <li>• Automatic renewals with email reminders</li>
              <li>• Priority customer support</li>
              <li>• Flexible payment options including local methods</li>
              <li>• Detailed billing history and invoices</li>
            </ul>
          </div>
        </div>
      </div>
    </div>
  )

  const renderBillingHistory = () => (
    <div className="space-y-6">
      <h3 className="text-lg font-semibold text-gray-900">Billing History</h3>

      {billingHistory.length === 0 ? (
        <div className="text-center py-12">
          <BanknotesIcon className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <h4 className="text-xl font-semibold text-gray-900 mb-2">No Billing History</h4>
          <p className="text-gray-600">Your payment history will appear here once you have subscriptions.</p>
        </div>
      ) : (
        <div className="bg-white border border-gray-200 rounded-lg overflow-hidden">
          <div className="overflow-x-auto">
            <table className="min-w-full divide-y divide-gray-200">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Date
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Amount
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Status
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Method
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Invoice
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {billingHistory.map((bill) => (
                  <tr key={bill.id} className="hover:bg-gray-50">
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                      {bill.paymentDate.toLocaleDateString()}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                      {formatCurrency(bill.amount, bill.currency)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                        bill.status === 'paid' ? 'bg-green-100 text-green-800' :
                        bill.status === 'failed' ? 'bg-red-100 text-red-800' :
                        bill.status === 'pending' ? 'bg-yellow-100 text-yellow-800' :
                        'bg-blue-100 text-blue-800'
                      }`}>
                        {bill.status}
                      </span>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                      {bill.paymentMethod}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm">
                      {bill.invoiceUrl ? (
                        <a
                          href={bill.invoiceUrl}
                          className="text-blue-600 hover:text-blue-900 flex items-center space-x-1"
                          target="_blank"
                          rel="noopener noreferrer"
                        >
                          <span>Download</span>
                          <ArrowPathIcon className="w-3 h-3" />
                        </a>
                      ) : (
                        <span className="text-gray-400">N/A</span>
                      )}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}
    </div>
  )

  if (!session) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-50">
        <div className="text-center">
          <CreditCardIcon className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <h2 className="text-2xl font-bold text-gray-900 mb-2">Subscription Management</h2>
          <p className="text-gray-600 mb-6">Please sign in to manage your subscriptions.</p>
          <a
            href="/auth/signin"
            className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            Sign In
          </a>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50 py-8">
      <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-gray-900">Subscription Management</h1>
          <p className="text-gray-600 mt-2">Manage your subscriptions, billing, and payment methods.</p>
        </div>

        {/* Tab Navigation */}
        <div className="border-b border-gray-200 mb-8">
          <nav className="-mb-px flex space-x-8">
            {[
              { key: 'subscriptions', label: 'My Subscriptions', icon: CreditCardIcon },
              { key: 'plans', label: 'Available Plans', icon: BanknotesIcon },
              { key: 'history', label: 'Billing History', icon: CalendarIcon }
            ].map(tab => (
              <button
                key={tab.key}
                onClick={() => setSelectedTab(tab.key as any)}
                className={`flex items-center space-x-2 py-2 px-1 border-b-2 font-medium text-sm ${
                  selectedTab === tab.key
                    ? 'border-blue-500 text-blue-600'
                    : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
              >
                <tab.icon className="w-5 h-5" />
                <span>{tab.label}</span>
              </button>
            ))}
          </nav>
        </div>

        {/* Tab Content */}
        <AnimatePresence mode="wait">
          <motion.div
            key={selectedTab}
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            exit={{ opacity: 0, y: -20 }}
            transition={{ duration: 0.2 }}
          >
            {selectedTab === 'subscriptions' && renderSubscriptions()}
            {selectedTab === 'plans' && renderPlans()}
            {selectedTab === 'history' && renderBillingHistory()}
          </motion.div>
        </AnimatePresence>
      </div>
    </div>
  )
}