'use client'

import { useState, useEffect } from 'react'
import { useSession } from 'next-auth/react'
import { motion, AnimatePresence } from 'framer-motion'
import {
  TicketIcon,
  ClockIcon,
  ExclamationTriangleIcon,
  CheckCircleIcon,
  ChatBubbleLeftRightIcon,
  PaperClipIcon,
  MagnifyingGlassIcon,
  FunnelIcon,
  PlusIcon,
  EyeIcon,
  UserIcon,
  TagIcon
} from '@heroicons/react/24/outline'

interface SupportTicket {
  id: string
  title: string
  description: string
  status: 'open' | 'in-progress' | 'waiting-response' | 'resolved' | 'closed'
  priority: 'low' | 'medium' | 'high' | 'urgent'
  category: 'technical' | 'billing' | 'general' | 'feature-request' | 'bug-report'
  createdAt: Date
  updatedAt: Date
  userId: string
  assignedAgent?: string
  attachments?: string[]
  responses: TicketResponse[]
  estimatedResolution?: string
}

interface TicketResponse {
  id: string
  content: string
  author: string
  authorType: 'user' | 'agent' | 'system'
  timestamp: Date
  attachments?: string[]
}

const MOCK_TICKETS: SupportTicket[] = [
  {
    id: 'TKT-001',
    title: 'Website loading slowly after recent update',
    description: 'After the recent update to our website, the loading speed has decreased significantly. Can you please help optimize the performance?',
    status: 'in-progress',
    priority: 'high',
    category: 'technical',
    createdAt: new Date('2025-10-30'),
    updatedAt: new Date('2025-10-31'),
    userId: 'user-1',
    assignedAgent: 'Sarah Ahmed',
    estimatedResolution: '24 hours',
    responses: [
      {
        id: 'resp-1',
        content: 'Thank you for reporting this issue. We\'ve identified some optimization opportunities and are working on them.',
        author: 'Sarah Ahmed',
        authorType: 'agent',
        timestamp: new Date('2025-10-31T09:00:00')
      }
    ]
  },
  {
    id: 'TKT-002', 
    title: 'Request for additional SEO keywords',
    description: 'We would like to add more SEO keywords to our content strategy. Can you provide recommendations for our industry?',
    status: 'open',
    priority: 'medium',
    category: 'feature-request',
    createdAt: new Date('2025-10-31'),
    updatedAt: new Date('2025-10-31'),
    userId: 'user-1',
    responses: []
  }
]

export default function SupportTicketSystem() {
  const { data: session } = useSession()
  const [tickets, setTickets] = useState<SupportTicket[]>(MOCK_TICKETS)
  const [selectedTicket, setSelectedTicket] = useState<SupportTicket | null>(null)
  const [showCreateForm, setShowCreateForm] = useState(false)
  const [searchQuery, setSearchQuery] = useState('')
  const [filterStatus, setFilterStatus] = useState<string>('all')
  const [filterPriority, setFilterPriority] = useState<string>('all')
  const [newTicketForm, setNewTicketForm] = useState({
    title: '',
    description: '',
    priority: 'medium' as SupportTicket['priority'],
    category: 'general' as SupportTicket['category']
  })
  const [newResponse, setNewResponse] = useState('')

  const filteredTickets = tickets.filter(ticket => {
    const matchesSearch = ticket.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         ticket.description.toLowerCase().includes(searchQuery.toLowerCase())
    const matchesStatus = filterStatus === 'all' || ticket.status === filterStatus
    const matchesPriority = filterPriority === 'all' || ticket.priority === filterPriority
    
    return matchesSearch && matchesStatus && matchesPriority
  })

  const getStatusColor = (status: SupportTicket['status']) => {
    switch (status) {
      case 'open': return 'bg-blue-100 text-blue-800 border-blue-200'
      case 'in-progress': return 'bg-yellow-100 text-yellow-800 border-yellow-200'
      case 'waiting-response': return 'bg-purple-100 text-purple-800 border-purple-200'
      case 'resolved': return 'bg-green-100 text-green-800 border-green-200'
      case 'closed': return 'bg-gray-100 text-gray-800 border-gray-200'
      default: return 'bg-gray-100 text-gray-800 border-gray-200'
    }
  }

  const getPriorityColor = (priority: SupportTicket['priority']) => {
    switch (priority) {
      case 'low': return 'bg-gray-100 text-gray-600'
      case 'medium': return 'bg-blue-100 text-blue-600'
      case 'high': return 'bg-orange-100 text-orange-600'
      case 'urgent': return 'bg-red-100 text-red-600'
      default: return 'bg-gray-100 text-gray-600'
    }
  }

  const getPriorityIcon = (priority: SupportTicket['priority']) => {
    switch (priority) {
      case 'urgent': return <ExclamationTriangleIcon className="w-4 h-4" />
      case 'high': return <ExclamationTriangleIcon className="w-4 h-4" />
      default: return null
    }
  }

  const handleCreateTicket = async () => {
    if (!newTicketForm.title.trim() || !newTicketForm.description.trim()) return

    const newTicket: SupportTicket = {
      id: `TKT-${String(tickets.length + 1).padStart(3, '0')}`,
      title: newTicketForm.title,
      description: newTicketForm.description,
      status: 'open',
      priority: newTicketForm.priority,
      category: newTicketForm.category,
      createdAt: new Date(),
      updatedAt: new Date(),
      userId: session?.user?.id || 'anonymous',
      responses: []
    }

    setTickets(prev => [newTicket, ...prev])
    setNewTicketForm({
      title: '',
      description: '',
      priority: 'medium',
      category: 'general'
    })
    setShowCreateForm(false)
  }

  const handleAddResponse = () => {
    if (!selectedTicket || !newResponse.trim()) return

    const response: TicketResponse = {
      id: `resp-${Date.now()}`,
      content: newResponse,
      author: session?.user?.name || 'Anonymous',
      authorType: 'user',
      timestamp: new Date()
    }

    const updatedTicket = {
      ...selectedTicket,
      responses: [...selectedTicket.responses, response],
      updatedAt: new Date(),
      status: 'waiting-response' as const
    }

    setTickets(prev => prev.map(t => t.id === selectedTicket.id ? updatedTicket : t))
    setSelectedTicket(updatedTicket)
    setNewResponse('')
  }

  if (!session) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gray-50">
        <div className="text-center">
          <TicketIcon className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <h2 className="text-2xl font-bold text-gray-900 mb-2">Support Tickets</h2>
          <p className="text-gray-600 mb-6">Please sign in to access your support tickets.</p>
          <a
            href="/auth/signin"
            className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            Sign In
          </a>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Header */}
        <div className="flex justify-between items-center mb-8">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Support Tickets</h1>
            <p className="text-gray-600 mt-2">Manage your support requests and track their progress</p>
          </div>
          <button
            onClick={() => setShowCreateForm(true)}
            className="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition-colors flex items-center space-x-2"
          >
            <PlusIcon className="w-5 h-5" />
            <span>New Ticket</span>
          </button>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Tickets List */}
          <div className="lg:col-span-2">
            {/* Filters */}
            <div className="bg-white rounded-lg shadow-sm p-4 mb-6">
              <div className="flex flex-col sm:flex-row space-y-4 sm:space-y-0 sm:space-x-4">
                <div className="flex-1">
                  <div className="relative">
                    <MagnifyingGlassIcon className="w-5 h-5 absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" />
                    <input
                      type="text"
                      placeholder="Search tickets..."
                      value={searchQuery}
                      onChange={(e) => setSearchQuery(e.target.value)}
                      className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500"
                    />
                  </div>
                </div>
                <select
                  value={filterStatus}
                  onChange={(e) => setFilterStatus(e.target.value)}
                  className="px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500"
                >
                  <option value="all">All Status</option>
                  <option value="open">Open</option>
                  <option value="in-progress">In Progress</option>
                  <option value="waiting-response">Waiting Response</option>
                  <option value="resolved">Resolved</option>
                  <option value="closed">Closed</option>
                </select>
                <select
                  value={filterPriority}
                  onChange={(e) => setFilterPriority(e.target.value)}
                  className="px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500"
                >
                  <option value="all">All Priority</option>
                  <option value="low">Low</option>
                  <option value="medium">Medium</option>
                  <option value="high">High</option>
                  <option value="urgent">Urgent</option>
                </select>
              </div>
            </div>

            {/* Tickets */}
            <div className="space-y-4">
              {filteredTickets.map((ticket) => (
                <motion.div
                  key={ticket.id}
                  onClick={() => setSelectedTicket(ticket)}
                  className={`bg-white rounded-lg shadow-sm p-6 cursor-pointer transition-all hover:shadow-md ${
                    selectedTicket?.id === ticket.id ? 'ring-2 ring-blue-500' : ''
                  }`}
                  whileHover={{ y: -2 }}
                >
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <div className="flex items-center space-x-3 mb-2">
                        <h3 className="font-semibold text-gray-900">{ticket.title}</h3>
                        <span className={`px-2 py-1 rounded-full text-xs font-medium border ${getStatusColor(ticket.status)}`}>
                          {ticket.status.replace('-', ' ')}
                        </span>
                        <div className={`flex items-center space-x-1 px-2 py-1 rounded-full text-xs font-medium ${getPriorityColor(ticket.priority)}`}>
                          {getPriorityIcon(ticket.priority)}
                          <span>{ticket.priority}</span>
                        </div>
                      </div>
                      <p className="text-gray-600 text-sm mb-3 line-clamp-2">{ticket.description}</p>
                      <div className="flex items-center space-x-4 text-xs text-gray-500">
                        <span className="flex items-center space-x-1">
                          <TicketIcon className="w-4 h-4" />
                          <span>{ticket.id}</span>
                        </span>
                        <span className="flex items-center space-x-1">
                          <ClockIcon className="w-4 h-4" />
                          <span>{ticket.createdAt.toLocaleDateString()}</span>
                        </span>
                        {ticket.assignedAgent && (
                          <span className="flex items-center space-x-1">
                            <UserIcon className="w-4 h-4" />
                            <span>{ticket.assignedAgent}</span>
                          </span>
                        )}
                        <span className="flex items-center space-x-1">
                          <ChatBubbleLeftRightIcon className="w-4 h-4" />
                          <span>{ticket.responses.length} replies</span>
                        </span>
                      </div>
                    </div>
                    <div className="flex items-center space-x-2">
                      <TagIcon className="w-4 h-4 text-gray-400" />
                      <span className="text-xs text-gray-500 capitalize">{ticket.category.replace('-', ' ')}</span>
                    </div>
                  </div>
                </motion.div>
              ))}

              {filteredTickets.length === 0 && (
                <div className="text-center py-12">
                  <TicketIcon className="w-16 h-16 text-gray-400 mx-auto mb-4" />
                  <h3 className="text-lg font-medium text-gray-900 mb-2">No tickets found</h3>
                  <p className="text-gray-600">Try adjusting your search or filters</p>
                </div>
              )}
            </div>
          </div>

          {/* Ticket Details */}
          <div className="lg:col-span-1">
            {selectedTicket ? (
              <div className="bg-white rounded-lg shadow-sm p-6 sticky top-6">
                <div className="flex items-center justify-between mb-4">
                  <h2 className="text-lg font-semibold text-gray-900">Ticket Details</h2>
                  <button
                    onClick={() => setSelectedTicket(null)}
                    className="text-gray-400 hover:text-gray-600"
                  >
                    <EyeIcon className="w-5 h-5" />
                  </button>
                </div>

                <div className="space-y-4">
                  <div>
                    <h3 className="font-medium text-gray-900 mb-2">{selectedTicket.title}</h3>
                    <div className="flex items-center space-x-2 mb-3">
                      <span className={`px-2 py-1 rounded-full text-xs font-medium border ${getStatusColor(selectedTicket.status)}`}>
                        {selectedTicket.status.replace('-', ' ')}
                      </span>
                      <div className={`flex items-center space-x-1 px-2 py-1 rounded-full text-xs font-medium ${getPriorityColor(selectedTicket.priority)}`}>
                        {getPriorityIcon(selectedTicket.priority)}
                        <span>{selectedTicket.priority}</span>
                      </div>
                    </div>
                    <p className="text-gray-600 text-sm">{selectedTicket.description}</p>
                  </div>

                  <div className="border-t pt-4">
                    <h4 className="font-medium text-gray-900 mb-2">Ticket Information</h4>
                    <div className="space-y-2 text-sm">
                      <div className="flex justify-between">
                        <span className="text-gray-500">ID:</span>
                        <span className="font-medium">{selectedTicket.id}</span>
                      </div>
                      <div className="flex justify-between">
                        <span className="text-gray-500">Created:</span>
                        <span>{selectedTicket.createdAt.toLocaleDateString()}</span>
                      </div>
                      <div className="flex justify-between">
                        <span className="text-gray-500">Updated:</span>
                        <span>{selectedTicket.updatedAt.toLocaleDateString()}</span>
                      </div>
                      {selectedTicket.assignedAgent && (
                        <div className="flex justify-between">
                          <span className="text-gray-500">Assigned to:</span>
                          <span>{selectedTicket.assignedAgent}</span>
                        </div>
                      )}
                      {selectedTicket.estimatedResolution && (
                        <div className="flex justify-between">
                          <span className="text-gray-500">ETA:</span>
                          <span className="text-green-600">{selectedTicket.estimatedResolution}</span>
                        </div>
                      )}
                    </div>
                  </div>

                  {/* Responses */}
                  <div className="border-t pt-4">
                    <h4 className="font-medium text-gray-900 mb-3">Conversation ({selectedTicket.responses.length})</h4>
                    <div className="space-y-3 max-h-64 overflow-y-auto">
                      {selectedTicket.responses.map((response) => (
                        <div
                          key={response.id}
                          className={`p-3 rounded-lg ${
                            response.authorType === 'user' 
                              ? 'bg-blue-50 border-l-4 border-blue-400' 
                              : 'bg-gray-50 border-l-4 border-gray-400'
                          }`}
                        >
                          <div className="flex justify-between items-start mb-2">
                            <span className="font-medium text-sm">
                              {response.author}
                            </span>
                            <span className="text-xs text-gray-500">
                              {response.timestamp.toLocaleString()}
                            </span>
                          </div>
                          <p className="text-sm text-gray-700">{response.content}</p>
                        </div>
                      ))}
                    </div>

                    {/* Add Response */}
                    <div className="mt-4">
                      <textarea
                        value={newResponse}
                        onChange={(e) => setNewResponse(e.target.value)}
                        placeholder="Add your response..."
                        className="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 resize-none"
                        rows={3}
                      />
                      <button
                        onClick={handleAddResponse}
                        disabled={!newResponse.trim()}
                        className="mt-2 w-full bg-blue-600 text-white py-2 rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                      >
                        Send Response
                      </button>
                    </div>
                  </div>
                </div>
              </div>
            ) : (
              <div className="bg-white rounded-lg shadow-sm p-6 text-center">
                <TicketIcon className="w-16 h-16 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">Select a Ticket</h3>
                <p className="text-gray-600">Choose a ticket from the list to view details and responses</p>
              </div>
            )}
          </div>
        </div>
      </div>

      {/* Create Ticket Modal */}
      <AnimatePresence>
        {showCreateForm && (
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            exit={{ opacity: 0 }}
            className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4"
            onClick={() => setShowCreateForm(false)}
          >
            <motion.div
              initial={{ opacity: 0, scale: 0.9 }}
              animate={{ opacity: 1, scale: 1 }}
              exit={{ opacity: 0, scale: 0.9 }}
              className="bg-white rounded-lg p-6 w-full max-w-md"
              onClick={(e) => e.stopPropagation()}
            >
              <h2 className="text-xl font-semibold text-gray-900 mb-4">Create New Ticket</h2>
              
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Title
                  </label>
                  <input
                    type="text"
                    value={newTicketForm.title}
                    onChange={(e) => setNewTicketForm({...newTicketForm, title: e.target.value})}
                    className="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500"
                    placeholder="Brief description of the issue"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Description
                  </label>
                  <textarea
                    value={newTicketForm.description}
                    onChange={(e) => setNewTicketForm({...newTicketForm, description: e.target.value})}
                    className="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500 resize-none"
                    rows={4}
                    placeholder="Detailed description of the issue or request"
                  />
                </div>

                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Priority
                    </label>
                    <select
                      value={newTicketForm.priority}
                      onChange={(e) => setNewTicketForm({...newTicketForm, priority: e.target.value as SupportTicket['priority']})}
                      className="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500"
                    >
                      <option value="low">Low</option>
                      <option value="medium">Medium</option>
                      <option value="high">High</option>
                      <option value="urgent">Urgent</option>
                    </select>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Category
                    </label>
                    <select
                      value={newTicketForm.category}
                      onChange={(e) => setNewTicketForm({...newTicketForm, category: e.target.value as SupportTicket['category']})}
                      className="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500"
                    >
                      <option value="general">General</option>
                      <option value="technical">Technical</option>
                      <option value="billing">Billing</option>
                      <option value="feature-request">Feature Request</option>
                      <option value="bug-report">Bug Report</option>
                    </select>
                  </div>
                </div>

                <div className="flex space-x-3 pt-4">
                  <button
                    onClick={() => setShowCreateForm(false)}
                    className="flex-1 py-2 px-4 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
                  >
                    Cancel
                  </button>
                  <button
                    onClick={handleCreateTicket}
                    disabled={!newTicketForm.title.trim() || !newTicketForm.description.trim()}
                    className="flex-1 py-2 px-4 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                  >
                    Create Ticket
                  </button>
                </div>
              </div>
            </motion.div>
          </motion.div>
        )}
      </AnimatePresence>
    </div>
  )
}