'use client'

import { motion } from 'framer-motion'
import { useState, useEffect } from 'react'
import { StarIcon } from '@heroicons/react/24/solid'

interface Testimonial {
  id: number
  name: string
  company: string
  role: string
  content: string
  rating: number
  image?: string
}

const testimonials: Testimonial[] = [
  {
    id: 1,
    name: 'Ahmed Hassan',
    company: 'Tech Solutions Ltd',
    role: 'CEO',
    content: 'Nuwair Systems has been instrumental in our digital transformation. Their enterprise-grade hosting solutions and 24/7 support have exceeded our expectations.',
    rating: 5
  },
  {
    id: 2,
    name: 'Sarah Khan',
    company: 'Digital Marketing Pro',
    role: 'Marketing Director',
    content: 'Outstanding service quality and reliability. We\'ve been with Nuwair Systems for 3 years and have experienced zero downtime. Highly recommended for businesses.',
    rating: 5
  },
  {
    id: 3,
    name: 'Muhammad Ali',
    company: 'E-Commerce Hub',
    role: 'Founder',
    content: 'The professional support team and robust infrastructure have helped us scale our e-commerce platform seamlessly. Best hosting partner we\'ve had.',
    rating: 5
  },
  {
    id: 4,
    name: 'Fatima Malik',
    company: 'Creative Agency',
    role: 'Operations Manager',
    content: 'From domain registration to hosting management, everything is handled professionally. Their corporate-grade solutions are perfect for growing businesses.',
    rating: 5
  }
]

export default function TestimonialsSlider() {
  const [currentIndex, setCurrentIndex] = useState(0)

  useEffect(() => {
    const timer = setInterval(() => {
      setCurrentIndex((prev) => (prev + 1) % testimonials.length)
    }, 5000)

    return () => clearInterval(timer)
  }, [])

  const handleDotClick = (index: number) => {
    setCurrentIndex(index)
  }

  return (
    <div className="relative max-w-4xl mx-auto">
      <div className="overflow-hidden">
        <motion.div
          key={currentIndex}
          initial={{ opacity: 0, x: 50 }}
          animate={{ opacity: 1, x: 0 }}
          exit={{ opacity: 0, x: -50 }}
          transition={{ duration: 0.5 }}
          className="bg-white rounded-xl border border-slate-200 p-8 md:p-12"
        >
          {/* Rating Stars */}
          <div className="flex gap-1 mb-4">
            {[...Array(testimonials[currentIndex].rating)].map((_, i) => (
              <StarIcon key={i} className="w-5 h-5 text-yellow-500" />
            ))}
          </div>

          {/* Testimonial Content */}
          <blockquote className="text-lg text-slate-700 leading-relaxed mb-6">
            "{testimonials[currentIndex].content}"
          </blockquote>

          {/* Author Info */}
          <div className="flex items-center gap-4">
            <div className="w-12 h-12 bg-blue-600 rounded-full flex items-center justify-center">
              <span className="text-white font-bold text-lg">
                {testimonials[currentIndex].name.charAt(0)}
              </span>
            </div>
            <div>
              <div className="font-bold text-slate-900">{testimonials[currentIndex].name}</div>
              <div className="text-sm text-slate-600">
                {testimonials[currentIndex].role} at {testimonials[currentIndex].company}
              </div>
            </div>
          </div>
        </motion.div>
      </div>

      {/* Navigation Dots */}
      <div className="flex justify-center gap-2 mt-6">
        {testimonials.map((_, index) => (
          <button
            key={index}
            onClick={() => handleDotClick(index)}
            className={`w-2 h-2 rounded-full transition-all duration-300 ${
              index === currentIndex 
                ? 'bg-blue-600 w-8' 
                : 'bg-slate-300 hover:bg-slate-400'
            }`}
            aria-label={`Go to testimonial ${index + 1}`}
          />
        ))}
      </div>
    </div>
  )
}
