'use client'

import { motion } from 'framer-motion'
import { useState } from 'react'
import { EnvelopeIcon } from '@heroicons/react/24/outline'
import { CheckCircleIcon } from '@heroicons/react/24/solid'
import ClientOnly from '@/components/ClientOnly'
export default function Contact() {
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    company: '',
    phone: '',
    subject: '',
    message: '',
    service: ''
  })

  const [isSubmitting, setIsSubmitting] = useState(false)

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsSubmitting(true)
    await new Promise(resolve => setTimeout(resolve, 2000))
    setFormData({
      name: '',
      email: '',
      company: '',
      phone: '',
      subject: '',
      message: '',
      service: ''
    })
    setIsSubmitting(false)
    alert('Thank you for your message! We\'ll get back to you soon.')
  }

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value
    })
  }

  return (
    <section id="contact" className="py-16 relative overflow-hidden bg-gradient-to-b from-blue-50 via-indigo-50 to-purple-50">
      {/* Subtle background pattern */}
      <div className="absolute inset-0 overflow-hidden opacity-30 pointer-events-none">
        <div className="absolute inset-0" style={{
          backgroundImage: `radial-gradient(circle at 2px 2px, rgba(59, 130, 246, 0.08) 1px, transparent 0)`,
          backgroundSize: '40px 40px'
        }} />
      </div>

      <div className="relative max-w-5xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Section header */}
        <motion.div
          initial={{ opacity: 0, y: 30 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.8, ease: [0.16, 1, 0.3, 1] }}
          className="text-center mb-10"
        >
          <h2 className="text-3xl md:text-4xl font-bold text-slate-900 mb-2 tracking-tight">
            Ready to Connect?
          </h2>
          <p className="text-base text-slate-600 max-w-xl mx-auto">
            Send us a message and our team will respond within 2 hours. Let's build something great together!
          </p>
        </motion.div>

        {/* Centered contact form with glassmorphism */}
        <div className="flex flex-col items-center justify-center">
          <div className="w-full max-w-2xl">
            <ClientOnly fallback={
              <div className="bg-white/80 shadow rounded-2xl p-8">
                <div className="animate-pulse space-y-4">
                  <div className="h-8 bg-slate-200 rounded"></div>
                  <div className="h-4 bg-slate-200 rounded w-3/4"></div>
                  <div className="grid grid-cols-2 gap-4 mt-6">
                    <div className="h-12 bg-slate-200 rounded"></div>
                    <div className="h-12 bg-slate-200 rounded"></div>
                  </div>
                  <div className="h-32 bg-slate-200 rounded"></div>
                </div>
              </div>
            }>
              <motion.div
                initial={{ opacity: 0, y: 30 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ duration: 0.8, ease: [0.16, 1, 0.3, 1] }}
                className="bg-white/90 shadow-xl rounded-2xl p-8 border border-slate-200 backdrop-blur-md"
              >
                <div className="mb-6 text-center">
                  <h3 className="text-xl font-semibold text-slate-900 mb-1">Send a Message</h3>
                  <p className="text-slate-500 text-xs">All fields are required</p>
                </div>
                <form onSubmit={handleSubmit} className="space-y-4">
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div className="group">
                      <label htmlFor="name" className="block text-xs font-medium text-slate-700 mb-1.5 group-focus-within:text-blue-600 transition-colors">
                        Full Name *
                      </label>
                      <input
                        type="text"
                        id="name"
                        name="name"
                        required
                        value={formData.name}
                        onChange={handleChange}
                        className="w-full px-4 py-2.5 bg-white border border-slate-300 rounded-lg text-sm text-slate-900 placeholder-slate-400 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all"
                        placeholder="John Doe"
                      />
                    </div>
                    <div className="group">
                      <label htmlFor="email" className="block text-xs font-medium text-slate-700 mb-1.5 group-focus-within:text-blue-600 transition-colors">
                        Email Address *
                      </label>
                      <input
                        type="email"
                        id="email"
                        name="email"
                        required
                        value={formData.email}
                        onChange={handleChange}
                        className="w-full px-4 py-2.5 bg-white border border-slate-300 rounded-lg text-sm text-slate-900 placeholder-slate-400 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all"
                        placeholder="john@company.com"
                      />
                    </div>
                  </div>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div className="group">
                      <label htmlFor="company" className="block text-xs font-medium text-slate-700 mb-1.5 group-focus-within:text-blue-600 transition-colors">
                        Company
                      </label>
                      <input
                        type="text"
                        id="company"
                        name="company"
                        value={formData.company}
                        onChange={handleChange}
                        className="w-full px-4 py-2.5 bg-white border border-slate-300 rounded-lg text-sm text-slate-900 placeholder-slate-400 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all"
                        placeholder="Your Company"
                      />
                    </div>
                    <div className="group">
                      <label htmlFor="phone" className="block text-xs font-medium text-slate-700 mb-1.5 group-focus-within:text-blue-600 transition-colors">
                        Phone Number
                      </label>
                      <input
                        type="tel"
                        id="phone"
                        name="phone"
                        value={formData.phone}
                        onChange={handleChange}
                        className="w-full px-4 py-2.5 bg-white border border-slate-300 rounded-lg text-sm text-slate-900 placeholder-slate-400 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all"
                        placeholder="+92 300 1234567"
                      />
                    </div>
                  </div>
                  <div className="group">
                    <label htmlFor="service" className="block text-xs font-medium text-slate-700 mb-1.5 group-focus-within:text-blue-600 transition-colors">
                      Service Interested In
                    </label>
                    <select
                      id="service"
                      name="service"
                      value={formData.service}
                      onChange={handleChange}
                      className="w-full px-4 py-2.5 bg-white border border-slate-300 rounded-lg text-sm text-slate-900 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all appearance-none cursor-pointer"
                    >
                      <option value="" className="bg-white">Select a service</option>
                      <option value="web-development" className="bg-white">Web Development</option>
                      <option value="mobile-app" className="bg-white">Mobile App Development</option>
                      <option value="digital-marketing" className="bg-white">Digital Marketing</option>
                      <option value="ui-ux-design" className="bg-white">UI/UX Design</option>
                      <option value="cloud-solutions" className="bg-white">Cloud Solutions</option>
                      <option value="seo" className="bg-white">SEO Services</option>
                      <option value="custom" className="bg-white">Custom Solution</option>
                    </select>
                  </div>
                  <div className="group">
                    <label htmlFor="subject" className="block text-xs font-medium text-slate-700 mb-1.5 group-focus-within:text-blue-600 transition-colors">
                      Subject *
                    </label>
                    <input
                      type="text"
                      id="subject"
                      name="subject"
                      required
                      value={formData.subject}
                      onChange={handleChange}
                      className="w-full px-4 py-2.5 bg-white border border-slate-300 rounded-lg text-sm text-slate-900 placeholder-slate-400 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all"
                      placeholder="How can we help you?"
                    />
                  </div>
                  <div className="group">
                    <label htmlFor="message" className="block text-xs font-medium text-slate-700 mb-1.5 group-focus-within:text-blue-600 transition-colors">
                      Message *
                    </label>
                    <textarea
                      id="message"
                      name="message"
                      required
                      rows={5}
                      value={formData.message}
                      onChange={handleChange}
                      className="w-full px-4 py-2.5 bg-white border border-slate-300 rounded-lg text-sm text-slate-900 placeholder-slate-400 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all resize-none"
                      placeholder="Tell us about your project..."
                    />
                  </div>
                  <motion.button
                    type="submit"
                    disabled={isSubmitting}
                    whileHover={{ scale: 1.01 }}
                    whileTap={{ scale: 0.99 }}
                    className="w-full bg-blue-600 hover:bg-blue-700 disabled:bg-blue-400 text-white px-6 py-3 rounded-lg font-semibold text-base shadow-lg hover:shadow-xl transition-all duration-200 flex items-center justify-center"
                  >
                    {isSubmitting ? (
                      <>
                        <div className="w-4 h-4 border-2 border-white/30 border-t-white rounded-full animate-spin mr-2" />
                        Sending...
                      </>
                    ) : (
                      <>
                        <EnvelopeIcon className="w-4 h-4 mr-2" />
                        Send Message
                      </>
                    )}
                  </motion.button>
                </form>
              </motion.div>
            </ClientOnly>
          </div>
        </div>

        {/* Trust indicators with modern badges */}
        <motion.div
          initial={{ opacity: 0, y: 30 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.8, delay: 0.2 }}
          className="mt-16 grid grid-cols-2 md:grid-cols-4 gap-6"
        >
          {[
            { icon: CheckCircleIcon, text: '2-Hour Response Time' },
            { icon: CheckCircleIcon, text: 'Free Consultation' },
            { icon: CheckCircleIcon, text: 'No Obligation Quote' },
            { icon: CheckCircleIcon, text: 'Expert Guidance' }
          ].map((item, index) => (
            <motion.div 
              key={index} 
              initial={{ opacity: 0, scale: 0.9 }}
              whileInView={{ opacity: 1, scale: 1 }}
              viewport={{ once: true }}
              transition={{ duration: 0.5, delay: index * 0.1 }}
              className="flex items-center gap-3 text-slate-700 bg-white/80 backdrop-blur-sm rounded-xl p-4 border border-slate-200"
            >
              <item.icon className="w-6 h-6 text-green-500 flex-shrink-0" />
              <span className="text-sm font-semibold">{item.text}</span>
            </motion.div>
          ))}
        </motion.div>

        {/* Dark CTA area at the bottom */}
        <motion.div
          initial={{ opacity: 0, y: 30 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.8, delay: 0.3 }}
          className="mt-20"
        >
          <div className="bg-gradient-to-r from-blue-900 via-blue-800 to-purple-900 rounded-2xl p-10 md:p-14 text-center shadow-xl">
            <h3 className="text-3xl md:text-4xl font-bold text-white mb-4">
              Ready to <span className="text-blue-300">10X Your Business?</span>
            </h3>
            <p className="text-lg text-blue-100 mb-8 max-w-2xl mx-auto">
              Get a free consultation and discover how we can help you grow. Our experts are ready to help you succeed.
            </p>
            <button className="bg-white text-blue-700 hover:bg-blue-100 px-10 py-4 rounded-lg font-semibold text-lg transition-colors duration-300 shadow-md">
              Get Free Consultation
            </button>
          </div>
        </motion.div>
      </div>
    </section>
  )
}
// ...existing code ends above, remove duplicate block below...
