'use client'

import { motion } from 'framer-motion'
import { useState } from 'react'
import Image from 'next/image'
import { EyeIcon, CodeBracketIcon, GlobeAltIcon } from '@heroicons/react/24/outline'

const categories = ['All', 'Web Development', 'Digital Marketing', 'Hosting Solutions', 'Mobile Apps']

const projects = [
  {
    id: 1,
    title: 'E-commerce Platform',
    category: 'Web Development',
    description: 'Modern e-commerce solution with payment integration and inventory management.',
    image: '/portfolio/ecommerce.jpg',
    tags: ['React', 'Node.js', 'Stripe', 'MongoDB'],
    liveUrl: '#',
    codeUrl: '#'
  },
  {
    id: 2,
    title: 'Digital Marketing Campaign',
    category: 'Digital Marketing',
    description: 'Comprehensive digital marketing strategy that increased ROI by 300%.',
    image: '/portfolio/marketing.jpg',
    tags: ['Google Ads', 'SEO', 'Social Media', 'Analytics'],
    liveUrl: '#',
    codeUrl: '#'
  },
  {
    id: 3,
    title: 'Cloud Infrastructure',
    category: 'Hosting Solutions',
    description: 'Scalable cloud hosting solution for high-traffic enterprise application.',
    image: '/portfolio/hosting.jpg',
    tags: ['AWS', 'Docker', 'Kubernetes', 'CI/CD'],
    liveUrl: '#',
    codeUrl: '#'
  },
  {
    id: 4,
    title: 'Mobile Banking App',
    category: 'Mobile Apps',
    description: 'Secure mobile banking application with biometric authentication.',
    image: '/portfolio/mobile.jpg',
    tags: ['React Native', 'Firebase', 'Security', 'UI/UX'],
    liveUrl: '#',
    codeUrl: '#'
  },
  {
    id: 5,
    title: 'Corporate Website',
    category: 'Web Development',
    description: 'Professional corporate website with CMS and SEO optimization.',
    image: '/portfolio/corporate.jpg',
    tags: ['Next.js', 'Tailwind', 'CMS', 'SEO'],
    liveUrl: '#',
    codeUrl: '#'
  },
  {
    id: 6,
    title: 'SaaS Platform',
    category: 'Web Development',
    description: 'Multi-tenant SaaS platform with subscription management.',
    image: '/portfolio/saas.jpg',
    tags: ['React', 'GraphQL', 'PostgreSQL', 'Stripe'],
    liveUrl: '#',
    codeUrl: '#'
  }
]

const containerVariants = {
  hidden: { opacity: 0 },
  visible: {
    opacity: 1,
    transition: {
      staggerChildren: 0.1
    }
  }
}

const itemVariants = {
  hidden: { opacity: 0, y: 20 },
  visible: {
    opacity: 1,
    y: 0,
    transition: { duration: 0.5 }
  }
}

export default function Portfolio() {
  const [activeCategory, setActiveCategory] = useState('All')

  const filteredProjects = activeCategory === 'All' 
    ? projects 
    : projects.filter(project => project.category === activeCategory)

  return (
    <section id="portfolio" className="py-20">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Section header */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.6 }}
          className="text-center mb-16"
        >
          <h2 className="text-4xl font-bold text-gray-900 mb-4 font-heading">
            Our <span className="text-gradient">Latest Projects</span>
          </h2>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            Explore our portfolio of successful projects across various industries and technologies
          </p>
        </motion.div>

        {/* Category filter */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.6, delay: 0.1 }}
          className="flex flex-wrap justify-center gap-4 mb-12"
        >
          {categories.map((category) => (
            <button
              key={category}
              onClick={() => setActiveCategory(category)}
              className={`px-6 py-2 rounded-full font-medium transition-all duration-200 ${
                activeCategory === category
                  ? 'bg-blue-600 text-white shadow-lg'
                  : 'bg-white text-gray-600 hover:bg-gray-100'
              }`}
            >
              {category}
            </button>
          ))}
        </motion.div>

        {/* Projects grid */}
        <motion.div
          variants={containerVariants}
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true }}
          className="grid grid-cols-1 lg:grid-cols-3 gap-8"
        >
          {filteredProjects.slice(-3).map((project) => (
            <motion.div
              key={project.id}
              variants={itemVariants}
              layout
              className="group bg-white rounded-xl overflow-hidden shadow-sm hover:shadow-lg transition-all duration-300 hover:-translate-y-1"
            >
              {/* Project image */}
              <div className="relative h-48 bg-gradient-to-br from-primary-100 to-purple-100 overflow-hidden">
                <div className="absolute inset-0 bg-gradient-to-br from-primary-600/20 to-purple-600/20" />
                <div className="absolute inset-0 flex items-center justify-center">
                  <GlobeAltIcon className="w-16 h-16 text-primary-600/50" />
                </div>
                
                {/* Overlay */}
                <div className="absolute inset-0 bg-black bg-opacity-0 group-hover:bg-opacity-70 transition-all duration-300 flex items-center justify-center opacity-0 group-hover:opacity-100">
                  <div className="flex space-x-4">
                    <a
                      href={project.liveUrl}
                      className="bg-white text-gray-900 p-2 rounded-full hover:bg-gray-100 transition-colors"
                    >
                      <EyeIcon className="w-5 h-5" />
                    </a>
                    <a
                      href={project.codeUrl}
                      className="bg-white text-gray-900 p-2 rounded-full hover:bg-gray-100 transition-colors"
                    >
                      <CodeBracketIcon className="w-5 h-5" />
                    </a>
                  </div>
                </div>
              </div>

              {/* Project details */}
              <div className="p-6">
                <div className="mb-2">
                  <span className="text-sm text-primary-600 font-medium">
                    {project.category}
                  </span>
                </div>
                <h3 className="text-xl font-semibold text-gray-900 mb-2">
                  {project.title}
                </h3>
                <p className="text-gray-600 mb-4 leading-relaxed">
                  {project.description}
                </p>
                
                {/* Tags */}
                <div className="flex flex-wrap gap-2">
                  {project.tags.map((tag) => (
                    <span
                      key={tag}
                      className="px-2 py-1 bg-gray-100 text-gray-600 text-xs font-medium rounded"
                    >
                      {tag}
                    </span>
                  ))}
                </div>
              </div>
            </motion.div>
          ))}
        </motion.div>

        {/* CTA */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.6, delay: 0.3 }}
          className="text-center mt-16"
        >
          <p className="text-gray-600 mb-6">
            Ready to start your next project with us?
          </p>
          <button className="bg-primary-600 hover:bg-primary-700 text-white px-8 py-3 rounded-lg font-semibold transition-colors duration-200">
            Start Your Project
          </button>
        </motion.div>
      </div>
    </section>
  )
}